<?php

namespace Marcolin\Models;
use AlgoliaSearch\Laravel\AlgoliaEloquentTrait;
use Auth;
use DB;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\App;
use Marcolin\Facades\MarcolinAssets;
use Venturecraft\Revisionable\RevisionableTrait;

/**
 * Class WebUnit
 * @package Marcolin\Models
 * @property int $id
 * @property string $name
 * @property string $part_detail
 * @property string $safe_name
 * @property string $override_name
 * @property-read Unit $unit
 * @property-read WebUnitFamily $family
 * @property-read WebUnitAttributeValue[] $attributeValues
 * @method static Builder latest()
 * @property string $part_number
 * @property integer $part_id
 * @property integer $webunitfamily_id
 * @property string $date_created
 * @property string $date_modified
 * @property string $date_uploaded
 * @property string $deleted_at
 * @property-read \Marcolin\Models\WebUnitFamily $webUnitFamily
 * @property-write mixed $web_part_attributes
 * @property-read mixed $part_name
 * @property-read mixed $family_name
 * @property-read mixed $context
 * @property-read mixed $default_context
 * @mixin \Eloquent
 * @method static WebUnit firstOrNew(array $attributes)
 */
class WebUnit extends WebModel
{
    protected $table = "WebUnits";
    protected $fillable = [];
    protected $casts = [
        ['is_visible' => 'boolean' , 'is_family_image' => 'boolean']
    ];

    use AlgoliaEloquentTrait;

    public $algoliaSettings = [
        'searchableAttributes' => [
            'unit.upc',
            'unit.style',
            'unit.name',
            'unit.color',
            'unit.size',
            'unit.colorDescription',
        ],
        'attributesForFaceting' => [
        ]
    ];

    public function _reindex($safe = true, $setSettings = true, $mergeOldSettings = false, \Closure $onInsert = null)
    {
        /** @var \AlgoliaSearch\Laravel\ModelHelper $modelHelper */
        $modelHelper = App::make('\AlgoliaSearch\Laravel\ModelHelper');

        $indices = $modelHelper->getIndices($this);
        $indicesTmp = $safe ? $modelHelper->getIndicesTmp($this) : $indices;

        if ($setSettings === true) {
            $setToTmpIndices = ($safe === true);
            $this->_setSettings($setToTmpIndices, $mergeOldSettings);
        }

        static::webVisible()
            ->has('webStyle.webBrand.webSilos')
            ->has('unit.style.brand')

            ->with('webStyle.webBrand.webSilos')
            ->with('unit.style.productDescription')
            ->with('unit.style.productShape')
            ->with('unit.style.productRimType')
            ->with('unit.style.productFitting')
            ->with('unit.style.brand')
            ->with('unit.productLensType')
            ->with('unit.productColorFamily')

            ->chunk(1000, function ($models) use ($indicesTmp, $modelHelper, $onInsert) {
                /** @var \AlgoliaSearch\Index $index */
                foreach ($indicesTmp as $index) {
                    $records         = [];
                    $recordsAsEntity = [];

                    foreach ($models as $model) {
                        if ($modelHelper->indexOnly($model, $index->indexName)) {
                            $records[] = $model->getAlgoliaRecordDefault($index->indexName);

//                            dd($records);

                            if ($onInsert && is_callable($onInsert)) {
                                $recordsAsEntity[] = $model;
                            }
                        }
                    }

                    $index->addObjects($records);

                    if ($onInsert && is_callable($onInsert)) {
                        call_user_func_array($onInsert, [$recordsAsEntity]);
                    }
                }

            })
        ;

        if ($safe) {
            for ($i = 0; $i < count($indices); $i++) {
                $modelHelper->algolia->moveIndex($indicesTmp[$i]->indexName, $indices[$i]->indexName);
            }

            $this->_setSettings(false); // To a setSettings to set the slave on the master
        }
    }

    public function indexOnly() {
        if (!$this->is_visible) return false;
        if (!$this->webStyle->is_visible) return false;

        return true;
    }

    public function getAlgoliaRecord() {
        DB::enableQueryLog();
        $record = array_only($this->toArray(), [
            'id',
        ]);

        $record['name'] = $this->webStyle->name;
        $record['slug'] = $this->webStyle->slug;

        $record['unit'] = $this->unit;

//        dd(DB::getQueryLog());
        return $record;
    }

    protected $indexSettings = [
        'analysis' => [
            'char_filter' => [
                'replace' => [
                    'type' => 'mapping',
                    'mappings' => [
                        '&=> and '
                    ],
                ],
            ],
            'filter' => [
                'word_delimiter' => [
                    'type' => 'word_delimiter',
                    'split_on_numerics' => false,
                    'split_on_case_change' => false,
                    'generate_word_parts' => true,
                    'generate_number_parts' => true,
                    'catenate_all' => true,
                    'preserve_original' => true,
                    'catenate_numbers' => true,
                ]
            ],
            'analyzer' => [
                'default' => [
                    'type' => 'custom',
                    'char_filter' => [
                        'html_strip',
                        'replace',
                    ],
                    'tokenizer' => 'whitespace',
                    'filter' => [
                        'lowercase',
                        'word_delimiter',
                    ],
                ],
            ],
        ],
    ];

    protected $mappingProperties = [
        'title' => [
            'type' => 'string',
            'analyzer' => 'standard'
        ]
    ];

//    use ElasticquentTrait;

    public static $elasticFuzziness = 1;

    /**
     * @description return elastic query with fuzziness
     */
    public static function buildElasticQuery($keyWords){
        $params=[
            'multi_match' => [
                'query'     => $keyWords,
                'fuzziness' => self::$elasticFuzziness,
                'fields'    => ['_all']
            ]
        ];
        return $params;
    }

//	use KeyWordsFunctional;

    /**
     * @description index all the entries in an Eloquent model (if needed)
     */
    public static function elasticIndexingIfNeeded(){
        $mappingExists = self::mappingExists();
        if (!$mappingExists) {
            self::createIndex();
            self::putMapping();
            self::addAllToIndex();
        }
	}
	
	public function unit() {
		return $this->belongsTo(Unit::class, 'unit_id', 'id');
	}

	/**
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function webStyle() {
		return $this->belongsTo(WebStyle::class, 'webstyle_id', 'id');
	}

	public function promos() {
		return $this->belongsToMany(Promo::class, 'Promo_WebUnit','webunit_id', 'promo_id');
	}

	public function scopeIsFamilyImage($query){
		$query->where('is_family_image',1)->visible()->whereHas('part', function ($query) {
            return $query->whereHas('inventoryItems', function ($query) {
                return $query->isValid();
            });
        });
	}

	public function scopePriceList($query, $priceList, $currency, $quantityLevel = 1) {
		return $query
			->leftJoin('PriceLists', 'PriceLists.part_no', '=', 'WebUnits.part_number')
			->where('price_list', $priceList)
			->where('currency', $currency)
			->where('quantity_level', '>=', $quantityLevel)
			->select('WebUnits.*', 'PriceLists.price')
		;
	}

	public function webAttributes() {
		return $this->belongsToMany(WebAttribute::class, 'WebAttribute_WebUnit', 'webunit_id', 'webattribute_id')
			->withPivot(['id','attribute_value']);
	}

	public function webAttributeValue($webAttributeId) {
		if (!$this->relationLoaded('webAttributes')) $this->load('webAttributes');
		$webAttributeValue = $this->webAttributes->find($webAttributeId);
		if (!is_null($webAttributeValue)) return $webAttributeValue->pivot->attribute_value;
		return null;
	}

	public function setWebAttributeValue($webAttributeId, $value) {
		$this->webAttributes()->sync([$webAttributeId => ['attribute_value' => $value]], false);
	}
	public function setWebAttributesAttribute($attributes) {
		$this->save();
		foreach ($attributes as $webAttributeId => $attributeValue) {
			$this->setWebAttributeValue($webAttributeId, $attributeValue);
		}
	}

	public function getPartNameAttribute() {
		$part = $this->unit;
		if (is_null($part)) return null;
		return $part->unit_desc;
	}
	
	public function getFamilyNameAttribute() {
		$family = $this->webStyle;
		if (is_null($family)) return null;
		return $family->name;
	}

    public function getCustomerPrice($customer = false) {
	    if ($customer === false) $customer = b2b()->activeCustomer();
        return $this->unit->getCustomerPrice($customer);
    }

    public function getCustomerDiscountAttribute() {
	    return $this->unit->customerDiscount;
    }

//	public static function getImageUrl($partNumber, $width = null, $height = null, $avoidCache=false) {
//		$url = config('marcolin.images.url');
//		if(!isset($url)) $url = config('marcolin-b2b.images.url');
//		$url .= '/' . $partNumber;
//		if ($width) $url .= "/$width";
//		if ($height) $url .= "/$height";
//
//		return ($avoidCache)?\ImgHelper::avoidCache($url):$url;
//	}

    public function getImageUrl($width = false, $height = false) {
        return MarcolinAssets::part($this, $width, $height);
    }

	public function scopePriceRange($query, $min, $max) {
		return $query->whereHas('part', function($part) use ($min, $max){
			$part->whereBetween('list_price', [$min, $max]);
		});
	}

	public function scopeManagedBy($query, $manager) {
		return $query->whereHas('part', function ($query) use ($manager) {
			return $query->where('pf_prod_mgr', $manager);
		});
	}

    public function scopeInCollection($query, $webCollectionId) {
        return $query->whereHas('webStyle', function ($query) use ($webCollectionId) {
            return $query->inCollection($webCollectionId);
        });
    }

    public function scopeInCategory($query, $webCategoryId) {
        return $query->whereHas('webStyle', function ($query) use ($webCategoryId) {
            return $query->inCategory($webCategoryId);
        });
    }

	public function scopeInGroup($query, $webBrandId) {
		return $query->whereHas('webStyle', function ($query) use ($webBrandId) {
			return $query->inGroup($webBrandId);
		});
	}

	public function scopeVisible(Builder $query, $webSiloId = false) {
		$query
            ->with('unit')
			->where('WebUnits.is_visible', '1')
		;

		return $query;
	}

	public function scopeWebVisible(Builder $query) {
        return $query
            ->visible()
            ->whereHas('unit', function ($query) {
                return $query->webVisible();
            })
        ;
    }

	public function visibleInSilo($webSilo) {
	    if (!$webSilo instanceof WebSilo) $webSilo = WebSilo::find($webSilo);
        $inSilo = true;
        return $inSilo;
    }

    public function getInStockAttribute() {
        return $this->unit->in_stock;
    }

    public function getInStockLabelAttribute() {
        $modelValue = array_get($this->attributes, 'in_stock_label');
        return strlen($modelValue) ? $modelValue : config('marcolin.inStockLabels.0', 'In Stock');
    }

    public function getOutOfStockLabelAttribute() {
        $modelValue = array_get($this->attributes, 'out_of_stock_label');
        return strlen($modelValue) ? $modelValue : config('marcolin.outOfStockLabels.0', 'Out of stock');
    }

    public function getStockStatusAttribute() {
        return $this->unit->outOfStock;
    }

    public function getStockLabelAttribute() {
        $status = $this->stock_status;
        switch ($status) {
            case Unit::IN_STOCK:
                return $this->in_stock_label;
            case Unit::OUT_OF_STOCK:
                return $this->out_of_stock_label;
            case Unit::DELAYED:
                return "Ships in 3-5 days";
        }
        return '';
    }

	use HasAssets, RevisionableTrait;

}
