<?php

namespace Hilco\Models;
use AlgoliaSearch\Laravel\AlgoliaEloquentTrait;
use DB;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Input;
use Venturecraft\Revisionable\RevisionableTrait;
use Storage;

/**
 * Class WebStyle
 * @package Hilco\Models
 */
class WebStyle extends WebModel implements HasLandingPageInterface
{
	protected $table = "WebStyles";
    protected $perPage = 24;

	public function getPerPage()
	{
		$tmpPerPage =Input::get('perPage');
		return (isset($tmpPerPage) && $tmpPerPage > 0)?$tmpPerPage:$this->perPage;
	}

	public function style() {
        return $this->belongsTo(Style::class);
    }

	public function webUnits() {
        return $this->hasMany(WebUnit::class, 'webstyle_id', 'id');
	}

    public function webBrand() {
		return $this->belongsTo(WebBrand::class, 'webbrand_id', 'id');
    }

    public function webDetailTabs() {
        return $this->belongsToMany(WebDetailTab::class, 'WebDetailTab_WebStyle', 'webstyle_id', 'webdetailtab_id')->withPivot('is_visible');
    }

    public function favoritedWebUsers() {
	    return $this->belongsToMany(WebUser::class, 'WebStyle_WebUser', 'webstyle_id', 'webuser_id');
    }

    public function isFavorite() {
	    return $this->favoritedWebUsers->where('id', auth()->id())->count() ? true : false;
    }

	/**
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function visibleWebUnits() {
		return $this
			->webUnits()
			->visible()
            ->with('unit')
            ->with('assets')
//            ->whereHas('unit', function ($query) {
//                return $query->where('Units.salesOrg', '=', 'UN01');
//            })
			->orderBy('name');
	}

    public function scopeVisible(Builder $query) {
        return $query
            ->where('WebStyles.is_visible', '1')
            ->whereHas('webUnits', function ($query) {
                return $query->visible();
            })
            ->whereHas('webBrand', function ($query) {
                return $query->visible();
            })
            ;
    }

    public function scopeDisplayOrder(Builder $query) {
        return $query->orderBy('pivot_display_order', 'asc')->orderBy('name', 'asc');
    }

    public function scopeAlphabetical(Builder $query) {
        return $query->orderBy('name', 'asc');
    }

	public function scopeHasAttributes($query, $filters) {
		if(!isset($filters) || is_null($filters)) return $query;
		$attrs = [];
		foreach ($filters as $filter){
			if(!isset($attrs[$filter->webattribute_id])){
				$attrs[$filter->webattribute_id] = [];
			}
			$attrs[$filter->webattribute_id] = array_merge($attrs[$filter->webattribute_id],$filter->webAttributeFilterValues->pluck('filter_value')->toArray());
		}
		return $query->where(function ($query) use ($attrs) {
            foreach ($attrs as $id=>$values) {
//                $query->whereHas('webAttributes', function ($query) use ($id, $values) {
//                    return $query->where('webattribute_id', $id);
//                });
            }
			return $query->whereHas('visibleWebUnits', function ($query) use ($attrs) {
				foreach ($attrs as $id=>$values) {
//					$query = $query->whereHas('webAttributes', function ($query) use ($id, $values) {
//						return $query->where('webattribute_id', $id)->whereIn('attribute_value', $values);
//					});
				}
				return $query;
			});
		});
	}
	public function scopeManagedBy($query, $product_manager) {
		return $query->whereHas('webUnits', function($query) use ($product_manager) {
			return $query->managedBy($product_manager);
		});
	}

	public function scopeInBrand($query, $webBrand) {
		$webBrandId = ($webBrand instanceof WebBrand) ? $webBrand->id : $webBrand;
		return $query->where('webBrand_id', '=', $webBrandId);
	}

    public function scopeInSilo($query, $webSilo) {
        $webSiloId = ($webSilo instanceof WebSilo) ? $webSilo->id : $webSilo;
        return $query->whereHas('webBrand', function ($query) use ($webSiloId) {
            return $query->inSilo($webSiloId);
        });
    }

	public static function in($model) {
		if ($model instanceof WebBrand) {return self::inGroup($model);}
		else if ($model instanceof WebCategory) return self::inCategory($model);
        else if ($model instanceof WebBrand) return self::inCollection($model);
        else if ($model instanceof WebSilo) return self::inSilo($model);
        else return self::newQuery();
	}

	public function scopeForDisplay(Builder $query, $slug) {
		return $query
			->visible()
            ->with(['visibleWebUnits' => function ($query) {
                $query->with(['assets', 'unit' => function ($query) {
//                    $query->with('customerPrices', 'inventoryItems');
                }]);
            }, 'assets'])
//            ->with('webAttributes')
            ->with(['favoritedWebUsers' => function ($query) {
                $query->where('WebUsers.id', auth()->id());
            }])
			->slug($slug)
		;
	}

	public function scopeForBreadcrumbs($query) {
        $query->with([
            'webBrand' => function ($lQuery) {
                if (session('breadcrumb.collectionId')) {
                    $lQuery->where('webBrand.id', '=', session('breadcrumb.collectionId'));
                }
                $lQuery->with(['webCategories' => function ($cQuery) {
                    if (session('breadcrumb.categoryId')) {
                        $cQuery->where('WebCategories.id', '=', session('breadcrumb.categoryId'));
                    }
                    $cQuery->with(['webBrand' => function ($gQuery) {
                        if (session('breadcrumb.groupId')) {
                            $gQuery->where('webBrand.id', '=', session('breadcrumb.groupId'));
                        }
                    }]);
                }]);
            }
        ]);
        return $query;
    }

	public function scopeForLandingPage($query, HasLandingPageInterface $lpModel) {
	    $query
            ->visible()
            ->notForSearchOnly()
            ->with('style')
            ->with('visibleWebUnits.assets')
            ->with('visibleWebUnits.unit')
            ->with(['favoritedWebUsers' => function ($query) {
                $query->where('WebUsers.id', auth()->id());
            }])
            ->whereHas('style', function ($query) {
                return $query->where('Styles.salesOrg', '=', 'UN01');
            })
            ->has('visibleWebUnits.assets')
            ->has('visibleWebUnits.unit')
            ->groupBy('WebStyles.id')
        ;
        if ($lpModel instanceof WebBrand) {
            $query->where('webbrand_id', $lpModel->id);
        }
        return $query;
    }

	public static function getImageUrl($partNumber, $width = null, $height = null) {
		$url = config('marcolin.images.url');
		if(!isset($url)) $url = config('hilco-b2b.images.url');
		$url .= '/' . $partNumber;
		if ($width) $url .= "/$width";
		if ($height) $url .= "/$height";
		return $url;
	}

	public function scopeJoinWebUnit(Builder $query){
		return $query->leftJoin('WebUnits', 'WebUnits.webfamily_id', '=', 'WebStyles.id');
	}

    public function slugUrl() {
        return route('family.slug', [$this->slug]);
    }

    public function scopeForSearchOnly(Builder $query){
        return $query->where('search_only',1);
    }

    public function scopeNotForSearchOnly(Builder $query){
        return $query->where('search_only',0);
    }

    public function scopeBestSellers(Builder $query) {
        return $query->where('best_seller', 1);
    }

    public function getNameAttribute($name) {
        if (isset($name) && strlen($name) > 0) {
            return $name;
        }

        $name = $this->style->name;
        if (isset($name) && strlen($name) > 0) {
            return $name;
        }

        return $this->slug;
    }

    public function getPrimaryImage() {
        $asset = $this->assetsByType('primary')->first();
        if (!is_null($asset)) return $asset;

        $webUnit = $this->visibleWebUnits->where('is_family_image', 1)->first();
        if (!is_null($webUnit)) {
            $asset = $webUnit->getPrimaryImage();
            if (!is_null($asset)) return $asset;
        }

        foreach ($this->webUnits as $webUnit) {
            if (!is_null($webUnit)) {
                $asset = $webUnit->getPrimaryImage();
                if (!is_null($asset)) {
                    return $asset;
                }
            }
        }

        return null;
    }

	use HasAssets, HasSlug, RevisionableTrait, HasLandingPage, ManyToManyRevisionableTrait;

    public function getParentRelationName()
    {
        return 'webBrand';
    }

    public function getLabelStyleAttribute() {
        return array_get(self::$labelStyles, $this->label);
    }

    public function getCustomerPriceAttribute() {
        $authUser = Auth::user();
        $soldToCustomer = $authUser->activeCustomer;
        return $this->style->getCustomerPrice($soldToCustomer);
    }

    public function getPriceRangeAttribute() {
        if (is_null($this->visibleWebUnits)) return null;

        $authUser = Auth::user();
        $soldToCustomer = $authUser->activeCustomer;

        $min = PHP_INT_MAX;
        $max = 0;
        foreach ($this->visibleWebUnits as $webUnit) {
            $price = $webUnit->getCustomerPrice($soldToCustomer);
            $min = min($min, $price);
            $max = max($max, $price);
        }
        return [$min, $max];
    }
}
