<?php

namespace Marcolin;

use Carbon\Carbon;
use Marcolin\Models\Customer;
use Marcolin\Models\WebPromotion;
use Marcolin\Models\PromotionCodeTrigger;
use Marcolin\Models\PromotionCodeTrigger_Code;
use Marcolin\Models\WebSilo;

/**
 * Class B2BHelper
 * Helper class, use b2b() to call public functions, e.g., b2b()->activeCustomer()
 * @package Marcolin
 */
class B2BHelper {
    protected $activeCustomer;
    protected $activeWebSilo;
    protected $activeShippingAddress;
    protected $activePromotion;
    protected $activePromoCode;
    /* Coupon Code not needed yet */
//    protected $activeCouponCode;

    public function __construct()
    {
        $activeShippingAddress = null;
        $this->loadActiveCustomer();
    }

//  ******************************** LOAD ********************************
    protected function loadActiveCustomer()
    {
        $this->activeCustomer = false;

        $aliasedCustomerId = session()->get('aliasedCustomerId', false);

        if ($aliasedCustomerId) {
            $this->activeCustomer = Customer::where('Customers.id', '=', $aliasedCustomerId)
                ->with('customerDiscounts')
                ->with('brands')
                ->first();
        } else {
            $user = auth()->user();
            if ($user) {
                $this->activeCustomer = $user
                    ->customer()
                    ->with('customerDiscounts')
                    ->with('brands')
                    ->first();
            }
        }

        $this->setActiveShippingAddress();

        $this->loadActiveWebSilo();

        $this->loadActivePromotion();
    }

    protected function loadActiveWebSilo() {
        $this->activeWebSilo = false;
        $user = auth()->user();

        $activeWebSiloId = false;

        if (!$activeWebSiloId) {
            if ($user) {
                if ($this->activeCustomer) {
                    $defaultCustomerWebSilo = $this->activeCustomer->webSilo;
                    if ($defaultCustomerWebSilo && count($defaultCustomerWebSilo) > 0) {
                        $activeWebSiloId = $defaultCustomerWebSilo[0]->id;
                    }

                    if (config('marcolin.buying_group_for_microsites') == 1) {
                        $defaultBuyingGroupWebSilo = $this->activeCustomer->buyingGroup->webSilo;
                        if ($defaultBuyingGroupWebSilo && count($defaultBuyingGroupWebSilo) > 0) {
                            $activeWebSiloId = $defaultBuyingGroupWebSilo[0]->id;
                        }
                    }
                }

                if (!$activeWebSiloId) {
                    $userSilo = $user->defaultWebSilo;
                    if ($userSilo) {
                        $activeWebSiloId = $userSilo->id;
                    }
                }
            }

            if (!$activeWebSiloId) $activeWebSiloId = WebSilo::first()->id;
        }
        $this->activeWebSilo = WebSilo::with('webLandingPage')->find($activeWebSiloId);
        session()->put('b2b.activeWebSiloId', $activeWebSiloId);

        return $this->activeWebSilo;
    }

    protected function loadActivePromotion() {
        $this->activePromotion = false;

        $activePromotionId = session()->get('activePromotionId', false);

        if ($activePromotionId) {
            $this->activePromotion = WebPromotion::find($activePromotionId);
        }

        $this->loadActivePromoCode();
    }

    protected function loadActivePromoCode() {
        $this->activePromoCode = false;

        $currentPromoCode = session()->get('promoCode', false);

        if ($this->activePromotion) {
            $codeTrigger = $this->activePromotion->codeTriggers->get(0);
            if ($currentPromoCode && $currentPromoCode === $codeTrigger->code) {
                $this->activePromoCode = $currentPromoCode;
            } else {
                $this->activePromoCode = $codeTrigger->code;
                session()->put('promoCode', $this->activePromoCode);
            }
        } else {
            session()->put('promoCode', false);
        }

        /* Coupon Code not needed yet */
//        $this->loadActiveCouponCode();
    }

    /* Coupon Code not needed yet */
//    protected function loadActiveCouponCode() {
//        $this->activeCouponCode = false;
//
//        $currentActiveCouponCode = session()->get('couponCode', false);
//
//        if ($currentActiveCouponCode && $this->activePromotion && $this->activePromoCode) {
//            $codeTrigger = $this->activePromotion->codeTriggers->get(0);
//            if ($codeTrigger->is_list && null !== $codeTrigger->byCouponCode($currentActiveCouponCode)->first()) {
//                $this->activeCouponCode = $currentActiveCouponCode;
//            }
//        } else {
//            session()->put('couponCode', false);
//        }
//    }

//  **********************************************************************

//  ******************************** ALIASING ****************************
    public function aliasAs($customerId = false)
    {
        session()->forget('aliasedCustomerId');
        session()->forget('activeWebSiloId');
        session()->forget('activePromotionId');
        session()->forget('promoCode');

        if ($customerId) {
            session()->put('aliasedCustomerId', $customerId);
        }

        $this->loadActiveCustomer();
    }

    public function isAliased()
    {
        return (session()->get('aliasedCustomerId', false));

    }

    public function stopAliasing()
    {
        $this->aliasAs(false);
    }
//  **********************************************************************

//  ******************************** SETTERS *****************************
    public function setActiveShippingAddress($customerShippingAddress = false)
    {
        if ($customerShippingAddress === false) {
            if ($this->activeCustomer) {
                $customerShippingAddress = $this->activeCustomer->default_shipping_address;
            }
        }
        $this->activeShippingAddress = $customerShippingAddress;
        return $this->activeShippingAddress;
    }

    /**
     * If $promotion is null, clear out activePromotionId, activePromoCode, and activeCouponCode
     * Otherwise, update the activePromotionId if necessary, and set activePromoCode to
     * the code set on the promotion's PromotionCodeTrigger object.
     * If $couponCode is not null, set activeCouponCode to be that code.
     * @param $promotion
     * @param null $couponCode
     */
    public function setActivePromotion($promotion, $couponCode = null) {
        if (is_null($promotion)) {
            session()->forget('activePromotionId');
            $this->activePromotion = false;
            session()->forget('promoCode');
            $this->activePromoCode = false;
            /* Coupon Code not needed yet */
//            session()->forget('couponCode');
//            $this->activeCouponCode = false;
        } else {
            $currentActivePromotionId = session()->get('activePromotionId', false);
            if (!$currentActivePromotionId || $currentActivePromotionId !== $promotion->id) {
                $this->activePromotion = $promotion;
                session()->put('activePromotionId', $promotion->id);

                $codeTrigger = $this->activePromotion->codeTriggers->get(0);

                /* Coupon Code not needed yet -- right now we should never enter this if-statement */
                if ($couponCode !== null &&
                    $codeTrigger->is_list &&
                    null !== $codeTrigger->byCouponCode($couponCode)->first()) {

                    /* Coupon Code not needed yet */
//                    $this->activeCouponCode = $couponCode;
//                    session()->put('couponCode', $this->activeCouponCode);

                }

                $this->activePromoCode = $codeTrigger->code;
                session()->put('promoCode', $this->activePromoCode);
            }
        }
    }
//  **********************************************************************

//  ******************************** GETTERS *****************************
    public function activeCustomer()
    {
        return $this->activeCustomer;
    }

    public function activeShippingAddress()
    {
        return $this->activeShippingAddress;
    }

    public function activeCountry()
    {
        if ($this->activeShippingAddress) return $this->activeShippingAddress()->country;

        return null;
    }

    public function activeCurrency()
    {
        $activeCurrency = 'USD';

        if ($this->activeCustomer) {
            $activeCurrency = $this->activeCustomer->currency;
        }

        return $activeCurrency;
    }

    public function activeWebHierarchy() {
        return $this->activeWebSilo()->webHierarchy;
    }

    public function activeWebSilo() {
        return $this->activeWebSilo;
    }

    public function activeSalesOrg()
    {
        return array_get($this->activeCustomer, 'salesOrg', 'UN01');
    }

    public function activePromotion() {
        return $this->activePromotion;
    }

    public function activePromoCode() {
        return $this->activePromoCode;
    }

    /* Coupon Code not needed yet */
//    public function activeCouponCode() {
//        return $this->activeCouponCode;
//    }

//  **********************************************************************

//  ******************************** PROMOS ******************************
    public function eligiblePromotions() {
        return ($this->activeCustomer ? WebPromotion::getEligiblePromotions() : []);
    }

    public function unQualifiedPromotions() {
        return ($this->activeCustomer ? WebPromotion::getUnQualifiedPromotions() : []);
    }

    /**
     * Validates that the code in $code corresponds to the promo code value on a valid codeTrigger,
     * or an unredeemed code value in the list of coupon codes on a valid codeTrigger. If the code
     * is technically valid but the codeTrigger has a list of coupon codes, return false anyway
     * because customers should be redeeming via coupon codes, no one should know about the
     * 'backend' promo code.
     * Returns $promotion object that the valid code corresponds to, boolean false otherwise
     * @param $code (promo or coupon)
     * @return bool
     */
    public function isValidPromoCode($code) {

        $codeTrigger = PromotionCodeTrigger::byCode($code)->first();

        if ($codeTrigger == null) {
            /* Coupon Code not needed yet -- so codeTriggerCode should always be null */
            $codeTriggerCode = PromotionCodeTrigger_Code::byCode($code)->first();
            if ($codeTriggerCode !== null && $codeTriggerCode->is_redeemed == 0) {
                $codeTrigger = $codeTriggerCode->promotionCodeTrigger;
                if ($codeTrigger->is_list == 0) {
                    // even through the coupon code is technically valid,
                    // if the code trigger on the promo isn't configured
                    // for a coupon list then it shouldn't be redeemable
                    $codeTrigger = null;
                }
            }
        } else if ($codeTrigger->is_list == 1) {
            // conversely, if someone somehow knew the backend promo code
            // for a promo that is supposed to have a coupon code list,
            // still return false because they shouldn't be allowed to redeem
            $codeTrigger = null;
        }

        if (isset($codeTrigger)) {
            // finally, verify that the promotion that this code matches to
            // actually is an eligible promotion to begin with
            $promotion = $codeTrigger->promotion()->first();
            if (isset($promotion)) {
                foreach ($this->eligiblePromotions() as $eligiblePromotion) {
                    if ($promotion->id === $eligiblePromotion->id) {
                        return $promotion;
                    }
                }
            }
        }

        return false;
    }
//  **********************************************************************

    public function currencySymbol($currencyCode = false)
    {
        if ($currencyCode === false) $currencyCode = $this->activeCurrency();
        if ($currencyCode == 'GBP') return '&pound;';

        return '$';
    }

    public function formatPrice($price, $decimals = -1, $symbol = '$')
    {
        if ($decimals = -1) {
            if ($price == round($price)) $decimals = 0;
            else $decimals = 2;
        }
        return $symbol . number_format($price, $decimals);
    }

    public function getEndOfCurrentQuarter($startDate = false) {
        if (!$startDate) {
            $current_month = date('m', $startDate);
            $current_year = date('Y', $startDate);
        } else {
            $current_month = $startDate->month;
            $current_year = $startDate->year;
        }

        if($current_month>=1 && $current_month<=3)
        {
            $end_of_quarter_date = strtotime('1-April-'.$current_year);  // timestamp or 1-April 12:00:00 AM means end of 31 March
        }
        else  if($current_month>=4 && $current_month<=6)
        {
            $end_of_quarter_date = strtotime('1-July-'.$current_year);  // timestamp or 1-July 12:00:00 AM means end of 30 June
        }
        else  if($current_month>=7 && $current_month<=9)
        {
            $end_of_quarter_date = strtotime('1-October-'.$current_year);  // timestamp or 1-October 12:00:00 AM means end of 30 September
        }
        else  if($current_month>=10 && $current_month<=12)
        {
            $end_of_quarter_date = strtotime('1-January-'.($current_year+1));  // timestamp or 1-January Next year 12:00:00 AM means end of 31 December this year
        }
        return $end_of_quarter_date;
    }

    function getRealQuery($query, $dumpIt = false) {
        $params = array_map(function ($item) {
            return "'{$item}'";
        }, $query->getBindings());
        $result = str_replace_array('\?', $params, $query->toSql());
        if ($dumpIt) {
            dd($result);
        }
        return $result;
    }
}