<?php
namespace Marcolin\Models;

use AlgoliaSearch\Laravel\AlgoliaEloquentTrait;
use Auth;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Config;
use Marcolin\Models\Traits\UploadableTrait;
use Marcolin\Models\Traits\RestrictableTrait;

class Content extends WebModel {
    use UploadableTrait;
    use RestrictableTrait;
    use HasCommitSequence;

    protected $table = 'CMSContent';
    protected $fillable = ['name', 'description', 'expiration_date', 'type', 'featured'];

    public function content_Algolia()
    {
        return $this->hasMany('Marcolin\Models\Content_Algolia', 'cmsContent_id', 'id');
    }

    public function publisher()
    {
        return $this->belongsTo('Marcolin\Models\WebUser', 'publisher_webUser_id');
    }

    public function categories() {
        return $this->belongsToMany('Marcolin\Models\Category', 'CMSCategory_CMSContent', 'cmsContent_id', 'cmsCategory_id');
    }

    public function scopeCollections($q)
    {
        $user = Auth::user();
        $salesRep = $user? Auth::user()->associatedSalesRep() : false;
        if ($salesRep) {
            $brandNames = $salesRep->brandNames();
            return $q->where(function (Builder $where) use ($brandNames) {
                $where->doesntHave('tags')->orWhereHas('tags', function (Builder $where) use ($brandNames) {
                    $where->where('group', '=', 'other')
                        ->orWhere(function (Builder $where) use ($brandNames) {
                            $where->where('group', '=', 'collection')
                                ->whereIn('name', $brandNames);
                        });
                });
            });
        } else {
            return $q;
        }
    }

    public function selectedSectionIndices() {
        $selectedSections = [];
        foreach ($this->categories as $categoryContainingThisContent){
            $selectedSections[] = $categoryContainingThisContent->id;
        }
        return $selectedSections;
    }

    public function section() {
        return $this->categories()->first();
    }

    public function subscribedSection(WebUser $user) {
        return $this->section();
        //not sure why the code below doesn't work for sub-subsections.  not going to spend time finding out right now.
//
//        $ids = [];
//        foreach($user->subscriptions as $subscription) {
//            $ids[$subscription->category->id] = $subscription->category->id;
//            foreach ($subscription->category->allChildrenFlat() as $child){
//                $ids[$child->id] = $child->id;
//            }
//        }
//        return $this->categories()->where(function ($query) use ($ids){
//            $query->whereIn('CMSCategories.id', $ids);
//        })->first();
    }

    public function tags() {
        return $this->morphToMany('Marcolin\Models\Tag', 'taggable', 'CMSTaggables', 'cmsTaggable_id', 'cmsTag_id');
    }

    public function restrictions() {
        return $this->morphToMany('Marcolin\Models\WebRole', 'restrictable', 'CMSRestrictables', 'cmsRestrictable_id', 'webRole_id');
    }

    public function hasRestriction($roleID) {
        foreach ($this->restrictions as $role) {
            if ($role->id == $roleID) return true;
        }
        return false;
    }

    public function collectionTags() {
        return $this->tags()->whereGroup('collection');
    }

    public function tag(Tag $tag) {
        $this->tags()->save($tag);
    }

    public function untag(Tag $tag) {
        $this->tags()->detach($tag);
    }

    public function untagAll() {
        foreach ($this->tags as $tag) {
            $this->untag($tag);
        }
        $this->save();
    }

    public function primaryUploadUploadDate() {
        $primaryUpload = $this->getPrimaryUpload();
        if ($primaryUpload){
            $timestamp = strtotime($primaryUpload->date_created);
        }else{
            $timestamp = strtotime($this->date_created);
        }
        return date('M d, Y', $timestamp);
    }

    /**
     * @return Upload
     */
    public function getPrimaryUpload() {
        return $this->firstOfUploadType('primary');
    }

    public function hasPrimaryUpload() {
        return $this->hasUploadType('primary');
    }

    public function getThumbPath() {
        if($this->hasUploadType('thumb')) {
            return $this->firstOfUploadType('thumb')->getContentsPath();
        } else {
            return Config::get('imagedefaults.content_thumb');
        }
    }

    public function getActionIcon() {
        if(Config::has('imagedefaults.action_icon_' . $this->type)) {
            return Config::get('imagedefaults.action_icon_' . $this->type);
        } else {
            return Config::get('imagedefaults.action_icon_default');
        }
    }

    public function shortName($length = 30) {
        return strlen($this->name) > $length ? substr($this->name,0,$length)."..." : $this->name;
    }

    public static function scopeFeatured($query) {
        return $query->where('featured','=',1);
    }

    public static function getCreateValidatorRules() {
        return [
            'name' => 'required',
            'expiration' => 'requires'
        ];
    }

    use AlgoliaEloquentTrait;

    public static $perEnvironment = true;

    public $algoliaSettings = [
        'searchableAttributes' => [
            'name',
            'description',
            'key',
            'type',
            'categories.name',
            'tags.name',
            'publisher.name',
            'publisher.username',
            'publisher.email',
        ],
        'attributesForFaceting' => [
            'categories.name',
            'tags.name',
            'type',
            'publisher.name',
        ]
    ];

    public function _reindex($safe = true, $setSettings = true, $mergeOldSettings = false, \Closure $onInsert = null)
    {
        /** @var \AlgoliaSearch\Laravel\ModelHelper $modelHelper */
        $modelHelper = App::make('\AlgoliaSearch\Laravel\ModelHelper');

        $indices = $modelHelper->getIndices($this);
        $indicesTmp = $safe ? $modelHelper->getIndicesTmp($this) : $indices;

        if ($setSettings === true) {
            $setToTmpIndices = ($safe === true);
            $this->_setSettings($setToTmpIndices, $mergeOldSettings);
        }

        static::has('categories')
            ->with('categories')
            ->with('tags')
            ->with('publisher')
            ->chunk(500, function ($models) use ($indicesTmp, $modelHelper, $onInsert) {
                /** @var \AlgoliaSearch\Index $index */
                foreach ($indicesTmp as $index) {
                    $records = [];
                    $recordsAsEntity = [];

                    foreach ($models as $model) {
                        if ($modelHelper->indexOnly($model, $index->indexName)) {
                            $records[] = $model->getAlgoliaRecordDefault($index->indexName);

//                            dd($records);

                            if ($onInsert && is_callable($onInsert)) {
                                $recordsAsEntity[] = $model;
                            }
                        }
                    }

                    $index->addObjects($records);

                    if ($onInsert && is_callable($onInsert)) {
                        call_user_func_array($onInsert, [$recordsAsEntity]);
                    }
                }

            });

        if ($safe) {
            for ($i = 0; $i < count($indices); $i++) {
                $modelHelper->algolia->moveIndex($indicesTmp[$i]->indexName, $indices[$i]->indexName);
            }

            $this->_setSettings(false); // To a setSettings to set the slave on the master
        }
    }

    public function indexOnly()
    {
        return true;
    }

    public function getAlgoliaRecord()
    {
        $record = array_only($this->toArray(), [
            'id',
        ]);

        $record['name'] = $this->name;
        $record['description'] = $this->description;
        $record['key'] = $this->key;
        $record['type'] = $this->type;
        $record['publisherName'] = $this->publisher->name;
        $record['publisherUsername'] = $this->publisher->username;
        $record['publisherEmail'] = $this->publisher->email;

        foreach ($this->categories as $key => $category) {
            $categoryArray = $category->toArray();

            $unwanted_keys = ['date_created', 'date_modified', 'date_uploaded', 'deleted_at', 'commit_sequence', 'parent', 'parent_cmsCategory_id',];

            $this->recursive_unset($categoryArray, $unwanted_keys);
            $record['categories'][] = $categoryArray;
        }

        foreach ($this->tags as $key => $tag) {
            $tagArray = $tag->toArray();

            $unwanted_keys = ['date_created', 'date_modified', 'date_uploaded', 'deleted_at', 'commit_sequence',];

            $this->recursive_unset($tagArray, $unwanted_keys);
            $record['tags'][] = $tagArray;
        }

        $unwanted_keys = ['date_created', 'date_modified', 'date_uploaded', 'deleted_at', 'commit_sequence'];
        $this->recursive_unset($record, $unwanted_keys);

        return $record;
    }

    function recursive_unset(&$array, $unwanted_keys)
    {
        foreach ($unwanted_keys as $unwanted_key) {
            unset($array[$unwanted_key]);
        }
        foreach ($array as &$value) {
            if (is_array($value)) {
                $this->recursive_unset($value, $unwanted_keys);
            }
        }
    }

    protected $indexSettings = [
        'analysis' => [
            'char_filter' => [
                'replace' => [
                    'type' => 'mapping',
                    'mappings' => [
                        '&=> and '
                    ],
                ],
            ],
            'filter' => [
                'word_delimiter' => [
                    'type' => 'word_delimiter',
                    'split_on_numerics' => false,
                    'split_on_case_change' => false,
                    'generate_word_parts' => true,
                    'generate_number_parts' => true,
                    'catenate_all' => true,
                    'preserve_original' => true,
                    'catenate_numbers' => true,
                ]
            ],
            'analyzer' => [
                'default' => [
                    'type' => 'custom',
                    'char_filter' => [
                        'html_strip',
                        'replace',
                    ],
                    'tokenizer' => 'whitespace',
                    'filter' => [
                        'lowercase',
                        'word_delimiter',
                    ],
                ],
            ],
        ],
    ];

    protected $mappingProperties = [
        'title' => [
            'type' => 'string',
            'analyzer' => 'standard'
        ]
    ];
}
