<?php

namespace Hilco\Models;
use AlgoliaSearch\Laravel\AlgoliaEloquentTrait;
use DB;
use Auth;
use Elasticquent\ElasticquentTrait;
use Hilco\Facades\HilcoAssets;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Venturecraft\Revisionable\RevisionableTrait;

/**
 * Class WebUnit
 * @package Hilco\Models
 * @property int $id
 * @property string $name
 * @property string $part_detail
 * @property string $safe_name
 * @property string $override_name
 * @property-read Unit $unit
 * @property-read WebUnitFamily $family
 * @property-read WebUnitAttributeValue[] $attributeValues
 * @method static Builder latest()
 * @property string $part_number
 * @property integer $part_id
 * @property integer $webunitfamily_id
 * @property string $date_created
 * @property string $date_modified
 * @property string $date_uploaded
 * @property string $deleted_at
 * @property-read \Hilco\Models\WebUnitFamily $webUnitFamily
 * @property-write mixed $web_part_attributes
 * @property-read mixed $part_name
 * @property-read mixed $family_name
 * @property-read mixed $context
 * @property-read mixed $default_context
 * @mixin \Eloquent
 * @property integer $webfamily_id
 * @method static WebUnit firstOrNew(array $attributes)
 */
class WebUnit extends WebModel
{
    protected $table = "WebUnits";
    protected $fillable = ['min_quantity', 'quantity_step', 'quantity_presets', 'name', 'webAttributes', 'keywords', 'webfamily_id', 'unit_id', 'part_number', 'is_visible', 'is_family_image', 'in_stock_label', 'out_of_stock_label'];
    protected $casts = [
        ['is_visible' => 'boolean' , 'is_family_image' => 'boolean']
    ];

    use AlgoliaEloquentTrait;

    public $algoliaSettings = [
        'searchableAttributes' => [
            'name',
            'part_number',
            'nodash_part_number',
        ],
        'attributesForFaceting' => [
            'webSiloIds',
            'validPlants',
            'is_rx',
        ]
    ];

    public function _reindex($safe = true, $setSettings = true, $mergeOldSettings = false, \Closure $onInsert = null)
    {
        /** @var \AlgoliaSearch\Laravel\ModelHelper $modelHelper */
        $modelHelper = App::make('\AlgoliaSearch\Laravel\ModelHelper');

        $indices = $modelHelper->getIndices($this);
        $indicesTmp = $safe ? $modelHelper->getIndicesTmp($this) : $indices;

        if ($setSettings === true) {
            $setToTmpIndices = ($safe === true);
            $this->_setSettings($setToTmpIndices, $mergeOldSettings);
        }

//        DB::enableQueryLog();
        static::has('webStyle.webCollections.webCategories.webBrands.webSilos')->whereHas('unit', function ($query) {
            $query->whereHas('inventoryItems', function ($query) {
                $query->isValid();
            });
        })
        ->where('is_visible', 1)
        ->with('unit.inventoryItems', 'webStyle.webCollections.webCategories.webBrands.webSilos')->chunk(100, function ($models) use ($indicesTmp, $modelHelper, $onInsert) {
            /** @var \AlgoliaSearch\Index $index */
//            dd(DB::getQueryLog());
//            DB::flushQueryLog();
            foreach ($indicesTmp as $index) {
                $records         = [];
                $recordsAsEntity = [];

                foreach ($models as $model) {
                    if ($modelHelper->indexOnly($model, $index->indexName)) {
                        $records[] = $model->getAlgoliaRecordDefault($index->indexName);

                        if ($onInsert && is_callable($onInsert)) {
                            $recordsAsEntity[] = $model;
                        }
                    }
//                    dd(DB::getQueryLog());
                }

                $index->addObjects($records);

                if ($onInsert && is_callable($onInsert)) {
                    call_user_func_array($onInsert, [$recordsAsEntity]);
                }
            }

        });

        if ($safe) {
            for ($i = 0; $i < count($indices); $i++) {
                $modelHelper->algolia->moveIndex($indicesTmp[$i]->indexName, $indices[$i]->indexName);
            }

            $this->_setSettings(false); // To a setSettings to set the slave on the master
        }
    }

    public function indexOnly() {
        if (!$this->is_visible) return false;
        if (!$this->webStyle->is_visible) return false;

        return true;
    }

    public function getAlgoliaRecord() {
        $record = array_only($this->toArray(), [
            'id',
            'part_id',
            'name',
            'part_number',
            'webStyle',
        ]);
        $record['nodash_part_number'] = str_replace('-', '', $this->unit_number);
        $paths = $this->hierarchyPaths;
        $silos = [];
        foreach ($paths as $path => $info) {
            $silos[$info['webSilo']->id] = $info['webSilo']->id;
        }
        $record['webSiloIds'] = array_keys($silos);
        $record['webStyle']['slug'] = !is_null($this->webStyle) ? $this->webStyle->slug : null;
        $record['validPlants'] = !is_null($this->unit) && !is_null($this->unit->inventoryItems) ? $this->unit->inventoryItems->pluck('plant') : [];
        $record['webCollectionIds'] = !is_null($this->webStyle) ? $this->webStyle->webCollections->pluck('id') : [];
        $record['is_rx'] = $this->unit->is_rx;
        return $record;
    }

    protected $indexSettings = [
        'analysis' => [
            'char_filter' => [
                'replace' => [
                    'type' => 'mapping',
                    'mappings' => [
                        '&=> and '
                    ],
                ],
            ],
            'filter' => [
                'word_delimiter' => [
                    'type' => 'word_delimiter',
                    'split_on_numerics' => false,
                    'split_on_case_change' => false,
                    'generate_word_parts' => true,
                    'generate_number_parts' => true,
                    'catenate_all' => true,
                    'preserve_original' => true,
                    'catenate_numbers' => true,
                ]
            ],
            'analyzer' => [
                'default' => [
                    'type' => 'custom',
                    'char_filter' => [
                        'html_strip',
                        'replace',
                    ],
                    'tokenizer' => 'whitespace',
                    'filter' => [
                        'lowercase',
                        'word_delimiter',
                    ],
                ],
            ],
        ],
    ];

    protected $mappingProperties = [
        'title' => [
            'type' => 'string',
            'analyzer' => 'standard'
        ]
    ];

//    use ElasticquentTrait;

    public static $elasticFuzziness = 1;

    /**
     * @description return elastic query with fuzziness
     */
    public static function buildElasticQuery($keyWords){
        $params=[
            'multi_match' => [
                'query'     => $keyWords,
                'fuzziness' => self::$elasticFuzziness,
                'fields'    => ['_all']
            ]
        ];
        return $params;
    }

//	use KeyWordsFunctional;

	public static function clearIsFamilyImage($familyId, $notIn=[]){
		self::where('webfamily_id', $familyId)->where('is_family_image',1)->whereNotIn('id',$notIn)->update(['is_family_image'=>0]);
	}

    /**
     * @description index all the entries in an Eloquent model (if needed)
     */
    public static function elasticIndexingIfNeeded(){
//		$elasticClient = ClientBuilder::create()->setHosts(config('elasticquent.config.hosts'))->build();
//		$instance = new static;

//		$basicParams = $instance->getBasicEsParams();
//		$indexParams = ['index' => $basicParams['index']];
//
//		$result = $elasticClient->indices()->exists($indexParams);
//		if (!$result)

//		self::createIndex();
        $mappingExists = self::mappingExists();
        if (!$mappingExists) {
            self::createIndex();
            self::putMapping();
            self::addAllToIndex();
        }

//		if(!$result || !$mappingExists)
//		return (!$result || !$mappingExists);
	}
	
	public function unit() {
		return $this->belongsTo(Unit::class, 'unit_id', 'id');
	}

	/**
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function webStyle() {
		return $this->belongsTo(WebStyle::class, 'webfamily_id', 'id');
	}

	public function promos() {
		return $this->belongsToMany(Promo::class, 'Promo_WebUnit','webunit_id', 'promo_id');
	}

	public function scopeJoinDiscount($query){
		$customerId = Auth::user()->customer_id;
		return $query->select(['WebUnits.*',DB::raw('SUM(CustomerDiscounts.disc_val) as disc_val')])->leftJoin('CustomerDiscounts', function($query)use($customerId){
			return $query->on('CustomerDiscounts.part_id','=','WebUnits.id')
						 ->orOn('CustomerDiscounts.productfamily_id','=','WebUnits.webfamily_id')
						 ->where('CustomerDiscounts.customer_id','=',$customerId);
		})->groupBy('id');
	}

	public function scopeIsFamilyImage($query){
		$query->where('is_family_image',1)->visible()->whereHas('part', function ($query) {
            return $query->whereHas('inventoryItems', function ($query) {
                return $query->isValid();
            });
        });
	}

	public function scopePriceList($query, $priceList, $currency, $quantityLevel = 1) {
		return $query
			->leftJoin('PriceLists', 'PriceLists.part_no', '=', 'WebUnits.part_number')
			->where('price_list', $priceList)
			->where('currency', $currency)
			->where('quantity_level', '>=', $quantityLevel)
			->select('WebUnits.*', 'PriceLists.price')
		;
	}

	public function webAttributes() {
		return $this->belongsToMany(WebAttribute::class, 'WebAttribute_WebUnit', 'webunit_id', 'webattribute_id')
			->withPivot(['id','attribute_value']);
	}

	public function webAttributeValue($webAttributeId) {
		if (!$this->relationLoaded('webAttributes')) $this->load('webAttributes');
		$webAttributeValue = $this->webAttributes->find($webAttributeId);
		if (!is_null($webAttributeValue)) return $webAttributeValue->pivot->attribute_value;
		return null;
	}

	public function setWebAttributeValue($webAttributeId, $value) {
		$this->webAttributes()->sync([$webAttributeId => ['attribute_value' => $value]], false);
	}
	public function setWebAttributesAttribute($attributes) {
		$this->save();
		foreach ($attributes as $webAttributeId => $attributeValue) {
			$this->setWebAttributeValue($webAttributeId, $attributeValue);
		}
	}

    public function relatedParent() {
        return $this->morphToMany(WebStyle::class, 'related', 'WebStyle_Related', 'related_id', 'webfamily_id')->withPivot(['related_category']);
    }

	public function getPartNameAttribute() {
		$part = $this->unit;
		if (is_null($part)) return null;
		return $part->unit_desc;
	}
	
	public function getFamilyNameAttribute() {
		$family = $this->webStyle;
		if (is_null($family)) return null;
		return $family->name;
	}

    public function getCustomerPrice($customer) {
        return $this->unit->getCustomerPrice($customer);
    }

    public function getCustomerDiscountAttribute() {
	    return $this->unit->customerDiscount;
    }

//	public static function getImageUrl($partNumber, $width = null, $height = null, $avoidCache=false) {
//		$url = config('marcolin.images.url');
//		if(!isset($url)) $url = config('hilco-b2b.images.url');
//		$url .= '/' . $partNumber;
//		if ($width) $url .= "/$width";
//		if ($height) $url .= "/$height";
//
//		return ($avoidCache)?\ImgHelper::avoidCache($url):$url;
//	}

    public function getImageUrl($width = false, $height = false) {
        return HilcoAssets::part($this, $width, $height);
    }

	public function scopePriceRange($query, $min, $max) {
		return $query->whereHas('part', function($part) use ($min, $max){
			$part->whereBetween('list_price', [$min, $max]);
		});
	}

	public function scopeManagedBy($query, $manager) {
		return $query->whereHas('part', function ($query) use ($manager) {
			return $query->where('pf_prod_mgr', $manager);
		});
	}

	public function scopeInFamily($query, $webStyleId) {
		return $query->whereHas('webStyle', function ($query) use ($webStyleId) {
			return $query->where('webfamily_id', $webStyleId);
		});
	}

    public function scopeInCollection($query, $webCollectionId) {
        return $query->whereHas('webStyle', function ($query) use ($webCollectionId) {
            return $query->inCollection($webCollectionId);
        });
    }

    public function scopeInCategory($query, $webCategoryId) {
        return $query->whereHas('webStyle', function ($query) use ($webCategoryId) {
            return $query->inCategory($webCategoryId);
        });
    }

	public function scopeInGroup($query, $webBrandId) {
		return $query->whereHas('webStyle', function ($query) use ($webBrandId) {
			return $query->inGroup($webBrandId);
		});
	}

	public function scopeVisible(Builder $query, $webSiloId = false) {
	    if ($webSiloId === false) {
	        $webSiloId = WebSilo::currentId();
        }
		return $query
			->where('WebUnits.is_visible', '1')
//			->whereHas('unit', function ($query) use ($webSiloId) {
//				$user = Auth::user();
//				$userCustomer = (isset($user))?$user->customer:null;
//				$pharmancyInfo = null;
//				if(isset($userCustomer)){
//					$pharmancyInfo = $userCustomer->pharmacyInfo()->first();
//				}
//				if(!isset($pharmancyInfo) || !$pharmancyInfo->checkRx()){
//					$query = $query->isNotRX();
//				}
//
//                $query->where('private_lbl', '0');
//                if (WebUser::activeCountry() != 'US') {
//                    $query->isNotRX();
//                }
//
//				return $query->whereHas('inventoryItems', function ($query) use ($webSiloId) {
//                    $activeCountry = WebUser::activeCountry();
//                    if ($activeCountry == 'US') $query->inPlant('PLAINVILLE');
//                    return $query->inSilo($webSiloId);
//				});
//			})
		;
	}

	public function visibleInSilo($webSilo) {
	    if (!$webSilo instanceof WebSilo) $webSilo = WebSilo::find($webSilo);
        $inSilo = true;
        return $inSilo;
    }

    public function getHasValidInventoryAttribute() {
        $status = 'none';
        $hasAll = true;
        if ($this->unit) {
            foreach ($this->unit->inventoryItems as $inventoryItem) {
                if ($inventoryItem->is_valid) {
                    $status = 'some';
                } else {
                    $hasAll = false;
                }
            }
        }

        if ($hasAll) $status = 'all';
        return $status;
    }

    public function getHierarchyPathsAttribute() {
        $isVisible = true;
        if (!$this->is_visible) $isVisible = 'Manually Hidden';
        if (is_null($this->unit)) $isVisible = 'Not in Point.man';
        if (is_null($this->webStyle)) $isVisible = 'Not in a family';
        if ($this->has_valid_inventory == 'none') $isVisible = 'No valid inventory';

        $paths = [];
        if (!is_null($this->webStyle)) {
            foreach ($this->webStyle->webCollections as $webCollection) {
                foreach ($webCollection->webCategories as $webCategory) {
                    foreach ($webCategory->webBrands as $webBrand) {
                        foreach ($webBrand->webSilos as $webSilo) {
                            if ($isVisible === true) {
                                if (!$this->visibleInSilo($webSilo)) $isVisible = 'Excluded due to silo rules';
                            }
                            if (!$this->webStyle->is_visible) $isVisible = 'Parent Family is not visible';
                            if (!$webCollection->is_visible) $isVisible = 'Parent Collection is not visible';
                            if (!$webCategory->is_visible) $isVisible = 'Parent Category is not visible';
                            if (!$webBrand->is_visible) $isVisible = 'Parent Group is not visible';
                            $paths[] = [
                                'webSilo' => $webSilo,
                                'webBrand' => $webBrand,
                                'webCategory' => $webCategory,
                                'webCollection' => $webCollection,
                                'webStyle' => $this->webStyle,
                                'isVisible' => $isVisible,
                            ];
                        }
                    }
                }
            }
        }
        return $paths;
    }

    public function hierarchyPaths() {
//        $data['webUnit'] = $this;
//        $isVisible = true;
//        if (!$this->is_visible) $isVisible = 'Manually Hidden';
//        if (is_null($this->unit)) $isVisible = 'Not in Point.man';
//        if (is_null($this->webStyle)) $isVisible = 'Not in a family';
//        if ($this->has_valid_inventory == 'none') $isVisible = 'No valid inventory';
//        $data['isVisible'] = $isVisible;
        $paths = ['webUnit' => $this];
        if ($this->webStyle) $paths = $this->webStyle->hierarchyPaths($paths);
        return $paths;
    }


    public function setWebfamilyIdAttribute($value) {
        if (!$value || $value < 1) $this->attributes['webfamily_id'] = null;
        else $this->attributes['webfamily_id'] = $value;
    }

    public function getInStockAttribute() {
        return $this->unit->in_stock;
    }

    public function getInStockLabelAttribute() {
        $modelValue = array_get($this->attributes, 'in_stock_label');
        return strlen($modelValue) ? $modelValue : config('marcolin.inStockLabels.0', 'In Stock');
    }

    public function getOutOfStockLabelAttribute() {
        $modelValue = array_get($this->attributes, 'out_of_stock_label');
        return strlen($modelValue) ? $modelValue : config('marcolin.outOfStockLabels.0', 'Out of stock');
    }

    public function getStockStatusAttribute() {
        return $this->unit->outOfStock;
    }

    public function getStockLabelAttribute() {
        $status = $this->stock_status;
        switch ($status) {
            case Unit::IN_STOCK:
                return $this->in_stock_label;
            case Unit::OUT_OF_STOCK:
                return $this->out_of_stock_label;
            case Unit::DELAYED:
                return "Ships in 3-5 days";
        }
        return '';
    }

	use HasAssets, RevisionableTrait;

}
