<?php

namespace Marcolin\Models;

use Auth;
use DB;
use Image as Image2;
use Storage;

/**
 * Marcolin\Models\Product
 *
 * @property mixed $id
 */
class Customer extends WebModel
{
    protected $table = 'Customers';

    public function brands()
    {
        return $this->belongsToMany(Brand::class, 'Brand_Customer')->wherePivot('deleted_at', '=', '0000-00-00 00:00:00');
    }

    public function salesDistrict()
    {
        return $this->belongsTo(SalesDistrict::class, 'salesDistrict_id', 'id');
    }

    public function customerCodeHierarchy()
    {
        return $this->belongsTo(CustomerCodeHierarchy::class, 'customerCodeHierarchy_id', 'id');
    }

    public function creditStatements()
    {
        return $this->hasMany(CreditStatement::class, 'customerCode_id', 'customerCode_id');
    }

    public function parentCustomer()
    {
        return $this->belongsTo(Customer::class, 'parent_customerCode_id', 'id');
    }

    public function childCustomers()
    {
        return $this->hasMany(Customer::class, 'parent_customerCode_id', 'parent_customerCode_id');
    }

    public function childCustomersShippable()
    {
        return $this->hasMany(Customer::class, 'parent_customerCode_id', 'parent_customerCode_id')
            ->where('shippable' ,'=', 1)
            ;
    }

    public function buyingGroup()
    {
        return $this->belongsTo(CustomerCodeHierarchy::class, 'customerCodeHierarchy_id', 'id');
    }

    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function webSilo() {
        return $this
            ->belongsToMany(WebSilo::class, 'Customer_WebSilo', 'customer_id', 'websilo_id')
//            ->withPivot(['can_order_hilco_parts', 'can_order_silo_parts', 'require_approval_hilco_parts', 'require_approval_silo_parts'])
            ->with('webLandingPage')
            ;
    }

    public function allSalesReps()
    {
        $salesReps = SalesRep::where('active', '=', 1)
            ->whereHas('salesDistrict.customers', function ($q) {
                return $q->where('id', '=', $this->id);
            })
            ->with('brands');
        return $salesReps->get();
    }

    public function activeSalesReps()
    {
        $brands = $this->brands->pluck('id');

        $salesReps = SalesRep::where('active', '=', 1)
            ->whereHas('salesDistrict.customers', function ($q) use ($brands) {
                return $q
                    ->where('Customers.id', '=', $this->id)
                    ->whereIn('SalesRep_Customer_Brand.brand_id', $brands);
            })
            ->with('brands');

        return $salesReps->get();
    }

    public function activeSalesRepsByBrand($brand_id)
    {
        $salesReps = SalesRep::where('active', '=', 1)
            ->where('deleted_at', '=', '0000-00-00 00:00:00')
            ->whereHas('salesDistrictTrashed.customersTrashed', function ($q) use ($brand_id) {
                return $q
                    ->where('Customers.id', '=', $this->id)
                    ->where('SalesRep_Customer_Brand.brand_id', '=', $brand_id);
            })
            ->with('brands');

        return $salesReps->get();
    }

    public function customerDiscounts()
    {
        return $this->hasMany(CustomerDiscount::class, 'customer_id', 'id');
    }

    public function orders()
    {
        return $this->hasMany(Order::class, 'soldTo_customer_id', 'id');
    }

    public function ordersShippedTo()
    {
        return $this->hasMany(Order::class, 'shipTo_customer_id', 'id');
    }


    public function salesOrders()
    {
        return $this->hasMany(SalesOrder::class, 'customer_id', 'id');
    }

    public function payerCustomer()
    {
        return $this->hasOne(Customer::class, 'customerCode', 'payerCode');
    }

    public function webUsers()
    {
        return $this->hasMany(WebUser::class, 'customer_id', 'id');
    }

    public function customerAging()
    {
        return $this->hasMany(CustomerAging::class, 'customer_id', 'id');
    }

    public function lastOrderedStyle($style)
    {
        return $this->orderHistories->where('style', $style)->sortByDesc('billingDate')->first();
    }

    public function getFormattedTelephoneAttribute()
    {
        return preg_replace('~.*(\d{3})[^\d]{0,7}(\d{3})[^\d]{0,7}(\d{4}).*~', '($1) $2-$3', $this->telephone) . "\n";
    }

    public function getFormattedTelephone2Attribute()
    {
        return preg_replace('~.*(\d{3})[^\d]{0,7}(\d{3})[^\d]{0,7}(\d{4}).*~', '($1) $2-$3', $this->telephone2) . "\n";
    }

    public function getFormattedFaxAttribute()
    {
        return preg_replace('~.*(\d{3})[^\d]{0,7}(\d{3})[^\d]{0,7}(\d{4}).*~', '($1) $2-$3', $this->telephone2) . "\n";
    }

    public function getProductVisibilityAttribute()
    {
        if ($this->brands->where('brandCode', 'HM')->count() > 0) {
            return 'harley';
        } else {
            return 'standard';
        }
    }

    public function getIsCreditHoldAttribute() {
        $customerAging = $this->customerAging->first();
        // If the customer aging record does not exist, the customer is ACTIVE.
        if (!isset($customerAging)) {
            return false;
        }

        $totalOverdue = $customerAging->totalOverdue;
        // If the customers total overdue amount is negative, the customer is ACTIVE.
        if (strpos($totalOverdue, '-') !== false) {
            return false;
        }

        $totalOverdue = str_replace('.', '', $totalOverdue);
        $totalOverdue = str_replace(',', '', $totalOverdue);
        // If the customers total overdue amount is 0, the customer is ACTIVE.
        if ($totalOverdue == 0) {
            return false;
        }

        // Otherwise the customer is on HOLD.
        return true;
    }

    public function getLockedBrands()
    {
        $brands = $this->brands->pluck('id');
        $lockedBrands = Brand::webVisible()->whereNotIn('id', $brands)->where('deleted_at', '=', '0000-00-00')->orderBy('brandName')->get();
        return $lockedBrands;
    }

    public function recentOrders($count = 5)
    {
        $orders = $this
            ->orders()
            ->with('orderItems')
            ->orderByDate()
            ->take($count)
            ->get();

        if ($orders->count() == 0) {
            $orders = $this
                ->ordersShippedTo()
                ->with('orderItems')
                ->orderByDate()
                ->take($count)
                ->get();
        }

        return $orders;
    }

    public function topPurchasedItems($count = 10)
    {
        $orders = $this
            ->orders()
            ->has('orderItems.unit.styleRelation.webStyle.webVisibleWebUnits.assets')
            ->with([
                'orderItems' => function ($q) {
                    return $q
                        ->has('unit.styleRelation.webStyle.webVisibleWebUnits.assets')
                        ->with([
                            'unit.styleRelation.webStyle.style',
                            'unit.styleRelation.webStyle.webVisibleWebUnits.assets',
                            'unit.styleRelation.webStyle.webVisibleWebUnits.unit.priceListValues',
                            'unit.styleRelation.webStyle.webVisibleWebUnits.unit.styleRelation',
                        ])
                        ;
                },
            ])
            ->orderBy('orderDate', 'desc')
            ->limit($count);

        $styles = [];
        foreach ($orders->get() as $order) {
            foreach ($order->orderItems as $orderItem) {
                $styles[] = $orderItem->unit->styleRelation->webStyle;
                if (count($styles) >= $count) {
                    break;
                }
            }

            if (count($styles) >= $count) {
                break;
            }
        }

        return $styles;
    }

    public function isVisionSource() {
        return $this->customerCodeHierarchy === '500142';
    }
}

