<?php

namespace Marcolin\Models;

/**
 * Class PromotionOrderValueTrigger
 * @package Marcolin\Models
 */
class PromotionOrderValueTrigger extends PromotionTriggerDetails {
    protected $table = 'PromotionOrderValueTriggers';
    protected $fillable = [
        'target_all_products',
        'minimum_value',
        'maximum_value',
        'target_sunglass', 'target_optical', 'target_clip',
    ];
    protected $casts = [
        'target_sunglass' => 'boolean',
        'target_optical' => 'boolean',
        'target_clip' => 'boolean',
    ];

    // Overrides Model::boot(), used to define a specific 'deleted' state
    public static function boot() {
        parent::boot();
        // make sure all related rows in PromotionOrderValueTrigger_Product are also deleted
        static::deleted(function (PromotionOrderValueTrigger $promotionOrderValueTrigger) {
            $promotionOrderValueTriggerProducts =
                PromotionOrderValueTrigger_Product
                    ::where('promotionordervaluetrigger_id', '=', $promotionOrderValueTrigger->id)
                    ->get();

            foreach ($promotionOrderValueTriggerProducts as $promotionOrderValueTriggerProduct) {
                $promotionOrderValueTriggerProduct->delete();
            }
        });
    }

    /**
     * Boolean, all promo trigger types need to implement to use their specific logic
     * @param Customer $soldToCustomer
     * @param $webCartItems
     * @return mixed
     */
    public function isTriggered(Customer $soldToCustomer, $webCartItems = []) {
        $totalItemValue = $this->getCurrentQualifyingCartTotal($soldToCustomer, $webCartItems);

        return
            $totalItemValue >= $this->minimum_value
            &&
            ($this->maximum_value == 0 || $totalItemValue <= $this->maximum_value)
        ;
    }

    /**
     * promo trigger types should implement if there post operations that should happen after promo is applied
     * @param $soldToCustomer
     * @return mixed
     */
    public function postTriggerOperation(Customer $soldToCustomer) {
        // TODO: Implement postTriggerOperation() method.
    }

    /**
     * Boolean, any promo trigger that can use the Almost Qualifying Header should implement this and return 'true'
     * otherwise return 'false'
     * @return bool
     */
    public function hasAlmostQualifyingField() {
        return true;
    }

    /**
     * Any promo trigger that can use the Almost Qualifying Header should implement this and
     * @param Customer $soldToCustomer
     * @param $webCartItems
     * @return mixed
     */
    public function isAlmostQualifying(Customer $soldToCustomer, $webCartItems) {
        $totalItemValue = $this->getCurrentQualifyingCartTotal($soldToCustomer, $webCartItems);

        return $totalItemValue < $this->minimum_value;
    }

    /**
     * Get all WebBrands specifically targeted by this trigger
     * @return mixed
     */
    public function brands() {
        return $this
            ->morphedByMany(Brand::class, 'product',
                'PromotionOrderValueTrigger_Product', 'promotionordervaluetrigger_id')
            ->where('PromotionOrderValueTrigger_Product.deleted_at', '=', '0000-00-00 00:00:00')
            ;
    }

    /**
     * Get all WebStyles specifically targeted by this trigger
     * @return mixed
     */
    public function styles() {
        return $this
            ->morphedByMany(Style::class, 'product',
                'PromotionOrderValueTrigger_Product', 'promotionordervaluetrigger_id')
            ->where('PromotionOrderValueTrigger_Product.deleted_at', '=', '0000-00-00 00:00:00')
            ;
    }

    /**
     * Get all WebUnits specifically targeted by this trigger
     * @return mixed
     */
    public function units() {
        return $this
            ->morphedByMany(Unit::class, 'product',
                'PromotionOrderValueTrigger_Product', 'promotionordervaluetrigger_id')
            ->where('PromotionOrderValueTrigger_Product.deleted_at', '=', '0000-00-00 00:00:00')
            ;
    }

    /**
     * Gets the total value of 'qualifying' items in the cart, where 'qualifying' is defined as
     * - contained in the list of targeted web products (units, styles, brands), if not targeting all products
     * @param Customer $customer
     * @param $webCartItems
     * @return int
     */
    public function getCurrentQualifyingCartTotal(Customer $customer, $webCartItems) {
        $totalItemValue = 0;

        if (isset($webCartItems) && count($webCartItems) > 0) {
            if ($this->target_all_products == 1) {
                foreach ($webCartItems as $webCartItem) {
                    $totalItemValue = $totalItemValue + $webCartItem->getExtendedPrice($customer);
                }
            } else if ($this->target_all_products == 0) {
                $relevantUnits = $this->units;
                $relevantStyles = $this->styles;
                $relevantBrands = $this->brands;

                foreach ($webCartItems as $webCartItem) {
                    $unit = $webCartItem->unit;
                    $productTypeCode = $unit->styleRelation;
                    $brand = $productTypeCode->brand;
                    if ((isset($unit) && $relevantUnits->contains($unit)) ||
                        (isset($productTypeCode) && $relevantStyles->contains($productTypeCode)) ||
                        (isset($brand) && $relevantBrands->contains($brand))) {
                        $totalItemValue = $totalItemValue + $webCartItem->getExtendedPrice($customer);
                    }
                }
            } else {
                foreach ($webCartItems as $webCartItem) {
                    $productTypeCode = $webCartItem->unit->styleRelation->productTypeObj->productType;
                    if (($this->target_sunglass && $productTypeCode == ProductType::SUNGLASS_CODE) ||
                        ($this->target_optical && $productTypeCode == ProductType::OPTICAL_CODE) ||
                        ($this->target_clip && $productTypeCode == ProductType::CLIP_CODE)) {

                        $totalItemValue = $totalItemValue + $webCartItem->getExtendedPrice($customer);
                    }
                }
            }
        }

        return $totalItemValue;
    }
}