<?php

namespace Marcolin\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Support\Facades\DB;

/**
 * Class WebPromotion
 * @package Hilco\Models
 *
 * @method static Builder active()
 */
class WebPromotion extends WebModel {
    protected $table = 'WebPromotions';
    protected $fillable = [
        'promotion_name', 'promotion_desc', 'is_enabled', 'is_visible', 'is_one_time', 'allow_for_daily',
        'start_date', 'end_date',
        'unqual_unapply_text', 'unqual_apply_text', 'qual_unapply_text', 'qual_apply_text',
        'unqual_unapply_title', 'unqual_apply_title', 'qual_unapply_title', 'qual_apply_title',
        'homepage_banner_text', 'homepage_banner_link',
        'commit_sequence'
    ];
    protected $casts = [
        'is_enabled' => 'boolean',
        'is_visible' => 'boolean',
        'is_one_time' => 'boolean',
    ];

    // Overrides Model::boot(), used to define custom 'deleted' state
    public static function boot() {
        parent::boot();

        // make sure the required code and 'eligibility' triggers are created with the promotion
        static::created(function (WebPromotion $webPromotion) {
            $webPromotion->triggers()->create(['trigger_type' => 'codeTrigger']);
            $webPromotion->triggers()->create(['trigger_type' => 'customerTrigger']);
//            $promotion->triggers()->create(['trigger_type' => 'sourceTrigger']); -- no need for sourceTrigger until ESB also syncs promotions
            $webPromotion->actions()->create(['action_type' => 'orderNoteAction']);
        });

        // make sure all associated triggers and actions are also deleted
        static::deleted(function (WebPromotion $webPromotion) {
            foreach ($webPromotion->triggers as $trigger) {
                $trigger->delete();
            }
            foreach ($webPromotion->actions as $action) {
                $action->delete();
            }
        });
    }

    // Get "Datepicker" attribute values for Start Date & End Date
    public function getStartDateDatepickerAttribute() {
        $date = ($this->attributes['start_date'] == '0000-00-00') ?
            Carbon::create(2017, 1, 1) :
            Carbon::createFromFormat('Y-m-d', $this->attributes['start_date']);

        return $date->format('Y-m-d');
    }

    public function getEndDateDatepickerAttribute() {
        $date = ($this->attributes['end_date'] == '0000-00-00') ?
            Carbon::now()->addYear(1) :
            Carbon::createFromFormat('Y-m-d', $this->attributes['end_date']);

        return $date->format('Y-m-d');
    }

    // Getters for the 'unqualified' text displays,
    // handles replacing the 'QTY_LEFT', 'AMT_LEFT' strings with the actual remainder values
    public function getUnqualUnapplyTextForWebAttribute() {
        $unqual_unapply_text = $this->unqual_unapply_text;
        $remainders = $this->getRemaindersToQualify();
        if (count($remainders)) {
            foreach ($remainders as $needle => $replacement) {
                $unqual_unapply_text = preg_replace($needle, $replacement, $unqual_unapply_text);
            }
        }
        return $unqual_unapply_text;
    }

    public function getUnqualApplyTextForWebAttribute() {
        $unqual_apply_text = $this->unqual_apply_text;
        $remainders = $this->getRemaindersToQualify();
        if (count($remainders)) {
            foreach ($remainders as $needle => $replacement) {
                $unqual_apply_text = preg_replace($needle, $replacement, $unqual_apply_text);
            }
        }
        return $unqual_apply_text;
    }


    // Status functions
    // Is promo eligible to be applied?
    public function isEligible(Customer $customer) {
        $allEligible = true;
        foreach ($this->eligibilityTriggers as $eligibilityTrigger) {
            $allEligible = $allEligible && $eligibilityTrigger->details->isTriggered($customer);
        }
        return $allEligible;
    }

    // If applied, are all promo requirements fully satisfied?
    public function isQualifying(Customer $customer, $webCartItems) {
        $allQualified = true;
        foreach ($this->qualificationTriggers as $qualificationTrigger) {
            $allQualified = $allQualified && $qualificationTrigger->details->isTriggered($customer, $webCartItems);
        }
        return $allQualified;
    }

    // Is this promo allowed to be used on daily orders?
    public function isAllowedForDailyOrders() {
        return $this->allow_for_daily != 0;
    }

    // Is this promo ONLY for use on daily orders?
    public function isOnlyForDailyOrders() {
        return $this->allow_for_daily == 2;
    }


    /**
     * Get array of remainder values a promotion needs to finish qualifying.
     * @param Customer|null $customer
     * @param null $webCartItems
     * @return array of placeholders to remainder values, e.g.
     * [
     *     '/QTY_LEFT/' => 5,
     *     '/AMT_LEFT/' => 25,
     *     '/QTY_MAX/'  => 10
     * ]
     */
    public function getRemaindersToQualify(Customer $customer = null, $webCartItems = null) {
        if (!isset($customer)) {
            $customer = b2b()->activeCustomer();
            if (!$customer) return [];
        }
        if (!isset($webCartItems)) {
            $webCartItems = app('cartSingleton')->getCartItems();
        }

        $remainderValues = [];
        foreach ($this->qualificationTriggers as $qualificationTrigger) {
            if ($qualificationTrigger->trigger_type === 'orderQuantityTrigger') {
                $qtyRemaining = 0;
                if (!$qualificationTrigger->details->isTriggered($customer, $webCartItems)) {
                    $qtyRemaining =
                        $qualificationTrigger->details->minimum_quantity -
                        $qualificationTrigger->details->getCurrentQualifyingCartTotal($webCartItems);
                }

                $remainderValues['/QTY_LEFT/'] = ($qtyRemaining > 0) ? $qtyRemaining : 0;

                if ($qualificationTrigger->details->maximum_quantity > 0) {
                    $remainderValues['/QTY_MAX/'] = $qualificationTrigger->details->maximum_quantity;
                }

            } else if ($qualificationTrigger->trigger_type === 'orderValueTrigger') {
                $amtRemaining = 0;
                if (!$qualificationTrigger->details->isTriggered($customer, $webCartItems)) {
                    $amtRemaining =
                        $qualificationTrigger->details->minimum_value -
                        $qualificationTrigger->details->getCurrentQualifyingCartTotal($customer, $webCartItems);
                }

                $amtRemaining = ($amtRemaining > 0) ? $amtRemaining : 0;
                $remainderValues['/AMT_LEFT/'] = "\\" . b2b()->currencySymbol() . number_format($amtRemaining, 2);

                if ($qualificationTrigger->details->maximum_value > 0) {
                    $remainderValues['/AMT_MAX/'] = "\\" . b2b()->currencySymbol() . $qualificationTrigger->details->maximum_value;
                }
            }
        }
        return $remainderValues;
    }

    /**
     * Return true if the associated codeTrigger for this promotion uses a code list (instead of a single-code)
     * @return bool
     */
    public function hasCodeList() {
        $codeTrigger = $this->codeTriggers()->first();
        return $codeTrigger && $codeTrigger->is_list;
    }

    /**
     * Redeem a coupon (if applicable) on a promo.
     * @param $couponCode
     * @return bool
     */
    public function redeemCouponCode($couponCode) {
        $codeTrigger = $this->codeTriggers()->first();
        if ($codeTrigger && $codeTrigger->is_list) {
            $codeTriggerCode = $codeTrigger->codes()->where('coupon_code', $couponCode)->first();
            if ($codeTriggerCode) {
                $codeTriggerCode->is_redeemed = 1;
                $codeTriggerCode->save();
                return true;
            }
        }
        return false;
    }


    // Morph Map defines the polymorphic relationship between promotions and triggers/actions
    // (Registered in SafiloB2BSharedProvider.php)
    protected static $morphMap = [
        'codeTrigger'                   => PromotionCodeTrigger::class,
        'customerTrigger'               => PromotionCustomerTrigger::class,
        'lastOrderTrigger'              => PromotionLastOrderTrigger::class,
        'orderQuantityTrigger'          => PromotionOrderQuantityTrigger::class,
        'orderValueTrigger'             => PromotionOrderValueTrigger::class,
        'sourceTrigger'                 => PromotionSourceTrigger::class,

        'itemDiscountAction'            => PromotionItemDiscountAction::class,
        'orderNoteAction'               => PromotionOrderNoteAction::class,
        'shippingDiscountAction'        => PromotionShippingDiscountAction::class,
    ];


    // Relationships
    public function triggers() {
        return $this->hasMany(PromotionTrigger::class, 'promotion_id', 'id');
    }

    public function actions() {
        return $this->hasMany(PromotionAction::class, 'promotion_id', 'id');
    }

    public function redemptions() {
        return $this->hasMany(PromotionRedemptions::class, 'promotion_id', 'id');
    }

    public function eligibilityTriggers() {
        return $this->triggers()->onlyEligibility();
    }

    public function qualificationTriggers() {
        return $this->triggers()->onlyQualification();
    }

    public function codeTriggers() {
        return $this->morphedByMany(PromotionCodeTrigger::class,
            'trigger', 'PromotionTriggers','promotion_id');
    }

    public function customerTriggers() {
        return $this->morphedByMany(PromotionCustomerTrigger::class,
            'trigger', 'PromotionTriggers', 'promotion_id');
    }

    public function orderQuantityTriggers() {
        return $this->morphedByMany(PromotionOrderQuantityTrigger::class,
            'trigger', 'PromotionTriggers', 'promotion_id');
    }

    public function orderValueTriggers() {
        return $this->morphedByMany(PromotionOrderValueTrigger::class,
            'trigger', 'PromotionTriggers', 'promotion_id');
    }

    public function sourceTriggers() {
        return $this->morphedByMany(PromotionSourceTrigger::class,
            'trigger', 'PromotionTriggers', 'promotion_id');
    }

    public function itemDiscountActions() {
        return $this->morphedByMany(PromotionItemDiscountAction::class,
            'action', 'PromotionActions', 'promotion_id');
    }

    public function orderNoteActions() {
        return $this->morphedByMany(PromotionOrderNoteAction::class,
            'action', 'PromotionActions', 'promotion_id');
    }

    public function shippingDiscountActions() {
        return $this->morphedByMany(PromotionShippingDiscountAction::class,
            'action', 'PromotionActions', 'promotion_id');
    }


    /**
     * Get the morph mapping of trigger/action types to their actual class models
     * @return array
     */
    public static function getMorphMap() {
        return self::$morphMap;
    }

    /**
     * Get all promotions scoped by scopeActive()
     * @return \Illuminate\Database\Eloquent\Builder[]|\Illuminate\Database\Eloquent\Collection
     */
    public static function getActivePromotions() {
        return WebPromotion::active()->get();
    }

    /**
     * Get an array of 'eligible' promotions, where 'eligible' is defined as
     * - 'active', see scopeActive() doc
     * AND
     * - scoped by scopeEligible(), see its doc
     * AND
     * - 'targeting' the 'active customer'
     * where the 'active customer is defined as the aliased account OR the account
     * tied to the currently authenticated web user, and 'targeting' is defined as either specifically targeting the
     * customer with a customer trigger list or by targeting ALL customers
     * @param Customer $customer (defaults to use B2BHelper->activeCustomer
     * @return mixed
     */
    public static function getEligiblePromotions(Customer $customer = null) {
        if (!isset($customer)) {
            $customer = b2b()->activeCustomer();
            if (!$customer) return [];
        }

        $eligiblePromotions = [];
        foreach (WebPromotion::active()->unredeemed($customer)->get() as $activePromotion) {
            if ($activePromotion->isEligible($customer)) {
                $eligiblePromotions[] = $activePromotion;
            }
        }
        return $eligiblePromotions;
    }

    /**
     * @param Customer $customer
     * @param WebUser $webUser
     * @return array
     */
    public static function getUnQualifiedPromotions(Customer $customer = null, WebUser $webUser = null) {
        if (!isset($customer)) {
            $customer = b2b()->activeCustomer();
            if (!$customer) return [];
        }
        if (!isset($webUser)) {
            $webUser = auth()->user();
            if (!isset($webUser)) return [];
        }

        $unQualifiedPromotions = [];
        $webCartItems = WebCart::cartItems($webUser->id);
        foreach (WebPromotion::getEligiblePromotions($customer) as $eligiblePromotion) {
            if (!$eligiblePromotion->isQualifying($customer, $webCartItems)) {
                $unQualifiedPromotions[] = $eligiblePromotion;
            }
        }
        return $unQualifiedPromotions;
    }

    /**
     * @param Customer|null $customer
     * @param WebUser|null $webUser
     * @return array
     */
    public static function getQualifiedPromotions(Customer $customer = null, WebUser $webUser = null) {
        if (!isset($customer)) {
            $customer = b2b()->activeCustomer();
            if (!$customer) return [];
        }
        if (!isset($webUser)) {
            $webUser = auth()->user();
            if (!isset($webUser)) return [];
        }

        $qualifiedPromotions = [];
        $webCartItems = WebCart::cartItems($webUser->id);
        foreach (WebPromotion::getEligiblePromotions($customer) as $eligiblePromotion) {
            if ($eligiblePromotion->isQualifying($customer, $webCartItems)) {
                $qualifiedPromotions[] = $eligiblePromotion;
            }
        }
        return $qualifiedPromotions;
    }


    // Scopes
    /**
     * Get all 'active' promotions, where active is defined as
     * - is_enabled == TRUE
     * AND
     * - current date is between the start and end dates OR after the start date OR before the end date, if either date is set
     * @param $query
     * @return mixed
     */
    public function scopeActive ($query) {
        return
            $query
                ->where('is_enabled', 1)
                ->where(function ($query) {
                    return
                        $query
                            ->where('start_date', '=', '0000-00-00 00:00:00')
                            ->orWhere('start_date', '<=', DB::raw('DATE(NOW())'))
                        ;
                })
                ->where(function ($query) {
                    return $query
                        ->where('end_date', '=', '0000-00-00 00:00:00')
                        ->orWhere('end_date', '>=', DB::raw('DATE(NOW())'))
                        ;
                })
            ;
    }

    /**
     * Get all promotions for 'eligibility' evaluation, based on the following:
     * - not one-time-use
     * OR
     * - one-time-use AND currently unredeemed by the 'active customer', where 'active customer' is defined as the
     * customer currently being aliased or otherwise associated with the authenticated web user (see B2BHelper.php)
     * @param $query
     * @param Customer $customer
     * @return mixed
     */
    public function scopeUnredeemed ($query, Customer $customer) {
        return
            $query
                ->where('is_one_time', 0)
                ->orWhere(function ($query) use ($customer) {
                    return
                        $query
                            ->where('is_one_time', 1)
                            ->whereNotExists(function ($query) use ($customer) {
                                return
                                    $query
                                        ->select(DB::raw('1'))
                                        ->from('PromotionRedemptions')
                                        ->whereRaw('promotion_id = WebPromotions.id')
                                        ->where('customer_id', $customer->id)
                                        ->where('deleted_at', '=', '0000-00-00 00:00:00')
                                    ;
                            })
                        ;
                })
            ;
    }

    public static function createActionFromType($actionType)
    {
        $class = self::getActionClassFromType($actionType);
        return new $class;
    }

    public static function getActionClassFromType($actionType)
    {
        return array_get(Relation::morphMap(), $actionType, $actionType);
    }

    public function getActionRelationFromType($actionType)
    {
        $relationName = $actionType . 's';
        $relation = $this->$relationName();
        return $relation;
    }
}
