<?php

namespace Marcolin\Models;

    use Faker\Provider\Uuid;
    use Marcolin\Facades\MarcolinAssets;
    use Illuminate\Database\Eloquent\Relations\Relation;
    use Illuminate\Http\UploadedFile;
    use Illuminate\Support\Facades\Storage;
    use Symfony\Component\HttpFoundation\File\File;
    use Symfony\Component\HttpFoundation\FileBag;

class WebAsset extends WebModel {
    protected $table = 'WebAssets';
    protected $fillable = ['authorized_only', 'asset_key', 'asset_category', 'original_filename', 'mime_type', 'name', 'description'];
    protected $appends = ['full_key', 'url', 'urls'];

    protected static $morphMap = [
        'unit' => Unit::class,
        'style' => Style::class,
        'brand' => Brand::class,

        'webUnit' => WebUnit::class,
        'webStyle' => WebStyle::class,
        'webBrand' => WebBrand::class,
        'webSilo' => WebSilo::class,
    ];

    protected static $mimeTypes = [
        'image/jpeg' => [
            'type' => 'image',
            'subtype' => 'jpeg',
            'displayName' => 'Image',
            'encodeAs' => 'jpg',
            'isImage' => true,
        ],
        'image/png' => [
            'type' => 'image',
            'subtype' => 'png',
            'displayName' => 'Image',
            'encodeAs' => 'png',
            'isImage' => true,
        ],
        'image/bmp' => [
            'type' => 'image',
            'subtype' => 'bmp',
            'displayName' => 'Image',
            'encodeAs' => 'jpg',
            'isImage' => true,
        ],
        'application/pdf' => [
            'type' => 'application',
            'subtype' => 'pdf',
            'displayName' => 'Image',
            'encodeAs' => 'jpg',
            'isImage' => false,
        ],
    ];

    protected static $assetCategories = [
        'part' => 'Part Images',
        'image' => 'Other Images',
        'banner' => 'Banners',
        'document' => 'Documents/Forms',
    ];

    protected static $assetableJoinTypes = [
        'primary' => 'Primary Image',
        'thumbnail' => 'Thumbnail Image',
        'additional' => 'Additional Image',
        'featured' => 'Featured Image',
        'banner' => 'Banner',
    ];

    public static function getMorphMap() {
        return self::$morphMap;
    }

    public static function getAssetableClass($morphMapKey) {
        return array_get(self::$morphMap, $morphMapKey);
    }

    public static function getMimeInfo($mimeType) {
        return array_get(self::$mimeTypes, $mimeType);
    }

    public static function getAssetableModel($morphMapKey) {
        $className = self::getAssetableClass($morphMapKey);
        return ($className) ? new $className : null;
    }


    public static function getAssetCategory($category = false) {
        return $category !== false  ? array_get(self::$assetCategories, $category) : self::$assetCategories;
    }

    public static function getAssetableJoinType($type = false) {
        return $type !== false  ? array_get(self::$assetableJoinTypes, $type) : self::$assetableJoinTypes;
    }

    public function webUnits() {
        return $this->morphedByMany(WebUnit::class, 'assetable', 'Assetables', 'webasset_id', 'assetable_id')->withPivot(['sequence', 'assetable_join_type']);
    }

    public function webStyles() {
        return $this->morphedByMany(WebStyle::class, 'assetable', 'Assetables', 'webasset_id', 'assetable_id')->withPivot(['sequence', 'assetable_join_type']);
    }

    public function getFullKeyAttribute() {
        return $this->asset_category . '/' . $this->asset_key;
    }

    public function getUrlAttribute() {
        return WebAsset::getSizedUrl($this);
    }

    public function getDownloadUrlAttribute() {
        return WebAsset::urlHelper(['id' => $this->id, 'download' => true]);
    }

    public function getUrlsAttribute() {
        $urls['original'] = WebAsset::getSizedUrl($this);
        foreach (config('marcolin.assets.sizes') as $size => $dimensions) {
            $urls[$size] = WebAsset::getSizedUrl($this, $dimensions['width'], $dimensions['height']);
        }
        return $urls;
    }

    public function getIconAttribute() {
        if (starts_with($this->mime_type, 'image')) return 'fa-file-image-o';
        if ($this->mime_type == 'application/pdf') return 'fa-file-pdf-o';
        return 'fa-file-o';
    }

    public function getMimeInfoAttribute() {
        return array_get(self::$mimeTypes, $this->mime_type);
    }

    public function getIsImageTypeAttribute() {
        return array_get($this->mime_info, 'isImage', false);
    }

    public function getEncodeAsAttribute() {
        return array_get($this->mime_info, 'encodeAs', false);
    }

    public function getMimeNameAttribute() {
        return array_get($this->mime_info, 'displayName', false);
    }

    public function __get($key) {
        if (starts_with($key, 'url_')) {
            return self::getPredefinedSizeUrl($this, str_replace('url_', '', $key));
        }
        return parent::__get($key);
    }

    public function getFileAttribute() {
        Storage::disk('assets')->get($this->full_key);
    }

    public static function compileUrlForKey($fullKey, $width = false, $height = false) {
        $api = config('marcolin.assets.url');
        $url = "$api/$fullKey";
        if ($width) $url .= "/$width";
        if ($height) $url .= "/$height";
        return $url;
    }

    public static function compileUrlForStub($stub) {
        $api = config('marcolin.assets.url');
        $spacer = starts_with($stub, '/') ? '' : '/';
        return $api . $spacer . $stub;
    }

    public static function getPredefinedSizeUrl($webAsset, $size, $keyIfNotFound = false) {
        $size = config("marcolin.assets.sizes.$size", [
            'width' => false,
            'height' => false,
        ]);
        return self::getSizedUrl($webAsset, $size['width'], $size['height'], $keyIfNotFound);
    }

    public static function getSizedUrl($webAsset, $width = false, $height = false, $keyIfNotFound = false) {
        if (!is_null($webAsset) && !($webAsset instanceof WebAsset)) $webAsset = WebAsset::find($webAsset);
        if ($keyIfNotFound === false) $keyIfNotFound = config('marcolin.assets.nopic');
        return self::compileUrlForKey(is_null($webAsset) ? $keyIfNotFound : $webAsset->full_key, $width, $height);
    }

    public static function getUrl($webAsset, $keyIfNotFound = false) {
        return self::getSizedUrl($webAsset, false, false, $keyIfNotFound);
    }

    public function scopeAuthorizedOnly($query){
        return $query->where('authorized_only',1);
    }

    public function scopeForAll($query){
        return $query->where('authorized_only',0);
    }

    public static function urlHelper($options) {
        if (!is_array($options)) {
            $options = ['prefix' => 'id', 'key' => $options];
        }
        $url = config('marcolin.assets.url');
        $qs = [];

        if (array_has($options, 'id')) {
            $options['prefix'] = 'id';
            $options['key'] = $options['id'];
        }
        if (array_get($options, 'key')) {
            $url .= '/' . array_get($options, 'prefix', 'image');
            $url .= '/' . array_get($options, 'key');
        } else {
            $url .= '/' . config('marcolin.assets.nopic');
        }

        if (array_has($options, 'size')) {
            $size = config("marcolin.assets.sizes.$options[size]", false);
            if ($size) {
                $options['width'] = $size['width'];
                $options['height'] = $size['height'];
            }
        }
        if (array_get($options, 'width')) $url .= '/' . array_get($options, 'width');
        if (array_get($options, 'height')) $url .= '/' . array_get($options, 'height');

        if (array_get($options, 'download')) $qs['dl'] = '1';
        if (array_get($options, 'token')) {
            if (auth()->user()) {
                $token = createAssetAccessToken();
                $qs['tt'] = $token['token'];
                $qs['ts'] = $token['timestamp'];
            }
            $qs['tr'] = urlencode(config('marcolin.b2b.url') . '/login');
        }

        if (count($qs)) {
            $delim = '?';
            foreach ($qs as $key => $value) {
                $url .= $delim . $key . '=' . $value;
                $delim = '&';
            }
        }

        return $url;
    }

    public function scopeCategory($query, $category) {
        return $query->where('asset_category', $category);
    }

    public function scopePartNumber($query, $partNumber) {
        return $query->where('asset_category', 'part')->where('name', $partNumber)->take(1);
    }

    public function scopeStyleSlug($query, $slug) {
        return $query
            ->where('asset_category', 'product')
            ->whereHas('webStyles', function ($query) use ($slug) {
                return $query->where('slug', $slug);
            })
            ->orderBy('date_created', 'desc')
            ->take(1)
        ;
    }

//    public function scopeContentType($query, $type) {
//        return $query->where('content_type', $type);
//    }

    protected static function boot() {
        parent::boot();

        static::deleted(function ($webAsset) {
            Storage::disk('assets')->delete($webAsset->full_key);
        });
    }
}