<?php

namespace Marcolin\Models;
use Carbon\Carbon;
use DB;

/**
 * Class PromotionLastOrderTrigger
 * @package Marcolin\Models
 */
class PromotionLastOrderTrigger extends PromotionTriggerDetails {
    protected $table = 'PromotionLastOrderTriggers';
    protected $fillable = [
        'limit_to_b2b', 'limit_to_esb',
        'all_time',
        'int_value', 'time_period',
        'commit_sequence'
    ];
    protected $casts = [
        'limit_to_b2b'  => 'boolean',
        'limit_to_esb'  => 'boolean',
        'all_time'      => 'boolean',
    ];
    protected static $timePeriods = [
        'YEAR'      => 'subYears',
        'MONTH'     => 'subMonths',
        'DAY'       => 'subDays',
    ];

    /**
     * Boolean, all promo trigger types need to implement to use their specific logic
     * @param Customer $soldToCustomer
     * @param array $webCartItems
     * @return mixed
     */
    public function isTriggered(Customer $soldToCustomer, $webCartItems = []) {
        if ($this->all_time) {
            $lastOrderedDate = null;
        } else {
            $now = Carbon::now();
            $carbonMethod = self::$timePeriods[$this->time_period];
            $lastOrderedDate = $now->$carbonMethod($this->int_value)->toDateString();
        }

        $result =
            DB::select(
                DB::raw(
                    'SELECT EXISTS (
                              SELECT 1 
                              FROM Customers 
                              WHERE parentCustomerCode = :parentCustomerCode 
                              AND EXISTS (
                                SELECT 1 
                                FROM SalesOrders 
                                WHERE orderOrigin = :orderOrigin 
                                AND orderDate > :orderDate 
                                AND SalesOrders.customer_id = Customers.id
                              )
                        ) AS hasOrdered'
                ),
                [
                    'parentCustomerCode'  => $soldToCustomer->parentCustomerCode,
                    'orderOrigin'   => $this->getOrderOriginToCheck(),
                    'orderDate'     => $lastOrderedDate
                ]
            );

        return count($result) > 0 && $result[0]->hasOrdered == 0;
    }

    private function getOrderOriginToCheck() {
        if ($this->limit_to_b2b && !$this->limit_to_esb) {
            return SalesOrder::ORDER_ORIGIN_B2B;
        } else if (!$this->limit_to_b2b && $this->limit_to_esb) {
            return SalesOrder::ORDER_ORIGIN_ESB;
        } else {
            return null;
        }
    }

    /**
     * promo trigger types should implement if there post operations that should happen after promo is applied
     * @param $soldToCustomer
     * @return mixed
     */
    public function postTriggerOperation(Customer $soldToCustomer)
    {
        // TODO: Implement postTriggerOperation() method.
    }

    /**
     * Boolean, any promo trigger that can use the Almost Qualifying Header should implement this and return 'true'
     * otherwise return 'false'
     * @return bool
     * @deprecated
     */
    public function hasAlmostQualifyingField() {
        return false;
    }

    /**
     * Any promo trigger that can use the Almost Qualifying Header should implement this and
     * @param Customer $soldToCustomer
     * @param $webCartItems
     * @return bool
     * @deprecated
     */
    public function isAlmostQualifying(Customer $soldToCustomer, $webCartItems) {
        return false;
    }
}