<?php

namespace Marcolin\Models;
use DB;

/**
 * Class PromotionCodeTrigger
 * @package Marcolin\Models
 */
class PromotionCodeTrigger extends PromotionTriggerDetails {
    protected $table = 'PromotionCodeTriggers';
    protected $fillable = ['code', 'is_list'];

    // Overrides Model:boot(), used to define a specific 'deleted' state
    public static function boot() {
        parent::boot();
        // make sure all related rows in PromotionCodeTrigger_Code are also deleted
        static::deleted(function (PromotionCodeTrigger $promotionCodeTrigger) {
            $promotionCodeTriggerCodes =
                PromotionCodeTrigger_Code
                    ::where('promotioncodetrigger_id', '=', $promotionCodeTrigger->id)
                    ->get()
            ;

            foreach ($promotionCodeTriggerCodes as $promotionCodeTriggerCode) {
                $promotionCodeTriggerCode->delete();
            }
        });
    }

    public function promotion() {
        return $this->morphToMany(WebPromotion::class, 'trigger',
            'PromotionTriggers', 'trigger_id', 'promotion_id');
    }

    public function codes() {
        return $this->hasMany(PromotionCodeTrigger_Code::class, 'promotioncodetrigger_id', 'id');
    }

    /**
     * Boolean, all promo trigger types need to implement to use their specific logic
     * @param Customer $soldToCustomer
     * @param $webCartItems
     * @return bool
     */
    public function isTriggered(Customer $soldToCustomer, $webCartItems = []) {
        $sessionCouponCode = session()->get('couponCode');
        if ($sessionCouponCode) {
            // if there is a coupon code set in the session,
            // validate it as a real coupon code set for this promo code trigger and that it is not yet redeemed
            return
                $this->is_list &&
                $this->codes()->pluck('coupon_code')->contains($sessionCouponCode) &&
                $this->codes()->where('coupon_code', $sessionCouponCode)->pluck('is_redeemed')->first() !== 1;
        } else {
            // else, check if there is a promo code set in the session
            $sessionPromoCode = session()->get('promoCode');
            if ($sessionPromoCode) {
                // if there is a promo code set in the session, validate it against the code on this code trigger
                return !$this->is_list && $this->code === $sessionPromoCode;
            }
        }
    }

    /**
     * promo trigger types should implement if there post operations that should happen after promo is applied
     * @param $soldToCustomer
     * @return mixed
     */
    public function postTriggerOperation(Customer $soldToCustomer) {
        // TODO: Implement postTriggerOperation() method.
    }

    /**
     * Boolean, any promo trigger that can use the Almost Qualifying Header should implement this and return 'true'
     * otherwise return 'false'
     * @return bool
     */
    public function hasAlmostQualifyingField() {
        return false;
    }

    /**
     * Any promo trigger that can use the Almost Qualifying Header should implement this and
     * @param Customer $soldToCustomer
     * @param $webCartItems
     * @return bool
     */
    public function isAlmostQualifying(Customer $soldToCustomer, $webCartItems) {
        return false;
    }


    // Scopes
    public function scopeByCode($query, $promoCode) {
        return $query->where('code', $promoCode);
    }

    public function scopeByCouponCode($query, $couponCode) {
        return
            $query
                ->where('is_list', DB::raw('1'))
                ->whereExists(function ($query) use ($couponCode) {
                    $query
                        ->select(DB::raw('1'))
                        ->from('PromotionCodeTrigger_Code')
                        ->whereRaw('promotioncodetrigger_id = PromotionCodeTriggers.id')
                        ->where('coupon_code', $couponCode)
                        ->where('deleted_at', DB::raw('0'))
                    ;
                })
            ;
    }
}