<?php

namespace Marcolin\Models;
use AlgoliaSearch\Laravel\AlgoliaEloquentTrait;
use DB;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Input;
use Storage;
use Venturecraft\Revisionable\RevisionableTrait;

/**
 * Class WebStyle
 * @package Marcolin\Models
 */
class WebStyle extends WebModel implements HasLandingPageInterface
{
	protected $table = "WebStyles";
    protected $perPage = 24;

    use AlgoliaEloquentTrait;

    public $algoliaSettings = [
        'searchableAttributes' => [
            'name',
            'style.style'
        ],
        'attributesForFaceting' => [
            'style.brand.brandName',
            'style.gender',
            'style.materialGroupDesc',
            'style.productShapeCode',

            'units.colorDescription',
        ]
    ];

    public function _reindex($safe = true, $setSettings = true, $mergeOldSettings = false, \Closure $onInsert = null)
    {
        /** @var \AlgoliaSearch\Laravel\ModelHelper $modelHelper */
        $modelHelper = App::make('\AlgoliaSearch\Laravel\ModelHelper');

        $indices = $modelHelper->getIndices($this);
        $indicesTmp = $safe ? $modelHelper->getIndicesTmp($this) : $indices;

        if ($setSettings === true) {
            $setToTmpIndices = ($safe === true);
            $this->_setSettings($setToTmpIndices, $mergeOldSettings);
        }

        static::visible()
            ->has('webBrand.webSilos')
            ->has('style.brand')
            ->has('style.units')

            ->with('webBrand.webSilos')
            ->with('style.brand')
            ->with('style.productDescription')
            ->with('style.productShape')
            ->with('style.productRimType')
            ->with('style.productFitting')
            ->with('style.units')

            ->chunk(1000, function ($models) use ($indicesTmp, $modelHelper, $onInsert) {
                /** @var \AlgoliaSearch\Index $index */
                foreach ($indicesTmp as $index) {
                    $records         = [];
                    $recordsAsEntity = [];

                    foreach ($models as $model) {
                        if ($modelHelper->indexOnly($model, $index->indexName)) {
                            $records[] = $model->getAlgoliaRecordDefault($index->indexName);

                            if ($onInsert && is_callable($onInsert)) {
                                $recordsAsEntity[] = $model;
                            }
                        }
                    }

                    $index->addObjects($records);

                    if ($onInsert && is_callable($onInsert)) {
                        call_user_func_array($onInsert, [$recordsAsEntity]);
                    }
                }
            })
        ;

        if ($safe) {
            for ($i = 0; $i < count($indices); $i++) {
                $modelHelper->algolia->moveIndex($indicesTmp[$i]->indexName, $indices[$i]->indexName);
            }

            $this->_setSettings(false); // To a setSettings to set the slave on the master
        }
    }

    public function indexOnly() {
        if (!$this->is_visible) return false;

        return true;
    }

    public function getAlgoliaRecord() {
        DB::enableQueryLog();
        $record = array_only($this->toArray(), [
            'id',
        ]);

        $record['name']         = $this->name;
        $record['slug']         = $this->slug;

        $record['style']        = $this->style;

//        dd(DB::getQueryLog());
        return $record;
    }

    protected $indexSettings = [
        'analysis' => [
            'char_filter' => [
                'replace' => [
                    'type' => 'mapping',
                    'mappings' => [
                        '&=> and '
                    ],
                ],
            ],
            'filter' => [
                'word_delimiter' => [
                    'type' => 'word_delimiter',
                    'split_on_numerics' => false,
                    'split_on_case_change' => false,
                    'generate_word_parts' => true,
                    'generate_number_parts' => true,
                    'catenate_all' => true,
                    'preserve_original' => true,
                    'catenate_numbers' => true,
                ]
            ],
            'analyzer' => [
                'default' => [
                    'type' => 'custom',
                    'char_filter' => [
                        'html_strip',
                        'replace',
                    ],
                    'tokenizer' => 'whitespace',
                    'filter' => [
                        'lowercase',
                        'word_delimiter',
                    ],
                ],
            ],
        ],
    ];

    protected $mappingProperties = [
        'title' => [
            'type' => 'string',
            'analyzer' => 'standard'
        ]
    ];

    public static $elasticFuzziness = 1;

    /**
     * @description return elastic query with fuzziness
     */
    public static function buildElasticQuery($keyWords){
        $params=[
            'multi_match' => [
                'query'     => $keyWords,
                'fuzziness' => self::$elasticFuzziness,
                'fields'    => ['_all']
            ]
        ];
        return $params;
    }

    /**
     * @description index all the entries in an Eloquent model (if needed)
     */
    public static function elasticIndexingIfNeeded(){
        $mappingExists = self::mappingExists();
        if (!$mappingExists) {
            self::createIndex();
            self::putMapping();
            self::addAllToIndex();
        }
    }

	public function getPerPage()
	{
		$tmpPerPage =Input::get('perPage');
		return (isset($tmpPerPage) && $tmpPerPage > 0)?$tmpPerPage:$this->perPage;
	}

	public function style() {
        return $this->belongsTo(Style::class);
    }

	public function webUnits() {
        return $this->hasMany(WebUnit::class, 'webstyle_id', 'id');
	}

    public function webBrand() {
		return $this->belongsTo(WebBrand::class, 'webbrand_id', 'id');
    }

    public function webDetailTabs() {
        return $this->belongsToMany(WebDetailTab::class, 'WebDetailTab_WebStyle', 'webstyle_id', 'webdetailtab_id')->withPivot('is_visible');
    }

    public function favoritedWebUsers() {
	    return $this->belongsToMany(WebUser::class, 'WebStyle_WebUser', 'webstyle_id', 'webuser_id');
    }

    public function isFavorite() {
	    return $this->favoritedWebUsers->where('id', auth()->id())->count() ? true : false;
    }

	public function scopeVisible(Builder $query) {
        $query
            ->with('style')
            ->with('webUnits')
            ->with('webBrand')
            ->where('WebStyles.is_visible', '1')
            ->whereHas('webUnits', function ($query) {
                return $query->visible();
            })
            ->whereHas('webBrand', function ($query) {
                return $query->visible();
            })
        ;

        $activeCustomer = b2b()->activeCustomer();
        if ($activeCustomer) {
            $query->whereHas('style', function ($query)  use ($activeCustomer) {
                return $query->where('Styles.salesOrg', '=', $activeCustomer->salesOrg);
            });
        } else {
            $query->whereHas('style', function ($query) {
                return $query->where('Styles.salesOrg', '=', 'UN01');
            });
        }

        return $query;
    }

    public function scopeDisplayOrder(Builder $query) {
        return $query->orderBy('pivot_display_order', 'asc')->orderBy('name', 'asc');
    }

    public function scopeAlphabetical(Builder $query) {
        return $query->orderBy('name', 'asc');
    }

	public function scopeHasAttributes($query, $filters) {
		if(!isset($filters) || is_null($filters)) return $query;
		$attrs = [];
		foreach ($filters as $filter){
			if(!isset($attrs[$filter->webattribute_id])){
				$attrs[$filter->webattribute_id] = [];
			}
			$attrs[$filter->webattribute_id] = array_merge($attrs[$filter->webattribute_id],$filter->webAttributeFilterValues->pluck('filter_value')->toArray());
		}
		return $query->where(function ($query) use ($attrs) {
            foreach ($attrs as $id=>$values) {
//                $query->whereHas('webAttributes', function ($query) use ($id, $values) {
//                    return $query->where('webattribute_id', $id);
//                });
            }
			return $query->whereHas('webUnits', function ($query) use ($attrs) {
				foreach ($attrs as $id=>$values) {
//					$query = $query->whereHas('webAttributes', function ($query) use ($id, $values) {
//						return $query->where('webattribute_id', $id)->whereIn('attribute_value', $values);
//					});
				}
				return $query;
			});
		});
	}
	public function scopeManagedBy($query, $product_manager) {
		return $query->whereHas('webUnits', function($query) use ($product_manager) {
			return $query->managedBy($product_manager);
		});
	}

	public function scopeInBrand($query, $webBrand) {
		$webBrandId = ($webBrand instanceof WebBrand) ? $webBrand->id : $webBrand;
		return $query->where('webBrand_id', '=', $webBrandId);
	}

    public function scopeInSilo($query, $webSilo) {
        $webSiloId = ($webSilo instanceof WebSilo) ? $webSilo->id : $webSilo;
        return $query->whereHas('webBrand', function ($query) use ($webSiloId) {
            return $query->inSilo($webSiloId);
        });
    }

	public static function in($model) {
		if ($model instanceof WebBrand) {return self::inGroup($model);}
		else if ($model instanceof WebCategory) return self::inCategory($model);
        else if ($model instanceof WebBrand) return self::inCollection($model);
        else if ($model instanceof WebSilo) return self::inSilo($model);
        else return self::newQuery();
	}

	public function scopeForDisplay(Builder $query, $slug) {
		return $query
			->visible()
            ->with(['webUnits' => function ($query) {
                $query->with(['assets', 'unit' => function ($query) {
//                    $query->with('customerPrices', 'inventoryItems');
                }]);
            }, 'assets'])
//            ->with('webAttributes')
            ->with(['favoritedWebUsers' => function ($query) {
                $query->where('WebUsers.id', auth()->id());
            }])
			->slug($slug)
		;
	}

	public function scopeForBreadcrumbs($query) {
        $query->with([
            'webBrand' => function ($lQuery) {
                if (session('breadcrumb.collectionId')) {
                    $lQuery->where('webBrand.id', '=', session('breadcrumb.collectionId'));
                }
                $lQuery->with(['webCategories' => function ($cQuery) {
                    if (session('breadcrumb.categoryId')) {
                        $cQuery->where('WebCategories.id', '=', session('breadcrumb.categoryId'));
                    }
                    $cQuery->with(['webBrand' => function ($gQuery) {
                        if (session('breadcrumb.groupId')) {
                            $gQuery->where('webBrand.id', '=', session('breadcrumb.groupId'));
                        }
                    }]);
                }]);
            }
        ]);
        return $query;
    }

    public function scopeWebVisible($query) {
	    return $query
            ->where('is_visible', '1')
            ->whereHas('style', function ($query) {
                return $query->webVisible();
            })
            ->has('webUnits')
        ;
    }

	public function scopeForLandingPage($query, HasLandingPageInterface $lpModel) {
        $query
            ->webVisible()
            ->notForSearchOnly()
            ->with('style.productDescription')
            ->with(['style' => function ($query) {
                $query
                    ->with('productDescription')
                    ->with('productShape')
                    ->with('productFitting')
                    ->with('productRimType')
                    ->with('units.priceListValues')
                ;

                $activeCustomer = b2b()->activeCustomer();
                if ($activeCustomer) {
                    $query->with(['orderHistories' => function ($query) use ($activeCustomer) {
                        return $query->forCustomer($activeCustomer)->orderBy('billingDate', 'DESC');
                    }]);
                }

                return $query;
            }])
            ->with('assets')
            ->with(['webUnits' => function ($unitsQuery) {
                return $unitsQuery->webVisible()->with('assets')->with('unit.priceListValues');
            }])
            ->with(['favoritedWebUsers' => function ($query) {
                return $query->where('WebUsers.id', auth()->id());
            }])
            ->whereHas('webUnits', function ($unitsQuery) {
                return $unitsQuery
                    ->webVisible()
//                    ->has('assets')
                    ->has('unit')
                ;
            })
            ->join('Styles', 'Styles.id', '=', 'WebStyles.style_id')
            ->leftJoin('ProductDescriptions', 'ProductDescriptions.style', '=', 'Styles.style')
            ->orderBy(DB::raw('COALESCE(IF(LENGTH(WebStyles.name) > 0, WebStyles.name, NULL), IF(LENGTH(ProductDescriptions.styleName) > 0, ProductDescriptions.styleName, NULL), IF(LENGTH(Styles.style) > 0, Styles.style, NULL))'), 'asc')
            ->groupBy('WebStyles.id')
            ->select(DB::raw('WebStyles.*'))
        ;

        if ($lpModel instanceof WebBrand) {
            $query->where('webbrand_id', $lpModel->id);
        }
        return $query;
    }

	public static function getImageUrl($partNumber, $width = null, $height = null) {
		$url = config('marcolin.images.url');
		if(!isset($url)) $url = config('marcolin-b2b.images.url');
		$url .= '/' . $partNumber;
		if ($width) $url .= "/$width";
		if ($height) $url .= "/$height";
		return $url;
	}

	public function scopeJoinWebUnit(Builder $query){
		return $query->leftJoin('WebUnits', 'WebUnits.webfamily_id', '=', 'WebStyles.id');
	}

    public function slugUrl() {
        return route('style.slug', [$this->slug]);
    }

    public function scopeForSearchOnly(Builder $query){
        return $query->where('search_only',1);
    }

    public function scopeNotForSearchOnly(Builder $query){
        return $query->where('search_only',0);
    }

    public function scopeBestSellers(Builder $query) {
        return $query->where('best_seller', 1);
    }

    public function getNameAttribute($name) {
        if (isset($name) && strlen($name) > 0) {
            return $name;
        }

        $name = $this->style->name;
        if (isset($name) && strlen($name) > 0) {
            return $name;
        }

        return $this->slug;
    }

    public function getPrimaryImage() {
        $asset = $this->assetsByType('primary')->first();
        if (!is_null($asset)) return $asset;

        $webUnit = $this->webUnits->where('is_family_image', 1)->first();
        if (!is_null($webUnit)) {
            $asset = $webUnit->getPrimaryImage();
            if (!is_null($asset)) return $asset;
        }

        foreach ($this->webUnits as $webUnit) {
            if (!is_null($webUnit)) {
                $asset = $webUnit->getPrimaryImage();
                if (!is_null($asset)) {
                    return $asset;
                }
            }
        }

        return null;
    }

	use HasAssets, HasSlug, RevisionableTrait, HasLandingPage, ManyToManyRevisionableTrait;

    public function getParentRelationName()
    {
        return 'webBrand';
    }

    public function getLabelStyleAttribute() {
        return array_get(self::$labelStyles, $this->label);
    }

    public function getCustomerPriceAttribute() {
        $soldToCustomer = b2b()->activeCustomer();
        return $this->style->getCustomerPrice($soldToCustomer);
    }

    public function getPriceRangeAttribute() {
        if (is_null($this->webUnits)) return null;

        $activeCustomer = b2b()->activeCustomer();

        $min = PHP_INT_MAX;
        $max = 0;
        foreach ($this->webUnits as $webUnit) {
            $price = $webUnit->getCustomerPrice($activeCustomer);
            $min = min($min, $price);
            $max = max($max, $price);
        }
        return [$min, $max];
    }

    public function getSizesAttribute() {
        $sizes = [];
        foreach ($this->webUnits as $webUnit) {
            $size = array_get($webUnit, 'unit.size', false);
            if ($size) $sizes[$size] = $size;
        }
        sort($sizes);
        return $sizes;
    }

    public function getStyleDescriptionAttribute() {
        $productDescriptions = array_get($this, 'style.productDescription', false);
        if (!$productDescriptions) return '';
        $productDescription = $productDescriptions->first();

        return array_get($productDescription, 'styleDesc');
    }

    public function getProductDescriptionAttribute() {
        $productDescriptions = array_get($this, 'style.productDescription', false);
        if (!$productDescriptions) return '';
        $productDescription = $productDescriptions->first();

        return array_get($productDescription, 'productDesc');
    }

    public function getShapeAttribute() {
        return array_get($this, 'style.productShape.shape', '');
    }

    public function getFittingAttribute() {
        return array_get($this, 'style.productFitting.fitting', '');
    }

    public function getRimTypeAttribute() {
        return array_get($this, 'style.productRimType.rimType', '');
    }
}
