<?php
namespace Marcolin\Models;

use Auth;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Config;
use Phaza\SingleTableInheritance\SingleTableInheritanceTrait;
use Illuminate\Database\Eloquent\SoftDeletes;
use Marcolin\Models\Traits\UploadableTrait;
use Marcolin\Models\Traits\SoftDeleteCascadeTrait;
use Cviebrock\EloquentSluggable\Sluggable;
use Marcolin\Models\Traits\RestrictableTrait;

class Category extends WebModel{
    use SingleTableInheritanceTrait;
    use UploadableTrait;
    use SoftDeleteCascadeTrait;
    use Sluggable;
    use RestrictableTrait;
    use HasCommitSequence;

    protected $table = 'CMSCategories';
    protected static $singleTableTypeField = 'type';
    protected static $singleTableSubclasses = ['Marcolin\Models\Home','Marcolin\Models\Department', 'Marcolin\Models\Section'];
    protected static $softDeleteCascadeRelationships = ['content','children'];

    protected $fillable = ['name','description', 'type', 'parent','parent_cmsCategory_id','key'];

    public function sluggable(){
        return [
            'key' => [
                'source' => 'name'
            ]
        ];
    }

    protected function usesSoftDeleting() {
       return true;
    }

    public function children()
    {
        return $this->hasMany('Marcolin\Models\Category','parent_cmsCategory_id');
    }

    public function childrenHaveContent() {
        foreach ($this->allChildrenWithContent() as $child) {
            if (!$child->allContentFlat()->isEmpty()){
                return true;
            }
        }
        return false;
    }

    public function parent()
    {
        return $this->belongsTo('Marcolin\Models\Category', 'parent_cmsCategory_id');
    }

    public function content()
    {
        $salesRep = Auth::user()->associatedSalesRep();
        if ($salesRep){
            $brandNames = $salesRep->brandNames();
            return $this->belongsToMany('Marcolin\Models\Content', 'CMSCategory_CMSContent', 'cmsCategory_id', 'cmsContent_id')->where(function (Builder $where) use ($brandNames){
                $where->doesntHave('tags')->orWhereHas('tags', function (Builder $where) use ($brandNames) {
                    $where->where('group', '=', 'other')
                        ->orWhere(function (Builder $where) use ($brandNames) {
                            $where->where('group', '=', 'collection')
                                ->whereIn('name', $brandNames);
                        });
                });
            });
        } else{
            return $this->belongsToMany('Marcolin\Models\Content', 'CMSCategory_CMSContent', 'cmsCategory_id', 'cmsContent_id');
        }

    }

//    public function contentInBrands($brandNames){
//        return $this->content()->where(function (Builder $where) use ($brandNames){
//            $where->doesntHave('tags')->orWhereHas('tags', function (Builder $where) use ($brandNames) {
//                $where->where('group', '=', 'other')
//                    ->orWhere(function (Builder $where) use ($brandNames) {
//                        $where->where('group', '=', 'collection')
//                            ->whereIn('name', $brandNames);
//                    });
//            });
//        })->get();
////
////        /** @var \Illuminate\Database\Query\Builder $builder */
////        $builder = $this->content()->where(function (Builder $where) use ($brandNames){
////            $where->doesntHave('tags')->orWhereHas('tags', function (Builder $where) use ($brandNames) {
////                $where->where('group', '=', 'other')
////                    ->orWhere(function (Builder $where) use ($brandNames) {
////                        $where->where('group', '=', 'collection')
////                            ->whereIn('name', $brandNames);
////                    });
////            });
////        });//->get();
////
////        $query = $builder->getQuery();
////        $sql = $query->toSql();
////        $bindings = $query->getBindings();
////
////        return $builder->get();
//    }

    public function subscribers() {
        return $this->belongsToMany('Marcolin\Models\WebUser', 'WebUser_CMSCategory_Subscription', 'cmsCategory_id', 'webUser_id');
    }

    public function selfAndDescendants() {
        $result = [$this->id => $this];
        foreach ($this->allChildren()->get() as $child){
            $result[$child->id]=$child;
        }
        return $result;
    }

    public function collectionTags() {

        $ids = array_keys($this->selfAndDescendants());
        return Tag::collections()->with(['content' => function($q) use ($ids) {
            $q->join('CMSCategory_CMSContent', 'CMSContent.id','=','cmsContent_id')
              ->whereIn('cmsCategory_id', $ids);
        }])->get();
    }

    public function restrictions() {
        return $this->morphToMany('Marcolin\Models\WebRole', 'restrictable', 'CMSRestrictables', 'cmsRestrictable_id', 'webRole_id');
    }

    public function hasRestriction($roleID) {
        foreach ($this->restrictions as $role) {
            if ($role->id == $roleID) return true;
        }
        return false;
    }

    public function childTypes($assoc = false)
    {
        $types = DB::table('Allowed_Child_Type')->where('type','=',$this->type)->lists('child_type');
        if($assoc) {
            $assoc = [];
            foreach($types as $type) {
                $assoc[$type] = $type;
            }
            return $assoc;
        } else {
            return $types;
        }
    }


    public function categoryLinkText(){
        if (!isset($this->parent) || $this->type === 'department') {
            $linkText = $this->name;
        }else{
            $linkText = $this->parent->categoryLinkText() . ' -> ' . $this->name;
        }
        return $linkText;
    }

    public function department()
    {
        if (!isset($this->parent) || $this->type === 'department') {
            return $this;
        } else {
            return $this->parent->department();
        }
    }

    public function getThumbPath() {
        return $this->getImagePathOfType('thumb');
    }

    public function getBannerPath(){
        return $this->getImagePathOfType('banner');
    }
    public function getImagePathOfType($type){
        if($this->hasUploadType($type)) {
            return $this->firstOfUploadType($type)->getContentsPath();
        } else {
            return Config::get('imagedefaults.category_'.$type);
        }
    }

    public function recentlyPublished($limit)
    {
       return $this -> content -> sortByDesc('date_created')-> take($limit);
    }

    public function allParents() {
        return $this->parent()->with(['allParents']);
    }

    public function allChildren() {
        return $this->children()->with(['allChildren']);
    }

    public function allChildrenWithContent() {
        return $this->children()->with(['content','allChildrenWithContent']);
    }

    public static function allChildrenFlat($children) {
        $collection = new Collection();
        foreach ($children as $child) {
            $collection->add($child);
            $collection = $collection->merge(self::allChildrenFlat($child->allChildrenWithContent));
        }
        return $collection;
    }

    public function allContentFlat($sortKey = 'date_created', $desc = false) {
//        $content = $this->content;
//        foreach($this->allChildrenFlat($this->allChildrenWithContent) as $children) {
//            $content = $content->merge($children->content);
//        }
//        return $content;
        $content = Category::allFlatContentHelper(new Collection([$this]))->sortBy($sortKey, $desc);
        //fix indices
        $return = new Collection();
        foreach($content as $item) {
            $return->add($item);
        }
        return $return;
    }

    public function allFlat($separator = ' -> ') {
        $objs = $this->allChildren()->get();
        return Category::allFlatHelper($objs, $separator);
    }

    private static function allFlatHelper($objs, $separator, $prefix = false) {
        $output = array();
        foreach ($objs as $obj) {
            $string = ($prefix ? "$prefix$separator" : '') . $obj->name;
            $output[$obj->id] = $string;
            $output += Category::allFlatHelper($obj->allChildren, $separator, $string);
        }
        return $output;

    }

    public function nameFlat($separator = ' -> ') {
        $string = $this->name;
        if (!$this->parent()->get()->isEmpty() && $this->parent()->first()->type != 'home') $string = $this->parent()->first()->nameFlat($separator) . $separator . $string;
        return $string;
    }

    private static function allFlatContentHelper(Collection $objs) {
        $output = new Collection();
        foreach($objs as $obj) {
            $output = $output -> merge ($obj -> content);
            $output = $output -> merge (Category::allFlatContentHelper($obj -> allChildren));
        }
        return $output;
    }

    public function controller() {
        return new CategoryController();
    }


}


