<?php

namespace Marcolin\Models;

use Auth;
use Debugbar;
use Illuminate\Database\Eloquent\Builder;
use Venturecraft\Revisionable\RevisionableTrait;

class WebSilo extends WebModel implements HasLandingPageInterface {
    protected $table = "WebSilos";
    protected $fillable = [
        'name',
        'slug',
        'weblandingpage_id',
        'customerCodeHierarchy_id',
        'customer_source',
        'is_default',
        'is_visible',
        'default_price_list',
        'contact_block',
        'free_shipping_on_daily_orders',
        'free_shipping_on_three_piece_orders',
        'has_rebates',
        'required_rebate_return_percent',
        'allow_credit_cards',
        'catalogs_weblink_id',
        'shipping_note',
        'webRoleIds',
        'commit_sequence'
    ];
    protected $casts = [
        'is_visible' => 'boolean'
    ];

    use HasCommitSequence;

    public function webBrands()
    {
        return $this->belongsToMany(WebBrand::class, 'WebBrand_WebSilo', 'websilo_id', 'webbrand_id')
            ->withPivot('sort_name')
            ->with('brand')
            ->orderBy('sort_name');
    }

    public function visibleWebBrands()
    {
        return $this
            ->webBrands()
            ->visible()
            ->forDisplay();
    }

    public function contactUs()
    {
        return $this->belongsTo(WebContactUs::class, 'contactus_id', 'id');
    }

    public function webLinkCategories()
    {
        return $this->belongsToMany(WebCategory::class, 'WebLinkCategory_WebSilo', 'websilo_id', 'weblinkcategory_id');
    }

    public function webLinks()
    {
        return $this->belongsToMany(WebCategory::class, 'WebLink_WebSilo', 'websilo_id', 'weblink_id');
    }

    public function webUrls()
    {
        return $this->hasMany(WebUrl::class, 'websilo_id', 'id');
    }

    public function customers() {
        return $this->belongsToMany(Customer::class, 'Customer_WebSilo', 'websilo_id', 'customer_id')
            ;
    }

    public function buyingGroups() {
        return $this
            ->belongsToMany(CustomerCodeHierarchy::class, 'WebSilo_BuyingGroup', 'websilo_id', 'customerCodeHierarchy_id')
            ;
    }

    public function webContactBlocks() {
        return $this->hasMany(WebContactBlock::class, 'websilo_id', 'id');
    }

    public function rebateLevels()
    {
        return $this->hasMany(RebateLevel::class, 'websilo_id', 'id');
    }

    public function getContactBlockAttribute() {
        $webSiloId = b2b()->activeWebSilo()->id;
        $language = 'en'; //b2b()->activeLanguage();

        $blocks = $this
            ->webContactBlocks
            ->filter(function ($webContactBlock) use ($webSiloId) {
                return array_get($webContactBlock, 'websilo_id', $webSiloId);
            })
            ->keyBy('language')
        ;

        return array_get($blocks, "$language.contact_block", array_get($blocks, "en.contact_block"));
    }

    public function webLandingPage() {
        return $this->belongsTo(WebLandingPage::class, 'weblandingpage_id', 'id');
    }

    public function webLinkWebSiloJoins() {
        return $this->hasMany(WebLink_WebSilo::class, 'websilo_id', 'id');
    }

    public function getFeaturedChildrenAttribute()
    {
        return $this->visibleWebBrands()->wherePivot('is_featured', 1)->get();
    }

    public function getChildrenAttribute()
    {
        return $this->visibleWebBrands()->get();
    }

    public function divisionRules()
    {
        return $this->belongsToMany(Division::class, 'WebSilo_Division', 'websilo_id', 'division_id');
    }

    public function customerCategorySummaryRules()
    {
        return $this->belongsToMany(CustomerCategorySummary::class, 'WebSilo_CustomerCategorySummary', 'websilo_id', 'customercategorysummary_id');
    }

    public function webCustomerTypeRules()
    {
        return $this->belongsToMany(WebCustomerType::class, 'WebSilo_WebCustomerType', 'websilo_id', 'webcustomertype_id');
    }

    public function getDivisionsAttribute()
    {
        return $this->divisionRules->pluck('id')->toArray();
    }

    public function setDivisionsAttribute($ids)
    {
        $this->divisionRules()->sync($ids);
    }

    public function getCustomerCategorySummariesAttribute()
    {
        return $this->customerCategorySummaryRules->pluck('id')->toArray();
    }

    public function setCustomerCategorySummariesAttribute($ids)
    {
        $this->customerCategorySummaryRules()->sync($ids);
    }

    public function getWebCustomerTypesAttribute()
    {
        return $this->webCustomerTypeRules->pluck('id')->toArray();
    }

    public function setWebCustomerTypesAttribute($ids)
    {
        $this->webCustomerTypeRules()->sync($ids);
    }

    public function getUrlsAttribute()
    {
        return $this->webUrls->pluck('id')->toArray();
    }

    public function setUrlsAttribute($ids)
    {
        $this->webUrls()->sync($ids);
    }

    public function scopeVisible(Builder $query)
    {
        return $query->where('WebSilos.is_visible', '1');
    }

    public function scopeCurrent(Builder $query)
    {
        $query->where('id', b2b()->activeWebSilo()->id);
    }

    public function scopeDefaultWebSilo (Builder $query) {
        return $query->where('is_default', '1');
    }

    public function getPriceListAttribute()
    {
        return 'Catalog';
    }

    public function slugUrl()
    {
        return route('silo.slug', [$this->slug]);
    }

    use HasAssets, HasSlug, HasLandingPage, RevisionableTrait;

    public function getParentRelationName()
    {
        return false;
    }
}
