<?php

namespace Marcolin\Models;

use Auth;
use DateInterval;
use DateTime;
use Illuminate\Database\Eloquent\Builder;
use Image as Image2;
use Storage;

/**
 * Marcolin\Models\Product
 *
 * @property mixed $id
 * @property string $date_created
 * @property string $date_modified
 * @property string $deleted_at
 * @method static \Illuminate\Database\Query\Builder|\Marcolin\Models\Unit whereId($value)
 */
class Unit extends WebModel
{
    protected $table = 'Units';

    const IN_STOCK = 'inStock';
    const OUT_OF_STOCK = 'outOfStock';
    const NOT_AVAILABLE = 'notAvailable';
    const UNKNOWN = 'unknown';

    public function styleRelation()
    {
        return $this->belongsTo(Style::class, 'style_id', 'id');
    }

    function webStyle()
    {
        return $this->hasOne(WebStyle::class, 'style_id', 'style_id');
    }

    public function webUnit()
    {
        return $this->hasOne(WebUnit::class);
    }

    public function productLensType()
    {
        return $this->belongsTo(ProductLensType::class, 'productLensType_id', 'id');
    }

    public function productColorFamily()
    {
        return $this->belongsTo(ProductColorFamily::class, 'productColorFamily_id', 'id');
    }

    public function productSalesStatus()
    {
        return $this->belongsTo(ProductSalesStatus::class, 'productSalesStatus_id', 'id');
    }

    public function productFrontMaterial()
    {
        return $this->belongsTo(ProductFrontMaterial::class, 'productFrontMaterial_id', 'id');
    }

    public function productTempleMaterial()
    {
        return $this->belongsTo(ProductTempleMaterial::class, 'productTempleMaterial_id', 'id');
    }

    public function getColorInfoAttribute()
    {
        return $this->color . ' - ' . $this->colorDescription;
    }

    public function getColorDescriptionAttribute()
    {
        return title_case($this->attributes['colorDescription']);
    }

    public function priceListValues()
    {
        return $this->hasMany(PriceListValue::class);
    }

    public function stockLevels()
    {
        return $this->belongsToMany(StockLevel::class, 'StockLevel_Unit', 'unit_id', 'stocklevel_id');
    }

    public function getCustomerPrice($customer)
    {
        $priceListValue = $this
            ->priceListValues
            ->where('priceList_id', $customer->priceList_id)
            ->where('deleted_at', '0000-00-00 00:00:00')
            ->first();

        if (isset($priceListValue)) {
            $price = $priceListValue->materialOpticalPrice;

            $brandID = $this->styleRelation->brand_id;
            $customerDiscount = $customer->customerDiscounts->where('brand_id', '=', $brandID)->first();
            if (isset($customerDiscount)) {
                $price = $price - ($price * ($customerDiscount->discountPercent / 100));
            }

            return number_format($price / 100, 2);
        } else {
            return 0;
        }
    }

    public function getCustomerDiscount($customer)
    {
//        $customerDiscount = $customer->customerDiscounts()->where('brand_id', '=', $this->styleRelation->brand_id)->first();
//        return $customerDiscount->discountPercent / 100;
        return 0;
    }

    public function getPriceByPriceListAndCurrency($priceList_id)
    {
        $priceListValue = $this
            ->priceListValues
            ->where('priceList_id', $priceList_id)
            ->where('deleted_at', '0000-00-00 00:00:00')
            ->first();

        if (isset($priceListValue)) {
            return $priceListValue->materialOpticalPrice / 100;
        } else {
            return 0;
        }
    }

    public function getFormattedColorAttribute()
    {
        return $this->color . ' - ' . $this->colorDescription;
    }

    public function getFormattedSizeAttribute()
    {
        return $this->size . '/' . $this->bridge . '-' . $this->templeLength;
    }

    public function primaryImage($width = false, $height = false)
    {
        return $this->webUnit->primaryImage($width, $height);
    }

    public function getIsWebVisibleAttribute()
    {
        return static::webVisible()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleAllAttribute()
    {
        return static::webVisibleAll()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleAutocatAttribute()
    {
        return static::webVisible()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleStandardAttribute()
    {
        return static::webVisibleStandard()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleHarleyDavidsonAttribute()
    {
        return static::webVisibleHarleyDavidson()->where('id', '=', $this->id)->count() > 0;
    }

    public function scopeWebVisible(Builder $query)
    {
        $productVisibilityOverride = config('marcolin.product_visibility_override');
        if (!isset($productVisibilityOverride)) {
            $customer = b2b()->activeCustomer();
            if ($customer !== null && $customer !== false) {
                $productVisibilityOverride = $customer->productVisibility;
            } else {
                $productVisibilityOverride = "STANDARD";
            }
        }

        switch (strtoupper($productVisibilityOverride)) {
            case "AUTOCAT":
                return $query->webVisibleAutocat();
            case "HARLEY":
                return $query->webVisibleHarleyDavidson();
            case "TOM_FORD_LIMITED":
                return $query->webVisibleTomFordLimited();
            case "STANDARD":
            default:
                return $query->webVisibleStandard();
        }
    }

    public function scopeWebVisibleAll(Builder $query) {
        $query
            ->where(function ($query) {
                return $query->webVisibleHarleyDavidson();
            })
            ->orWhere(function ($query) {
                return $query->webVisibleStandard();
            })
            ->orWhere(function ($query) {
                return $query->webVisibleAutocat();
            })
            ->orWhere(function ($query) {
                return $query->webVisibleTomFordLimited();
            });
        ;
        return $query;
    }

    public function scopeWebVisibleAutocat(Builder $query) {
        $query
            ->where(function ($query) {
                return $query
                    ->whereRaw("`Units`.`endDate` >= DATE_SUB(NOW(), INTERVAL 2 YEAR)")
                    ->orWhereRaw("`Units`.`releaseDate` >= DATE(NOW())")
                    ->orWhereRaw("`Units`.`salesStatusCode` = '04'")
                    ->orWhereRaw("`Units`.`salesStatusCode` = '01'")
                    ;
            })
            ->orWhere(function ($query) {
                return $query->webVisibleHarleyDavidson();
            })
            ->orWhere(function ($query) {
                return $query->webVisibleStandard();
            });
        return $query;
    }

    public function scopeWebVisibleTomFordLimited(Builder $query) {
        $query
            ->where(function ($query) {
                return $query
                    ->where(function ($query) {
                        return $query
                            ->whereRaw("`Units`.`releaseDate` < DATE_SUB(NOW(), INTERVAL 90 DAY)")
                            ->whereHas("styleRelation", function (Builder $where) {
                                $where
                                    ->whereRaw("`Styles`.`brandCode` = 'FT'");
                            })
                            ;
                    })
                    ->orWhere(function ($query) {
                        return $query
                            ->whereHas("styleRelation", function (Builder $where) {
                                $where
                                    ->whereRaw("`Styles`.`brandCode` != 'FT'");
                            })
                            ;
                    })
                    ;
            })
            ->Where(function ($query) {
                return $query->webVisibleStandard();
            });
        return $query;
    }

    public function scopeWebVisibleStandard(Builder $query)
    {
        $query
            ->whereHas("webUnit", function (Builder $where) {
                $where
                    ->whereRaw("`WebUnits`.`is_visible` = 1");
            })
            ->whereHas("styleRelation", function (Builder $where) {
                $where
                    ->whereHas("productType", function (Builder $where) {
                        $where
                            ->whereRaw("`ProductTypes`.`productType` IN ('S', 'V', 'C')");
                    });
            })
            ->whereHas("styleRelation.webStyle", function (Builder $where) {
                $where
                    ->whereRaw("`WebStyles`.`is_visible` = 1");
            })
            ->has("styleRelation.brand")
            ->whereHas("styleRelation.brand.webBrand", function (Builder $where) {
                $where
                    ->whereRaw("`WebBrands`.`is_visible` = 1");
            })
            ->whereRaw("`Units`.`commercialDate` != '0000-00-00'")
            ->where(function (Builder $where) {
                $where
                    ->where(function (Builder $where) {
                        $where
                            ->whereHas("productSalesStatus", function (Builder $where) {
                                $where
                                    ->whereRaw("`ProductSalesStatuses`.`salesStatusCode` = ''");
                            });
                    })
                    ->orWhere(function (Builder $where) {
                        $where
                            ->whereHas("productSalesStatus", function (Builder $where) {
                                $where
                                    ->whereRaw("`ProductSalesStatuses`.`salesStatusCode` = '03'");
                            })
                            ->whereRaw("`Units`.`endDate` >= DATE_SUB(NOW(), INTERVAL 6 MONTH)")
                            ->whereHas('stockLevels', function (Builder $where) {
                                $where
                                    ->whereRaw("`StockLevels`.`availableQuantity` >= 10");
                            })
                        ;
                    });
            });
        return $query;
    }

    public function scopeWebVisibleHarleyDavidson(Builder $query)
    {
        $query
            ->whereHas("webUnit", function (Builder $where) {
                $where
                    ->whereRaw("`WebUnits`.`is_visible` = 1");
            })
            ->whereHas("styleRelation", function (Builder $where) {
                $where
                    ->whereHas("productType", function (Builder $where) {
                        $where
                            ->whereRaw("`ProductTypes`.`productType` IN ('S', 'V', 'C')");
                    });
            })
            ->whereHas("styleRelation.webStyle", function (Builder $where) {
                $where
                    ->whereRaw("`WebStyles`.`is_visible` = 1");
            })
            ->has("styleRelation.brand")
            ->whereHas("styleRelation.brand.webBrand", function (Builder $where) {
                $where
                    ->whereRaw("`WebBrands`.`is_visible` = 1");
            })
            ->whereRaw("`Units`.`retailCommercialDate` != '0000-00-00'")
            ->whereRaw("`Units`.`retailCommercialDate` <= NOW()")
            ->where(function (Builder $where) {
                $where
                    ->where(function (Builder $where) {
                        $where
                            ->whereHas("productSalesStatus", function (Builder $where) {
                                $where
                                    ->whereRaw("`ProductSalesStatuses`.`salesStatusCode` = ''");
                            });
                    });
            });
        return $query;
    }

    public function scopeWebVisibleKeyAccounts(Builder $query)
    {
        return $query;
    }

    public function scopeMissingImages(Builder $query)
    {
        $query
            ->doesntHave('webUnit.assets');
    }

    public function getInStockLabelAttribute()
    {
        return config('marcolin.inStockLabels.0', 'In Stock');
    }

    public function getOutOfStockLabelAttribute()
    {
        return config('marcolin.outOfStockLabels.0', 'Out of stock');
    }


    public function getStockStatusAttribute()
    {
        $stock = $this->stockLevels->first();
        if (isset($stock)) {
            if ($stock->eta == '0000-00-00') {
                return Unit::IN_STOCK;
            } else {
                return Unit::OUT_OF_STOCK;
            }
        } else {
            return Unit::NOT_AVAILABLE;
        }
    }

    public function getStockLabelAttribute()
    {
        $status = $this->stock_status;
        switch ($status) {
            case Unit::IN_STOCK:
                return $this->in_stock_label;
            case Unit::OUT_OF_STOCK:
                $stock = $this->stockLevels->first();
                $label = $this->out_of_stock_label;
                if (isset($stock)) {
                    if ($stock->eta != '9999-12-31') {
                        $label .= " (" . $stock->adjusted_eta . ')';
                    } else {
                        if ($this->riskFactor == 1) {
                            $date = new DateTime();
                            $date->add(new DateInterval('P120D'));
                            $label .= " (" . $date->format('m-d-Y') . ')';
                        }
                    }
                }
                return $label;
            case Unit::NOT_AVAILABLE:
                return 'Not Available';
        }
        return '';
    }
}

