<?php

namespace Marcolin\Models;

/**
 * Class PromotionOrderQuantityTrigger
 * @package Marcolin\Models
 */
class PromotionOrderQuantityTrigger extends PromotionTriggerDetails {
    protected $table = 'PromotionOrderQuantityTriggers';
    protected $fillable = [
        'minimum_quantity',
        'maximum_quantity',
        'target_all_products'
    ];
    protected $casts = ['target_all_products' => 'boolean'];

    // Overrides Model::boot(), used to define a specific 'deleted' state
    public static function boot() {
        parent::boot();
        // make sure all related rows in PromotionOrderQuantityTrigger_Product are also deleted
        static::deleted(function (PromotionOrderQuantityTrigger $promotionOrderQuantityTrigger) {
            $promotionOrderQuantityTriggerProduct =
                PromotionOrderQuantityTrigger_Product
                    ::where('promotionorderquantitytrigger_id', '=', $promotionOrderQuantityTrigger->id)
                    ->get();

            foreach ($promotionOrderQuantityTriggerProduct as $promotionOrderQuantityTriggerProduct) {
                $promotionOrderQuantityTriggerProduct->delete();
            }
        });
    }

    /**
     * Boolean, all promo trigger types need to implement to use their specific logic
     * @param Customer $soldToCustomer
     * @param $webCartItems
     * @return mixed
     */
    public function isTriggered(Customer $soldToCustomer, $webCartItems = []) {
        $totalQualifyingCartItems = $this->getCurrentQualifyingCartTotal($webCartItems);

        return
            $totalQualifyingCartItems >= $this->minimum_quantity
            &&
            (
                $this->maximum_quantity == 0
                ||
                (
                    $totalQualifyingCartItems <= $this->maximum_quantity
                    &&
                    count($webCartItems) <= $this->maximum_quantity
                )
            )
            ;
    }

    /**
     * promo trigger types should implement if there post operations that should happen after promo is applied
     * @param $soldToCustomer
     * @return mixed
     */
    public function postTriggerOperation(Customer $soldToCustomer) {
        // TODO: Implement postTriggerOperation() method.
    }

    /**
     * Boolean, any promo trigger that can use the Almost Qualifying Header should implement this and return 'true'
     * otherwise return 'false'
     * @return bool
     */
    public function hasAlmostQualifyingField() {
        return false;
    }

    /**
     * Any promo trigger that can use the Almost Qualifying Header should implement this and
     * @param Customer $soldToCustomer
     * @param $webCartItems
     * @return mixed
     */
    public function isAlmostQualifying(Customer $soldToCustomer, $webCartItems) {
        return false;
    }

    /**
     * Get all WebBrands specifically targeted by this trigger
     * @return mixed
     */
    public function brands() {
        return $this
            ->morphedByMany(Brand::class, 'product',
                'PromotionOrderQuantityTrigger_Product', 'promotionorderquantitytrigger_id')
            ->where('PromotionOrderQuantityTrigger_Product.deleted_at', '=', '0000-00-00 00:00:00')
            ;
    }

    /**
     * Get all WebStyles specifically targeted by this trigger
     * @return mixed
     */
    public function styles() {
        return $this
            ->morphedByMany(Style::class, 'product',
                'PromotionOrderQuantityTrigger_Product', 'promotionorderquantitytrigger_id')
            ->where('PromotionOrderQuantityTrigger_Product.deleted_at', '=', '0000-00-00 00:00:00')
            ;
    }

    /**
     * Get all WebUnits specifically targeted by this trigger
     * @return mixed
     */
    public function units() {
        return $this
            ->morphedByMany(Unit::class, 'product',
                'PromotionOrderQuantityTrigger_Product', 'promotionorderquantitytrigger_id')
            ->where('PromotionOrderQuantityTrigger_Product.deleted_at', '=', '0000-00-00 00:00:00')
            ;
    }

    /**
     * Gets the total number of 'qualifying' items in the cart, where 'qualifying' is defined as
     * - contained in the list of targeted web products (units, styles, brands), if not targeting all products
     * @param $webCartItems
     * @return int
     */
    public function getCurrentQualifyingCartTotal($webCartItems) {
        $totalItemQuantity = 0;

        if (isset($webCartItems) && count($webCartItems) > 0) {
            if ($this->target_all_products) {
                foreach ($webCartItems as $webCartItem) {
                    $totalItemQuantity = $totalItemQuantity + $webCartItem->quantity;
                }
            } else {
                $relevantUnits = $this->units;
                $relevantStyles = $this->styles;
                $relevantBrands = $this->brands;

                foreach ($webCartItems as $webCartItem) {
                    $unit = $webCartItem->unit;
                    $style = $unit->styleRelation;
                    $brand = $style->brand;
                    if ((isset($unit) && $relevantUnits->contains($unit)) ||
                        (isset($style) && $relevantStyles->contains($style)) ||
                        (isset($brand) && $relevantBrands->contains($brand))) {
                        $totalItemQuantity = $totalItemQuantity + $webCartItem->quantity;
                    }
                }
            }
        }

        return $totalItemQuantity;
    }
}