<?php

namespace Marcolin;

use Carbon\Carbon;
use Marcolin\Models\Customer;
use Marcolin\Models\WebSilo;

class B2BHelper
{
    protected $activeCustomer;
    protected $activeWebSilo;
    protected $activeShippingAddress;

    public function __construct()
    {
        $activeShippingAddress = null;
        $this->loadActiveCustomer();
    }

    protected function loadActiveCustomer()
    {
        $this->activeCustomer = false;

        $aliasedCustomerId = session()->get('aliasedCustomerId', false);

        if ($aliasedCustomerId) {
            $this->activeCustomer = Customer::where('Customers.id', '=', $aliasedCustomerId)
                ->with('customerDiscounts')
                ->with('brands')
                ->first();
        } else {
            $user = auth()->user();
            if ($user) {
                $this->activeCustomer = $user
                    ->customer()
                    ->with('customerDiscounts')
                    ->with('brands')
                    ->first();
            }
        }

        $this->setActiveShippingAddress();

        $this->loadActiveWebSilo();
    }

    protected function loadActiveWebSilo() {
        $this->activeWebSilo = false;
        $user = auth()->user();

        $activeWebSiloId = false;

        if (!$activeWebSiloId) {
            if ($user) {
                if ($this->activeCustomer) {
                    $defaultCustomerWebSilo = $this->activeCustomer->webSilo;
                    if ($defaultCustomerWebSilo && count($defaultCustomerWebSilo) > 0) {
                        $activeWebSiloId = $defaultCustomerWebSilo[0]->id;
                    }

                    if (config('marcolin.buying_group_for_microsites') == 1) {
                        $defaultBuyingGroupWebSilo = $this->activeCustomer->buyingGroup->webSilo;
                        if ($defaultBuyingGroupWebSilo && count($defaultBuyingGroupWebSilo) > 0) {
                            $activeWebSiloId = $defaultBuyingGroupWebSilo[0]->id;
                        }
                    }
                }

                if (!$activeWebSiloId) {
                    $userSilo = $user->defaultWebSilo;
                    if ($userSilo) {
                        $activeWebSiloId = $userSilo->id;
                    }
                }
            }

            if (!$activeWebSiloId) $activeWebSiloId = WebSilo::first()->id;
        }
        $this->activeWebSilo = WebSilo::with('webLandingPage')->find($activeWebSiloId);
        session()->put('b2b.activeWebSiloId', $activeWebSiloId);

        return $this->activeWebSilo;
    }

    public function activeWebHierarchy() {
        return $this->activeWebSilo()->webHierarchy;
    }

    /** @return WebSilo */
    public function activeWebSilo() {
        return $this->activeWebSilo;
    }

    public function aliasAs($customerId = false)
    {
        if ($customerId) {
            session()->put('aliasedCustomerId', $customerId);
        } else {
            session()->forget('aliasedCustomerId');
            session()->forget('activeWebSiloId');
            $this->loadActiveWebSilo();
        }
        $this->loadActiveCustomer();
    }

    public function isAliased()
    {
        return (session()->get('aliasedCustomerId', false));

    }

    public function stopAliasing()
    {
        $this->aliasAs(false);
    }

    public function setActiveShippingAddress($customerShippingAddress = false)
    {
        if ($customerShippingAddress === false) {
            if ($this->activeCustomer) {
                $customerShippingAddress = $this->activeCustomer->default_shipping_address;
            }
        }
        $this->activeShippingAddress = $customerShippingAddress;
        return $this->activeShippingAddress;
    }

    public function activeCustomer()
    {
        return $this->activeCustomer;
    }

    public function activeShippingAddress()
    {
        return $this->activeShippingAddress;
    }

    public function activeCountry()
    {
        if ($this->activeShippingAddress) return $this->activeShippingAddress()->country;

        return null;
    }

    public function activeCurrency()
    {
        $activeCurrency = 'USD';

        if ($this->activeCustomer) {
            $activeCurrency = $this->activeCustomer->currency;
        }

        return $activeCurrency;
    }

    public function currencySymbol($currencyCode = false)
    {
        if ($currencyCode === false) $currencyCode = $this->activeCurrency();
        if ($currencyCode == 'GBP') return '&pound;';

        return '$';
    }

    public function formatPrice($price, $decimals = -1, $symbol = '$')
    {
        if ($decimals = -1) {
            if ($price == round($price)) $decimals = 0;
            else $decimals = 2;
        }
        return $symbol . number_format($price, $decimals);
    }

    public function activeSalesOrg()
    {
        return array_get($this->activeCustomer, 'salesOrg', 'UN01');
    }

    public function getEndOfCurrentQuarter($startDate = false) {
        if (!$startDate) {
            $current_month = date('m', $startDate);
            $current_year = date('Y', $startDate);
        } else {
            $current_month = $startDate->month;
            $current_year = $startDate->year;
        }

        if($current_month>=1 && $current_month<=3)
        {
            $end_of_quarter_date = strtotime('1-April-'.$current_year);  // timestamp or 1-April 12:00:00 AM means end of 31 March
        }
        else  if($current_month>=4 && $current_month<=6)
        {
            $end_of_quarter_date = strtotime('1-July-'.$current_year);  // timestamp or 1-July 12:00:00 AM means end of 30 June
        }
        else  if($current_month>=7 && $current_month<=9)
        {
            $end_of_quarter_date = strtotime('1-October-'.$current_year);  // timestamp or 1-October 12:00:00 AM means end of 30 September
        }
        else  if($current_month>=10 && $current_month<=12)
        {
            $end_of_quarter_date = strtotime('1-January-'.($current_year+1));  // timestamp or 1-January Next year 12:00:00 AM means end of 31 December this year
        }
        return $end_of_quarter_date;
    }

    function getRealQuery($query, $dumpIt = false) {
        $params = array_map(function ($item) {
            return "'{$item}'";
        }, $query->getBindings());
        $result = str_replace_array('\?', $params, $query->toSql());
        if ($dumpIt) {
            dd($result);
        }
        return $result;
    }
}