<?php

namespace Marcolin\Models;

use AlgoliaSearch\Laravel\AlgoliaEloquentTrait;
use Auth;
use DB;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\App;
use Marcolin\Facades\MarcolinAssets;
use Venturecraft\Revisionable\RevisionableTrait;

/**
 * Class WebUnit
 * @package Marcolin\Models
 * @property int $id
 * @property string $name
 * @property string $part_detail
 * @property string $safe_name
 * @property string $override_name
 * @property-read Unit $unit
 * @property-read WebUnitFamily $family
 * @property-read WebUnitAttributeValue[] $attributeValues
 * @method static Builder latest()
 * @property string $part_number
 * @property integer $part_id
 * @property integer $webunitfamily_id
 * @property string $date_created
 * @property string $date_modified
 * @property string $date_uploaded
 * @property string $deleted_at
 * @property-read \Marcolin\Models\WebUnitFamily $webUnitFamily
 * @property-write mixed $web_part_attributes
 * @property-read mixed $part_name
 * @property-read mixed $family_name
 * @property-read mixed $context
 * @property-read mixed $default_context
 * @mixin \Eloquent
 * @method static WebUnit firstOrNew(array $attributes)
 */
class WebUnit extends WebModel
{
    protected $table = "WebUnits";
    protected $fillable = [];
    protected $casts = [
        ['is_visible' => 'boolean', 'is_family_image' => 'boolean']
    ];

    use AlgoliaEloquentTrait;

    public static $perEnvironment = true;

    public $algoliaSettings = [
        'searchableAttributes' => [
            'style_suffixes',
            'unit.upc',
            'unit.name',
            'unit.color',
            'unit.size',
            'unit.colorDescription',
        ],
        'attributesForFaceting' => [
            'unit.salesOrg',
            'visibility',
        ]
    ];

    public function _reindex($safe = true, $setSettings = true, $mergeOldSettings = false, \Closure $onInsert = null)
    {
        /** @var \AlgoliaSearch\Laravel\ModelHelper $modelHelper */
        $modelHelper = App::make('\AlgoliaSearch\Laravel\ModelHelper');

        $indices = $modelHelper->getIndices($this);
        $indicesTmp = $safe ? $modelHelper->getIndicesTmp($this) : $indices;

        if ($setSettings === true) {
            $setToTmpIndices = ($safe === true);
            $this->_setSettings($setToTmpIndices, $mergeOldSettings);
        }

        static::has('webStyle.webBrand.webSilos')
            ->has('unit.styleRelation.brand')
            ->with('webStyle.webBrand.webSilos')
            ->with('unit.styleRelation.productDescription')
            ->with('unit.styleRelation.productShape')
            ->with('unit.styleRelation.productRimType')
            ->with('unit.styleRelation.productFitting')
            ->with('unit.styleRelation.brand')
            ->with('unit.productLensType')
            ->with('unit.productColorFamily')
            ->webVisibleAll()
            ->chunk(500, function ($models) use ($indicesTmp, $modelHelper, $onInsert) {
                /** @var \AlgoliaSearch\Index $index */
                foreach ($indicesTmp as $index) {
                    $records = [];
                    $recordsAsEntity = [];

                    foreach ($models as $model) {
                        if ($modelHelper->indexOnly($model, $index->indexName)) {
                            $records[] = $model->getAlgoliaRecordDefault($index->indexName);

//                            dd($records);

                            if ($onInsert && is_callable($onInsert)) {
                                $recordsAsEntity[] = $model;
                            }
                        }
                    }

                    $index->addObjects($records);

                    if ($onInsert && is_callable($onInsert)) {
                        call_user_func_array($onInsert, [$recordsAsEntity]);
                    }
                }

            });

        if ($safe) {
            for ($i = 0; $i < count($indices); $i++) {
                $modelHelper->algolia->moveIndex($indicesTmp[$i]->indexName, $indices[$i]->indexName);
            }

            $this->_setSettings(false); // To a setSettings to set the slave on the master
        }
    }

    public function indexOnly()
    {
        return true;
    }

    public function getAlgoliaRecord()
    {
        $record = array_only($this->toArray(), [
            'id',
        ]);

        $record['displayName'] = $this->webStyle->displayName;
        $record['name'] = $this->webStyle->name;
        $record['slug'] = $this->webStyle->slug;
        $record['visibility'] = [];
        if ($this->isWebVisibleStandard) {
            $record['visibility'][] = 'standard';
        }
        if ($this->isWebVisibleHarleyDavidson) {
            $record['visibility'][] = 'harley';
        }
        if ($this->isWebVisibleAutocat) {
            $record['visibility'][] = 'autocat';
        }
        if ($this->isWebVisibleTomFordLimited) {
            $record['visibility'][] = 'tom_ford_limited';
        }

        $record['unit'] = $this->unit->toArray();

        $unwanted_keys = ['date_created', 'date_modified', 'date_uploaded', 'deleted_at', 'commit_sequence'];
        $this->recursive_unset($record, $unwanted_keys);

        for($index = 0; $index < strlen($record['unit']['style']); $index++) {
            $record['style_suffixes'][] = substr($record['unit']['style'], $index);
        }

        return $record;
    }

    function recursive_unset(&$array, $unwanted_keys)
    {
        foreach ($unwanted_keys as $unwanted_key) {
            unset($array[$unwanted_key]);
        }
        foreach ($array as &$value) {
            if (is_array($value)) {
                $this->recursive_unset($value, $unwanted_keys);
            }
        }
    }

    protected $indexSettings = [
        'analysis' => [
            'char_filter' => [
                'replace' => [
                    'type' => 'mapping',
                    'mappings' => [
                        '&=> and '
                    ],
                ],
            ],
            'filter' => [
                'word_delimiter' => [
                    'type' => 'word_delimiter',
                    'split_on_numerics' => false,
                    'split_on_case_change' => false,
                    'generate_word_parts' => true,
                    'generate_number_parts' => true,
                    'catenate_all' => true,
                    'preserve_original' => true,
                    'catenate_numbers' => true,
                ]
            ],
            'analyzer' => [
                'default' => [
                    'type' => 'custom',
                    'char_filter' => [
                        'html_strip',
                        'replace',
                    ],
                    'tokenizer' => 'whitespace',
                    'filter' => [
                        'lowercase',
                        'word_delimiter',
                    ],
                ],
            ],
        ],
    ];

    protected $mappingProperties = [
        'title' => [
            'type' => 'string',
            'analyzer' => 'standard'
        ]
    ];

//    use ElasticquentTrait;

    public static $elasticFuzziness = 1;

    /**
     * @description return elastic query with fuzziness
     */
    public static function buildElasticQuery($keyWords)
    {
        $params = [
            'multi_match' => [
                'query' => $keyWords,
                'fuzziness' => self::$elasticFuzziness,
                'fields' => ['_all']
            ]
        ];
        return $params;
    }

//	use KeyWordsFunctional;

    /**
     * @description index all the entries in an Eloquent model (if needed)
     */
    public static function elasticIndexingIfNeeded()
    {
        $mappingExists = self::mappingExists();
        if (!$mappingExists) {
            self::createIndex();
            self::putMapping();
            self::addAllToIndex();
        }
    }

    public function unit()
    {
        return $this->belongsTo(Unit::class, 'unit_id', 'id');
    }

    public function webUnit_Algolia()
    {
        return $this->hasMany(WebUnit_Algolia::class, 'webUnit_id', 'id');
    }

    public function algoliaTable()
    {
        return $this->hasMany(WebUnit_Algolia::class, 'webUnit_id', 'id');
    }

    public function getAlgoliaClassAttribute() {
        return WebUnit_Algolia::class;
    }

    public function getAlgoliaIDFieldAttribute() {
        return 'webUnit_id';
    }

    public function toString() {
        return $this->unit->style . ' - ' . $this->unit->color;
    }

    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function webStyle()
    {
        return $this->belongsTo(WebStyle::class, 'webstyle_id', 'id');
    }

    public function promos()
    {
        return $this->belongsToMany(Promo::class, 'Promo_WebUnit', 'webunit_id', 'promo_id');
    }

    public function scopeIsFamilyImage($query)
    {
        $query->where('is_family_image', 1)->visible()->whereHas('part', function ($query) {
            return $query->whereHas('inventoryItems', function ($query) {
                return $query->isValid();
            });
        });
    }

    public function scopePriceList($query, $priceList, $currency, $quantityLevel = 1)
    {
        return $query
            ->leftJoin('PriceLists', 'PriceLists.part_no', '=', 'WebUnits.part_number')
            ->where('price_list', $priceList)
            ->where('currency', $currency)
            ->where('quantity_level', '>=', $quantityLevel)
            ->select('WebUnits.*', 'PriceLists.price');
    }

    public function webAttributes()
    {
        return $this->belongsToMany(WebAttribute::class, 'WebAttribute_WebUnit', 'webunit_id', 'webattribute_id')
            ->withPivot(['id', 'attribute_value']);
    }

    public function webAttributeValue($webAttributeId)
    {
        if (!$this->relationLoaded('webAttributes')) $this->load('webAttributes');
        $webAttributeValue = $this->webAttributes->find($webAttributeId);
        if (!is_null($webAttributeValue)) return $webAttributeValue->pivot->attribute_value;
        return null;
    }

    public function setWebAttributeValue($webAttributeId, $value)
    {
        $this->webAttributes()->sync([$webAttributeId => ['attribute_value' => $value]], false);
    }

    public function setWebAttributesAttribute($attributes)
    {
        $this->save();
        foreach ($attributes as $webAttributeId => $attributeValue) {
            $this->setWebAttributeValue($webAttributeId, $attributeValue);
        }
    }

    public function getPartNameAttribute()
    {
        $part = $this->unit;
        if (is_null($part)) return null;
        return $part->unit_desc;
    }

    public function getFamilyNameAttribute()
    {
        $family = $this->webStyle;
        if (is_null($family)) return null;
        return $family->name;
    }

    public function getCustomerPrice($customer = false)
    {
        if ($customer === false) $customer = b2b()->activeCustomer();
        return $this->unit->getCustomerPrice($customer);
    }

    public function getCustomerDiscount($customer)
    {
        return $this->unit->getCustomerDiscount($customer);
    }

//	public static function getImageUrl($partNumber, $width = null, $height = null, $avoidCache=false) {
//		$url = config('marcolin.images.url');
//		if(!isset($url)) $url = config('marcolin-b2b.images.url');
//		$url .= '/' . $partNumber;
//		if ($width) $url .= "/$width";
//		if ($height) $url .= "/$height";
//
//		return ($avoidCache)?\ImgHelper::avoidCache($url):$url;
//	}

    public function scopePriceRange($query, $min, $max)
    {
        return $query->whereHas('part', function ($part) use ($min, $max) {
            $part->whereBetween('list_price', [$min, $max]);
        });
    }

    public function scopeManagedBy($query, $manager)
    {
        return $query->whereHas('part', function ($query) use ($manager) {
            return $query->where('pf_prod_mgr', $manager);
        });
    }

    public function scopeInCollection($query, $webCollectionId)
    {
        return $query->whereHas('webStyle', function ($query) use ($webCollectionId) {
            return $query->inCollection($webCollectionId);
        });
    }

    public function scopeInCategory($query, $webCategoryId)
    {
        return $query->whereHas('webStyle', function ($query) use ($webCategoryId) {
            return $query->inCategory($webCategoryId);
        });
    }

    public function scopeInGroup($query, $webBrandId)
    {
        return $query->whereHas('webStyle', function ($query) use ($webBrandId) {
            return $query->inGroup($webBrandId);
        });
    }

    public function getIsWebVisibleAttribute()
    {
        return static::webVisible()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleAllAttribute()
    {
        return static::webVisibleAll()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleAutocatAttribute()
    {
        return static::webVisibleAutocat()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleStandardAttribute()
    {
        return static::webVisibleStandard()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleHarleyDavidsonAttribute()
    {
        return static::webVisibleHarleyDavidson()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleTomFordLimitedAttribute()
    {
        return static::webVisibleTomFordLimited()->where('id', '=', $this->id)->count() > 0;
    }

    public function scopeWebVisible($query)
    {
        $query
            ->whereHas('unit', function ($query) {
                return $query->webVisible();
            });
        return $query;
    }

    public function scopeWebVisibleAll(Builder $query)
    {
        $query
            ->whereHas('unit', function ($query) {
                return $query->webVisibleAll();
            });
        return $query;
    }

    public function scopeWebVisibleAutocat(Builder $query)
    {
        $query
            ->whereHas('unit', function ($query) {
                return $query->webVisibleAutocat();
            });
        return $query;
    }

    public function scopeWebVisibleStandard($query)
    {
        $query
            ->whereHas('unit', function ($query) {
                return $query->webVisibleStandard();
            });
        return $query;
    }

    public function scopeWebVisibleHarleyDavidson($query)
    {
        $query
            ->whereHas('unit', function ($query) {
                return $query->webVisibleHarleyDavidson();
            });
        return $query;
    }

    public function scopeWebVisibleTomFordLimited($query)
    {
        $query
            ->whereHas('unit', function ($query) {
                return $query->webVisibleTomFordLimited();
            });
        return $query;
    }

    public function visibleInSilo($webSilo)
    {
        if (!$webSilo instanceof WebSilo) $webSilo = WebSilo::find($webSilo);
        $inSilo = true;
        return $inSilo;
    }

    public function getInStockLabelAttribute()
    {
        return $this->unit->in_stock_label;
    }

    public function getOutOfStockLabelAttribute()
    {
        return $this->unit->out_of_stock_lavel;
    }

    public function getStockStatusAttribute()
    {
        return $this->unit->stock_status;
    }

    public function getStockLabelAttribute()
    {
        return $this->unit->stock_label;
    }

    use HasAssets, RevisionableTrait;

}
