<?php

namespace Hilco\Models;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use LogicException;

/**
 * Class Promotion
 * @package Hilco\Models
 *
 * @method static Builder active()
 */
class Promotion extends WebModel
{
    protected $table = 'Promotions';
    protected $fillable = ['name', 'start_date', 'end_date', 'enabled'];
    protected $casts = [
        'enabled' => 'boolean',
        'start_date' => 'datetime',
        'end_date' => 'datetime',
    ];

    public function customerTriggers() {
        return $this->morphedByMany(PromotionCustomerTrigger::class, 'trigger', 'PromotionTriggers', 'promotion_id')
            ->withPivot(['sequence'])
        ;
    }

    public function currentOrderTriggers() {
        return $this->morphedByMany(PromotionCurrentOrderTrigger::class, 'trigger', 'PromotionTriggers', 'promotion_id')
            ->withPivot(['sequence'])
        ;
    }

    public function getAllTriggersAttribute() {
        return collect([
            $this->customerTriggers->flatten(),
            $this->currentOrderTriggers->flatten(),
        ])->flatten();
    }

    public static function getTriggerClassFromType($triggerType) {
        return array_get(Relation::morphMap(), $triggerType, $triggerType);
    }

    public static function createTriggerFromType($triggerType) {
        $class = self::getTriggerClassFromType($triggerType);
        return new $class;
    }

    public static function findTriggerFromType($triggerType, $triggerId) {
        $class = self::getTriggerClassFromType($triggerType);
        return $class::find($triggerId);
    }

    public function getTriggerRelationFromType($triggerType) {
        $relationName = $triggerType . 's';
        $relation = $this->$relationName();
        return $relation;
    }

    public function scopeActive($query) {
        $query
            ->where('enabled', 1)
            ->where(function ($query) {
                return $query
                    ->where('start_date', '=', '0000-00-00 00:00:00')
                    ->orWhere('start_date', '<=', DB::raw('NOW()'))
                    ;
            })
            ->where(function ($query) {
                return $query
                    ->where('end_date', '=', '0000-00-00 00:00:00')
                    ->orWhere('end_date', '>=', DB::raw('NOW()'))
                    ;
            })
        ;
    }
}
