<?php

namespace Hilco\Models;
use Illuminate\Database\Eloquent\Builder;

/**
 * Class Promotion
 * @package Hilco\Models
 *
 * @method static Builder active()
 */
class PromotionCurrentOrderTrigger extends PromotionTriggerDetails
{
    protected $table = 'PromotionCurrentOrderTriggers';
    protected $fillable = ['via_web', 'via_cst', 'minimum_dollar', 'almost_qualified', 'commit_sequence'];

    public function hasAlmostQualifyingField() {
        return true;
    }

    public function webGroups() {
        return $this
            ->morphedByMany(WebGroup::class, 'product',
                            'PromotionCurrentOrderTrigger_Product', 'promotioncurrentordertrigger_id')
            ->withPivot(['excluded'])
            ->whereNull('PromotionCurrentOrderTrigger_Product.deleted_at')
        ;
    }

    public function webCategories() {
        return $this
            ->morphedByMany(WebCategory::class, 'product',
                            'PromotionCurrentOrderTrigger_Product', 'promotioncurrentordertrigger_id')
            ->withPivot(['excluded'])
            ->whereNull('PromotionCurrentOrderTrigger_Product.deleted_at')
        ;
    }

    public function webCollections() {
        return $this
            ->morphedByMany(WebCollection::class, 'product',
                            'PromotionCurrentOrderTrigger_Product', 'promotioncurrentordertrigger_id')
            ->withPivot(['excluded'])
            ->whereNull('PromotionCurrentOrderTrigger_Product.deleted_at')
        ;
    }

    public function webFamilies() {
        return $this
            ->morphedByMany(WebFamily::class, 'product',
                            'PromotionCurrentOrderTrigger_Product', 'promotioncurrentordertrigger_id')
            ->withPivot(['excluded'])
            ->whereNull('PromotionCurrentOrderTrigger_Product.deleted_at')
        ;
    }

    public function webParts() {
        return $this
            ->morphedByMany(WebPart::class, 'product',
                            'PromotionCurrentOrderTrigger_Product', 'promotioncurrentordertrigger_id')
            ->withPivot(['excluded'])
            ->whereNull('PromotionCurrentOrderTrigger_Product.deleted_at')
        ;
    }

    public function getRelationForProductTableType($tableType) {
        if($tableType === 'webFamily' || $tableType === 'webCategory') {
            $relationName = substr($tableType, 0, -1);
            $relationName .= 'ies';
        } else {
            $relationName = $tableType . 's';
        }
        return $this->$relationName();
    }

    public function isTriggered(Customer $soldToCustomer, $webCartItems, $distinctPartInfo = []) {
        $totalCartPrice = $this->getCurrentQualifyingCartTotal($soldToCustomer, $webCartItems, $distinctPartInfo);

        return $totalCartPrice >= $this->minimum_dollar;
    }

    public function isAlmostQualifying(Customer $soldToCustomer, $webCartItems, $distinctPartInfo = []) {
        $almostQualifyingCount = $this->getCurrentQualifyingCartTotal($soldToCustomer, $webCartItems, $distinctPartInfo);
        return ($almostQualifyingCount >= $this->almost_qualified) ?
            $this->minimum_dollar - $almostQualifyingCount : -1;
    }

    public function getCurrentQualifyingCartTotal(Customer $soldToCustomer, $webCartItems, $distinctPartInfo = []) {
        $distinctPartQuantities = [];
        foreach ($webCartItems as $webCartItem)
            array_increment($distinctPartQuantities, $webCartItem->webPart->part->id, $webCartItem->quantity);

        $webSilo = b2b()->activeWebSilo();
        $customerPriceListName = array_get($soldToCustomer, 'activeSegment.def_price_list');
        $currencyCode = $soldToCustomer->currency;
        $siloPriceListName = $webSilo->priceList;

        $customerPrices = PriceList
            ::where('price_list', '=', $customerPriceListName)
            ->where('currency', '=', $currencyCode)
            ->whereIn('part_id', array_keys($distinctPartQuantities))
            ->orderBy('quantity_level', 'DESC')
            ->get();

        $siloPrices = PriceList
            ::where('price_list', '=', $siloPriceListName)
            ->where('currency', '=', $currencyCode)
            ->whereIn('part_id', array_keys($distinctPartQuantities))
            ->orderBy('quantity_level', 'DESC')
            ->get();

        $totalCartPrice = 0;
        foreach ($webCartItems as $webCartItem) {
            $part = $webCartItem->webPart->part;

            $partTotalQuantity = array_get($distinctPartQuantities, $part->id, 0);

            $customerPriceList = PriceList::filterForPartQuantity($customerPrices, $part->id, $partTotalQuantity);

            if ($customerPriceList) {
                $basePrice = $customerPriceList->price;
                $isContractPrice = $customerPriceList->contract_flag;
                $usedPriceList = $customerPriceListName;
            } else {
                $siloPriceList = PriceList::filterForPartQuantity($siloPrices, $part->id, $partTotalQuantity);
                if ($siloPriceList) {
                    $basePrice = $siloPriceList->price;
                    $isContractPrice = false;
                    $usedPriceList = $siloPriceListName;
                } else {
                    $basePrice = $part->getListPriceForCurrencyCode($currencyCode);
                    $isContractPrice = false;
                    $usedPriceList = 'PART_TABLE_LIST_PRICE';
                }
            }
            $totalCartPrice += $basePrice * $webCartItem->quantity;
        }

        return $totalCartPrice;
    }

    public function setMinimumDollarAttribute($value) {
        $this->attributes['minimum_dollar'] = $value;
    }

    public function postTriggerOperation($soldToCustomer) {
        // TODO: Implement postTriggerOperation() method.
    }

    public static function boot() {
        parent::boot();

        static::deleted(function (PromotionCurrentOrderTrigger $promotionCurrentOrderTrigger) {
            $promotionCurrentOrderTriggerProducts =
                PromotionCurrentOrderTrigger_Product::where('promotioncurrentordertrigger_id', '=', $promotionCurrentOrderTrigger->id);
            foreach ($promotionCurrentOrderTriggerProducts as $promotionCurrentOrderTriggerProduct) {
                $promotionCurrentOrderTriggerProduct->delete();
            }
        });
    }
}
