<?php

namespace Hilco\Models;
use AlgoliaSearch\Laravel\AlgoliaEloquentTrait;
use DB;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Venturecraft\Revisionable\RevisionableTrait;

class WebCollection extends WebModel implements HasLandingPageInterface {
    protected $table = "WebCollections";
    protected $fillable = ['name', 'slug', 'keywords', 'webCategoryIds', 'is_visible', 'detail', 'sort_priority'];
    protected $casts = [
        ['is_visible' => 'boolean']
    ];

//    use KeyWordsFunctional;
    use AlgoliaEloquentTrait;

    public $algoliaSettings = [
        'searchableAttributes' => [
            'name',
        ],
        'attributesForFaceting' => [
            'webSiloIds',
        ]
    ];

    public function _reindex($safe = true, $setSettings = true, $mergeOldSettings = false, \Closure $onInsert = null)
    {
        /** @var \AlgoliaSearch\Laravel\ModelHelper $modelHelper */
        $modelHelper = App::make('\AlgoliaSearch\Laravel\ModelHelper');

        $indices = $modelHelper->getIndices($this);
        $indicesTmp = $safe ? $modelHelper->getIndicesTmp($this) : $indices;

        if ($setSettings === true) {
            $setToTmpIndices = ($safe === true);
            $this->_setSettings($setToTmpIndices, $mergeOldSettings);
        }

        static::has('webCategories.webGroups.webSilos')->with('webCategories.webGroups.webSilos.plants')->chunk(100, function ($models) use ($indicesTmp, $modelHelper, $onInsert) {
            /** @var \AlgoliaSearch\Index $index */
            foreach ($indicesTmp as $index) {
                $records         = [];
                $recordsAsEntity = [];

                foreach ($models as $model) {
                    if ($modelHelper->indexOnly($model, $index->indexName)) {
                        $records[] = $model->getAlgoliaRecordDefault($index->indexName);

                        if ($onInsert && is_callable($onInsert)) {
                            $recordsAsEntity[] = $model;
                        }
                    }
                }

                $index->addObjects($records);

                if ($onInsert && is_callable($onInsert)) {
                    call_user_func_array($onInsert, [$recordsAsEntity]);
                }
            }

        });

        if ($safe) {
            for ($i = 0; $i < count($indices); $i++) {
                $modelHelper->algolia->moveIndex($indicesTmp[$i]->indexName, $indices[$i]->indexName);
            }

            $this->_setSettings(false); // To a setSettings to set the slave on the master
        }

    }

    public function getAlgoliaRecord() {
        $record = array_only($this->toArray(), [
            'id',
            'name',
            'slug',
        ]);
        $paths = $this->hierarchyPaths;
        $silos = [];
        foreach ($paths as $path => $info) {
            $silos[$info['webSilo']->id] = $info['webSilo']->id;
        }
        $record['webSiloIds'] = array_keys($silos);
        return $record;
    }

    public function webCategories() {
        return $this->belongsToMany(WebCategory::class, 'WebCollection_WebCategory', 'webcollection_id', 'webcategory_id')->withPivot('display_order');
    }

    /**
     * @return BelongsToMany
     */
    public function webFamilies() {
        return $this->belongsToMany(WebFamily::class, 'WebFamily_WebCollection', 'webcollection_id', 'webfamily_id')->withPivot(['is_featured', 'display_order']);
    }

    public function visibleWebFamilies() {
        return $this->belongsToMany(WebFamily::class, 'WebFamily_WebCollection', 'webcollection_id', 'webfamily_id')
            ->withPivot(['is_featured', 'display_order'])
            ->visible()
            ->displayOrder()
        ;
    }

    public function scopeDisplayOrder(Builder $query) {
        return $query->orderBy('pivot_display_order', 'asc')->orderBy('name', 'asc');
    }

    public function scopeAlphabetical(Builder $query) {
        return $query->orderBy('name', 'asc');
    }

    public function getFeaturedChildrenAttribute() {
        return $this->visibleWebFamilies()->wherePivot('is_featured', 1)->get();
    }

    public function getChildrenAttribute() {
        return $this->visibleWebFamilies()->get();
    }

    public function scopeInCategory($query, $webCategoryId) {
        return $query->whereHas('webCategories', function ($query) use ($webCategoryId) {
            $query->where('webcategory_id', $webCategoryId);
        });
    }

    public function scopeHavingFamily($query, $webFamilyId) {
        return $query->whereHas('webFamilies', function ($query) use ($webFamilyId) {
            $query->where('webfamily_id', $webFamilyId);
        });
    }

    public function setWebCategoryIdsAttribute($ids) {
        $this->save();
        $this->createRevisionRecord('webCategoryIds', $this->webCategories->pluck('id')->toJson(), json_encode($ids));
        $this->webCategories()->sync($ids);
    }

    public function getHierarchyPathsAttribute() {
        $paths = [];
        foreach ($this->webCategories as $webCategory) {
            foreach ($webCategory->webGroups as $webGroup) {
                foreach ($webGroup->webSilos as $webSilo) {
                    $paths[] = [
                        'webSilo' => $webSilo,
                        'webGroup' => $webGroup,
                        'webCategory' => $webCategory,
                    ];
                }
            }
        }
        return $paths;
    }

    public function scopeInCollection($query, $webCollection) {
        return $query->whereHas('webCollections', function ($query) use ($webCollection) {
            $query->where('webcollection_id', $webCollection->id);
        });
    }

    public function scopeInGroup($query, $webGroupId) {
        return $query->whereHas('webCategories', function ($query) use ($webGroupId) {
            return $query->inGroup($webGroupId);
        });
    }

    public function scopeInSilo($query, $webSiloId) {
        return $query->whereHas('webCategories', function ($query) use ($webSiloId) {
            return $query->inSilo($webSiloId);
        });
    }    

    public function getAssetPrefix() {
        return "webCollections/{$this->id}";
    }

    public function scopeVisible(Builder $query) {
        return $query
            ->where('WebCollections.is_visible', '1')
            ->whereHas('webCategories', function ($query) {
                return $query->visible();
            })
        ;
    }

    public function slugUrl() {
        return route('collection.slug', [$this->slug]);
    }

    public function getParentRelationName() {
        return 'webCategories';
    }

    public function getPrimaryImage() {
        $asset = $this->assetsByType('primary')->first();
        if (!is_null($asset)) return $asset;

        $webFamily = $this->visibleWebFamilies->first();
        if ($webFamily) return $webFamily->getPrimaryImage();
    }


    use HasAssets, HasSlug, HasLandingPage, RevisionableTrait, ManyToManyRevisionableTrait;
}
