<?php

namespace Hilco\Models;

use DB;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Venturecraft\Revisionable\RevisionableTrait;

class WebGroup extends WebModel implements HasLandingPageInterface, HasWebSiloOverridesInterface {
	protected $table = "WebGroups";
	protected $fillable = ['name', 'slug', 'webHierarchyIds' ,'is_visible', 'detail'];
	protected $casts = [
	    'is_visible' => 'boolean',
        'layout' => 'array',
	];
    protected $with = ['webSiloOverride'];

    protected $hierarchyParent = false;

    public function webHierarchies() {
        return $this->belongsToMany(WebHierarchy::class, 'WebGroup_WebHierarchy', 'webgroup_id', 'webhierarchy_id');
    }

//	public function webSilos() {
//		return $this->belongsToMany(WebSilo::class, 'WebGroup_WebSilo', 'webgroup_id', 'websilo_id')->withPivot('display_order');
//	}

	public function webCategories() {
		return $this->belongsToMany(WebCategory::class, 'WebCategory_WebGroup', 'webgroup_id', 'webcategory_id')->withPivot('display_order');
	}

	public function webTranslations(){
        return $this->hasMany(Translations_WebGroup::class, 'webgroup_id', 'id');
    }

    public function visibleWebCategories() {
        return $this->belongsToMany(WebCategory::class, 'WebCategory_WebGroup', 'webgroup_id', 'webcategory_id')
            ->withPivot('display_order', 'is_featured')
            ->webVisible()
            ->displayOrder()
        ;
    }

    public function scopeDisplayOrder(Builder $query) {
        return $query->orderBy('pivot_display_order', 'asc')->orderBy('name', 'asc');
    }

    public function scopeAlphabetical(Builder $query) {
        return $query->orderBy('name', 'asc');
    }

	public function webCategoriesOrdered() {
        return $this->webCategories()->orderBy('pivot_display_order')->orderBy('name')
            ;
    }

	public function webLandingPages() {
		return $this->belongsToMany(WebLandingPage::class, 'WebGroup_WebLandingPage', 'webgroup_id', 'weblandingpage_id');
	}

	public function getFeaturedChildrenAttribute() {
        return $this->visibleWebCategories()->wherePivot('is_featured', 1)->get();
    }

    public function getChildrenAttribute() {
        $children = $this->visibleWebCategories();
        if (request()->get('ap', false) == 1) $children->webSiloApproved();

        return $children->get();
    }

    public function getDisplayNameForUACAttribute() {
        return $this->name;
    }

    public function getNameAttribute() {
        if (config('hilco.ignoreActiveWebSilo')) {
            return $this->getTranslation('name', 'en', $this->attributes['name']);
        } else {
            return (isset($this->webSiloOverride) && $this->webSiloOverride->is_visible == 1) ?
                $this->webSiloOverride->name :
                $this->getTranslation('name');
        }
    }

    public function getNameForEditAttribute() {
        return $this->getTranslation('name', 'en', $this->attributes['name']);
    }

    public function getDetailAttribute() {
        if (config('hilco.ignoreActiveWebSilo')) {
            return $this->getTranslation('detail', 'en', $this->attributes['detail']);
        } else {
            return (isset($this->webSiloOverride) && $this->webSiloOverride->is_visible == 1) ?
                $this->webSiloOverride->detail :
                $this->getTranslation('detail')
            ;
        }
    }

    public function setWebHierarchyIdsAttribute($ids) {
		$this->save();
        $this->createRevisionRecord('webHierarchyIds', $this->webHierarchies->pluck('id')->toJson(), json_encode($ids));
        $this->webHierarchies()->sync($ids);
	}

	public function scopeHasVisibleChildren(Builder $query, $activeWebSilo = false, $activeCountry = false, $activePlant = false) {
        return $query
            ->whereHas('webCategories', function ($query) use ($activeWebSilo, $activeCountry, $activePlant) {
                return $query->isVisible()->hasVisibleChildren($activeWebSilo, $activeCountry, $activePlant);
            })
        ;
    }

    public function scopeIsVisible(Builder $query) {
        return $query
            ->where('WebGroups.is_visible', '1')
            ->where(function ($where) {
                $where
                    ->whereDoesntHave('webSiloOverride')
                    ->orWhereHas('webSiloOverride', function ($query) {
                        return $query->where('WebSiloOverrides.is_visible', '=', 1);
                    })
                ;
            })
        ;
    }

    public function scopeHasVisibleParents(Builder $query, $activeWebSilo = false) {
        return $query
            ->whereHas('webHierarchies', function ($query) use ($activeWebSilo) {
               return $query->hasVisibleParents($activeWebSilo);
            })
        ;
    }

	public function scopeVisible(Builder $query) {
		$query
			->where('WebGroups.is_visible', '1')
            ->has('webHierarchies')
		;
//
//        if (b2b()->activeCountry() != 'US') {
//            $query->where('slug', '!=', 'pharmaceuticals');
//        }

        return $query;
	}


    public function scopeWebVisible(Builder $query) {
        return $query
            ->where('WebGroups.is_visible', '1')
            ->where(function ($where) {
                $where
                    ->whereDoesntHave('webSiloOverride')
                    ->orWhereHas('webSiloOverride', function ($query) {
                        return $query->where('WebSiloOverrides.is_visible', '=', 1);
                    })
                ;
            })
            ->whereHas('webCategories', function ($webCategoriesQuery) {
                return $webCategoriesQuery->webVisible();
            })
            ->with('webHierarchies')
        ;
    }

    public function scopeWebSiloApproved(Builder $query) {
        return $query
            ->whereHas('webCategories', function ($webCategoriesQuery) {
                return $webCategoriesQuery->webSiloApproved();
            })
            ;
    }

	public function scopeInSilo($query, $webSiloId) {
        return $query->whereHas('webSilos', function ($query) use ($webSiloId) {
            $query->where('websilo_id', $webSiloId);
        });
    }

    public function getHierarchyPathsAttribute() {
        $paths = [];
        foreach ($this->webHierarchies as $webHierarchy) {
            $paths[] = [
                'webHierarchy' => $webHierarchy,
            ];
        }
        return $paths;
    }

    public function slugUrl($approvedOnly = false) {
	    $params = [$this->slug];
	    if ((request()->get('ap', false) == 1) || $approvedOnly) $params['ap'] = 1;
        return route('group.slug', $params);
    }

    public function getParentRelationName() {
        return false;
    }

    public function getPrimaryImage() {
	    $asset = array_get($this, 'webSiloOverride.primaryImage', null);
        if (!is_null($asset)) return $asset;

        $asset = $this->assetsByType('primary')->first();
        if (!is_null($asset)) return $asset;

        $webCategory = $this->visibleWebCategories->first();
        if ($webCategory) return $webCategory->getPrimaryImage();
    }

    use HasAssets, HasSlug, HasLandingPage, RevisionableTrait, ManyToManyRevisionableTrait, HasWebSiloOverrides, HasCommitSequence, HasModelTranslations;
}
