<?php

namespace Hilco\Models;

use Carbon\Carbon;
use DB;
use Auth;
use Hilco\Facades\HilcoAssets;
use HilcoAPI\BODIngesters\TestModels\LegacyParts;
use HilcoB2B\M3Request;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\App;
use Log;
use Venturecraft\Revisionable\RevisionableTrait;

/**
 * Class WebPart
 * @package Hilco\Models
 * @property int $id
 * @property string $name
 * @property string $part_detail
 * @property string $safe_name
 * @property string $override_name
 * @property-read Part $part
 * @property-read WebPartFamily $family
 * @property-read WebPartAttributeValue[] $attributeValues
 * @method static Builder latest()
 * @property string $part_number
 * @property integer $part_id
 * @property integer $webpartfamily_id
 * @property string $date_created
 * @property string $date_modified
 * @property string $date_uploaded
 * @property string $deleted_at
 * @property-read \Hilco\Models\WebPartFamily $webPartFamily
 * @property-write mixed $web_part_attributes
 * @property-read mixed $part_name
 * @property-read mixed $family_name
 * @property-read mixed $context
 * @property-read mixed $default_context
 * @mixin \Eloquent
 * @property integer $webfamily_id
 * @method static WebPart firstOrNew(array $attributes)
 */
class WebPart extends WebModel
{
    protected $table = "WebParts";
    protected $fillable = [
        'min_quantity', 'quantity_step', 'quantity_presets',
        'name', 'webAttributes', 'webfamily_id', 'part_id', 'part_number',
        'keywords', 'is_visible', 'is_family_image',
        'in_stock_label','low_stock_label', 'out_of_stock_label'
    ];
    protected $casts = [
        ['is_visible' => 'boolean' , 'is_family_image' => 'boolean']
    ];

    public static $perEnvironment = true;

    public function indexOnly() {
        if (!array_get($this, 'is_visible', false)) return false;
        if (!array_get($this, 'webFamily.is_visible', false)) return false;

        return true;
    }

    public function getSolrRecord($webSilos) {
        $record = array_only($this->toArray(), [
            'id',
            'part_id',
            'name',
            'part_number',
            'webFamily',
            'legacy_part_no',
        ]);
        $record['nodash_part_number'] = str_replace('-', '', $this->part_number);
        $record['legacy_part_no'] = $this->part->legacy_part_no;
        $record['nodash_legacy_part_no'] = str_replace('-', '', $this->part->legacy_part_no);
        $record['webFamily_slug'] = !is_null($this->webFamily) ? $this->webFamily->slug : null;

        foreach ($record as $key => $item) {
            $record[$key."_searchable"] = preg_replace("/[^A-Za-z0-9]/", '', $item);
        }

        $record['webSiloIds'] = [];
        if ($webSilos === null) {
            $webSilos = WebSilo::visible()->get();
        }

        foreach ($webSilos as $webSilo) {
            if ($this->visibleInSilo($webSilo)) $record['webSiloIds'][] = $webSilo->id;
        }

        $record['webFamily_slug'] = !is_null($this->webFamily) ? $this->webFamily->slug : null;
        $record['webCollectionIds'] = !is_null($this->webFamily) ? $this->webFamily->webCollections->pluck('id')->toArray() : [];
        $record['is_rx'] = $this->part->is_rx;
        $record['validPlants'] = !is_null($this->part->inventoryItems) ? $this->part->inventoryItems->pluck('plant')->toArray() : [];
        $record['b2b_web_visible'] = $this->is_solr_visible;

        $record['assortments'] = [];
        foreach ($this->part->assortments as $assortment) {
            $record['assortments'][] = $assortment->assortment_name;
        }
        if (count($record['assortments']) == 0) {
            $record['assortments'][] = 'NO_ASSORTMENT';
        }

        return $record;
    }

    public function solrTable()
    {
        return $this->hasMany(WebPart_Solr::class, 'webPart_id', 'id');
    }

    public function getSolrClassAttribute() {
        return WebPart_Solr::class;
    }

    public function getSolrIDFieldAttribute() {
        return 'webPart_id';
    }

    public function getIsSolrVisibleAttribute() {
        return static::solrVisible()->where('id', $this->id)->count() > 0;
    }

    public function scopeSolrVisible(Builder $query) {
        return
            $query
                ->isVisible()
                ->hasSolrVisibleParents()
                ->hasSolrVisibleChildren()
            ;
    }

    public function scopeHasSolrVisibleParents(Builder $query) {
        return
            $query
                ->whereHas('webFamily', function ($query) {
                    return $query->hasSolrVisibleParents();
                })
            ;
    }

    public function scopeHasSolrVisibleChildren(Builder $query) {
        return $query
            ->whereHas('part', function ($query) {
                return $query->hasSolrVisibleChildren();
            })
            ;
    }

    public function scopeForLegacyPartNumber(Builder $query, $legacyPartNumber) {
        return $query
            ->whereHas('part', function ($query) use ($legacyPartNumber){
                return $query->forLegacyPartNumber($legacyPartNumber);
            });
    }
// *********************************************************************************************************************

// ***************************************ELASTICSEARCH METHODS (PROBABLY UNUSED)***************************************
    protected $indexSettings = [
        'analysis' => [
            'char_filter' => [
                'replace' => [
                    'type' => 'mapping',
                    'mappings' => [
                        '&=> and '
                    ],
                ],
            ],
            'filter' => [
                'word_delimiter' => [
                    'type' => 'word_delimiter',
                    'split_on_numerics' => false,
                    'split_on_case_change' => false,
                    'generate_word_parts' => true,
                    'generate_number_parts' => true,
                    'catenate_all' => true,
                    'preserve_original' => true,
                    'catenate_numbers' => true,
                ]
            ],
            'analyzer' => [
                'default' => [
                    'type' => 'custom',
                    'char_filter' => [
                        'html_strip',
                        'replace',
                    ],
                    'tokenizer' => 'whitespace',
                    'filter' => [
                        'lowercase',
                        'word_delimiter',
                    ],
                ],
            ],
        ],
    ];

    protected $mappingProperties = [
        'title' => [
            'type' => 'string',
            'analyzer' => 'standard'
        ]
    ];

    public static $elasticFuzziness = 1;

    /**
     * @description return elastic query with fuzziness
     */
    public static function buildElasticQuery($keyWords){
        $params=[
            'multi_match' => [
                'query'     => $keyWords,
                'fuzziness' => self::$elasticFuzziness,
                'fields'    => ['_all']
            ]
        ];
        return $params;
    }

//	use KeyWordsFunctional;

	public static function clearIsFamilyImage($familyId, $notIn=[]){
		self::where('webfamily_id', $familyId)->where('is_family_image',1)->whereNotIn('id',$notIn)->update(['is_family_image'=>0]);
	}

    /**
     * @description index all the entries in an Eloquent model (if needed)
     */
    public static function elasticIndexingIfNeeded(){
        $mappingExists = self::mappingExists();
        if (!$mappingExists) {
            self::createIndex();
            self::putMapping();
            self::addAllToIndex();
        }
	}
// *********************************************************************************************************************

// ***************************************RELATIONSHIPS*****************************************************************
	public function part() {
		return $this->belongsTo(Part::class, 'part_id', 'id');
	}

	/**
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function webFamily() {
		return $this->belongsTo(WebFamily::class, 'webfamily_id', 'id');
	}

    public function legacyPart(){
	    return $this->hasOne(LegacyPart::class, 'part_id', 'id');
    }

    public function webAttributes() {
        return $this->belongsToMany(WebAttribute::class, 'WebAttribute_WebPart', 'webpart_id', 'webattribute_id')
            ->withPivot(['id','attribute_value']);
    }

    public function relatedParent() {
        return $this->morphToMany(WebFamily::class, 'related', 'WebFamily_Related', 'related_id', 'webfamily_id')->withPivot(['related_category']);
    }

    public function customerPrices() {
        return $this->part->customerPrices();
    }

    public function hierarchyPaths() {
        $paths = ['webPart' => $this];
        if ($this->webFamily) $paths = $this->webFamily->hierarchyPaths($paths);
        return $paths;
    }

    public function partAsset() {
        return $this->hasOne(WebAsset::class, 'name', 'part_number')->where('asset_category', 'part')->orderBy('date_created', 'DESC');
    }
// *********************************************************************************************************************

// ***************************************GETTERS/SETTERS***************************************************************
    public function webAttributeValue($webAttributeId) {
        if (!$this->relationLoaded('webAttributes')) $this->load('webAttributes');
        $webAttributeValue = $this->webAttributes->find($webAttributeId);
        if (!is_null($webAttributeValue)) return $webAttributeValue->pivot->attribute_value;
        return null;
    }

    public function setWebAttributeValue($webAttributeId, $value) {
        $this->webAttributes()->sync([$webAttributeId => ['attribute_value' => $value]], false);
    }

    public function setWebAttributesAttribute($attributes) {
        $this->save();
        foreach ($attributes as $webAttributeId => $attributeValue) {
            $this->setWebAttributeValue($webAttributeId, $attributeValue);
        }
    }

    public function getPartNameAttribute() {
        $part = $this->part;
        if (is_null($part)) return null;
        return $part->part_desc;
    }

    public function getFamilyNameAttribute() {
        $family = $this->webFamily;
        if (is_null($family)) return null;
        return $family->name;
    }

    public function getNameAttribute($name)
    {
        $activeLanguage = b2b()->activeLanguage();
        if($activeLanguage != "en"){
            $translated = Translations_WebPart::where('language', '=', $activeLanguage)
                ->where('part_no', '=', $this->part_number)
                ->where('deleted_at', '=', '0')
                ->first();
            if(isset($translated)) {
                return $translated->name;
            }
        }
        return $name;
    }

    public function getDisplayNameForUACAttribute() {
        return $this->name;
    }

    public function getListPriceAttribute() {
        return $this->part->list_price;
    }

    public function getCustomerPriceAttribute() {
        return $this->part->customer_price;
    }

    public function getCustomerPrice($quantity = 1) {
        return $this->part->getCustomerPrice($quantity);
    }

    public function getCustomerDiscount() {
        return $this->part->getCustomerDiscount();
    }

    public function getImageLink($width = null, $height = null, $avoidCache = false) {
        return $this->primaryImage($width ? $width : false, $height ? $height : false);
    }

    public function getImageUrl($width = false, $height = false) {
        return HilcoAssets::part($this, $width, $height);
    }

    public function getIsRXAttribute() {
        $part = $this->part;
        if (is_null($part)) return null;
        return (!is_null($part->pharmacyFamilyExclusion) || !is_null($part->pharmacyPartExclusion));
    }

    public function getHasValidInventoryAttribute() {
        $status = 'none';
        $hasAll = true;
        if ($this->part) {
            foreach ($this->part->inventoryItems as $inventoryItem) {
                if ($inventoryItem->is_valid) {
                    $status = 'some';
                } else {
                    $hasAll = false;
                }
            }
        }

        if ($hasAll) $status = 'all';
        return $status;
    }

    public function getHierarchyPathsAttribute() {
        $isVisible = true;
        if (!$this->is_visible) $isVisible = trans('hilco::app.manually_hidden');
        if (is_null($this->part)) $isVisible = trans('hilco::app.not_in_point_man');
        if (is_null($this->webFamily)) $isVisible = trans('hilco::app.not_in_a_family');
        if ($this->has_valid_inventory == 'none') $isVisible = trans('hilco::app.no_valid_inventory');

        $paths = [];
        if (!is_null($this->webFamily)) {
            foreach ($this->webFamily->webCollections as $webCollection) {
                foreach ($webCollection->webCategories as $webCategory) {
                    foreach ($webCategory->webGroups as $webGroup) {
                        foreach ($webGroup->webHierarchies as $webHierarchy) {
                            if (!$this->webFamily->is_visible)
                                $isVisible = trans('hilco::app.parent_family_not_visible');
                            if (!$webCollection->is_visible)
                                $isVisible = trans('hilco::app.parent_collection_not_visible');
                            if (!$webCategory->is_visible)
                                $isVisible = trans('hilco::app.parent_category_not_visible');
                            if (!$webGroup->is_visible) $isVisible = trans('hilco::app.parent_group_not_visible');
                            $paths[] = [
                                'webHierarchy' => $webHierarchy,
                                'webGroup' => $webGroup,
                                'webCategory' => $webCategory,
                                'webCollection' => $webCollection,
                                'webFamily' => $this->webFamily,
                                'isVisible' => $isVisible,
                            ];
                        }
                    }
                }
            }
        }
        return $paths;
    }

    public function setWebfamilyIdAttribute($value) {
        if (!$value || $value < 1) $this->attributes['webfamily_id'] = null;
        else $this->attributes['webfamily_id'] = $value;
    }

    public function getInStockAttribute() {
        return $this->part->in_stock;
    }

    public function getInStockLabelAttribute() {
        $modelValue = array_get($this->attributes, 'in_stock_label');
        return strlen($modelValue) ? $modelValue : config('hilco.inStockLabels.0', 'In Stock');
    }

    public function getLowStockLabelAttribute() {
        $modelValue = array_get($this->attributes, 'low_stock_label');
        return strlen($modelValue) ? $modelValue : config('hilco.lowStockLabels.0', 'Low Stock');
    }

    public function getOutOfStockLabelAttribute() {
        $modelValue = array_get($this->attributes, 'out_of_stock_label');
        return strlen($modelValue) ? $modelValue : config('hilco.outOfStockLabels.0', 'Out of stock');
    }

    public function getStockStatusAttribute() {
        $atpData = [$this->part_number];
        $atps = [];
        try {
            $atps = WebPart::fetchAllItemsStockInfo($atpData);
        }catch(Exception $e){
            Log::error("Exception caught while fetching stock info: " . $e->getMessage());
            return Part::UNKNOWN;
        }

        return $atps[0]['status'];
    }

    public function getPartTaxability($linePrice){
        $client = null;
        $guzzle = new M3Request();
        if(empty($guzzle->client)){
            Log::error('Exception Caught while performing ' . __FUNCTION__ . ': Guzzle Http Client does not exist. Base URI may be missing!');
            return Part::UNKNOWN;
        }else{
            $client = $guzzle->client;
        }

        $shippingAddress = b2b()->activeShippingAddress();
        $queryAttributes = [
            'requestFrom' => 'b2b',
            'streetAddress1' => $shippingAddress->addr_1,
            'streetAddress2' => $shippingAddress->addr_2,
            'streetAddress3' => $shippingAddress->addr_3,
            'streetAddress4' => $shippingAddress->addr_4,
            'streetAddress5' => $shippingAddress->addr_5,
            'city' => $shippingAddress->city,
            'mainDivision' => $shippingAddress->state,
            'postalCode' => $shippingAddress->postal_cd,
            'country' => $shippingAddress->country,
            'part_number' => $this->part->part_no,
            'productClass' => $this->part->vertexid,
            'unit_price' => $linePrice
        ];

        $response = $client->get('v4/' . __FUNCTION__, ['query' => http_build_query($queryAttributes), 'headers' => ['Accept' => 'application/json']]);
        $taxResults = $response->getBody()->getContents();
        $taxResults = json_decode($taxResults, true);

        return $taxResults['taxes'];
    }

    public function getStockLabel($status){
        switch ($status) {
            case Part::IN_STOCK:
                return $this->in_stock_label;
            case Part::LOW_STOCK:
                return $this->low_stock_label;
            case Part::OUT_OF_STOCK:
                return $this->out_of_stock_label;
            case Part::DELAYED:
                return "Ships in 3-5 days";
            case Part::UNKNOWN:
                return ucfirst(Part::UNKNOWN);
        }
    }

    public function getStockLabelAttribute() {
        $status = $this->stock_status;
        switch ($status) {
            case Part::IN_STOCK:
                return $this->in_stock_label;
            case Part::OUT_OF_STOCK:
                return $this->out_of_stock_label;
            case Part::DELAYED:
                return "Ships in 3-5 days";
            case Part::UNKNOWN:
                return ucfirst(Part::UNKNOWN);
        }
    }

    public function getPrimaryImage() {
        $webAsset = $this->assetsByType('primary')->first();
        if (is_null($webAsset)) {
            $webAsset = $this->partAsset;
            if (is_null($webAsset) && isset($this->part)) {
                $webAsset = $this->part->legacyPartAsset;
            }
        }

        return $webAsset;
    }

    public function getIsSiloPartAttribute() {
        foreach (array_get($this, 'part.webSilos', []) as $partWebSilo) {
            if ($partWebSilo->id == b2b()->activeWebSilo()->id) return true;
        }
        return false;
    }

    public function getRequireApprovalAttribute() {
//        foreach (array_get($this, 'part.webSilos', []) as $partWebSilo) {
//            if ($partWebSilo->id == b2b()->activeWebSilo()->id) {
//                if (array_get($partWebSilo, 'pivot.require_approval', false)) return true;
//            }
//        }
//        return false;
        return $this->part->require_approval;
    }

    public function getIsWebVisibleAttribute() {
        return static::webVisible()->where('id', $this->id)->count() > 0;
    }
// *********************************************************************************************************************

// ***************************************SCOPES************************************************************************
    public function scopeJoinDiscount($query){
        $customerId = Auth::user()->customer_id;
        return $query->select(['WebParts.*',DB::raw('SUM(CustomerDiscounts.disc_val) as disc_val')])->leftJoin('CustomerDiscounts', function($query)use($customerId){
            return $query->on('CustomerDiscounts.part_id','=','WebParts.id')
                ->orOn('CustomerDiscounts.productfamily_id','=','WebParts.webfamily_id')
                ->where('CustomerDiscounts.customer_id','=',$customerId);
        })->groupBy('id');
    }

	public function scopeIsFamilyImage($query){
		$query->where('is_family_image',1)->visible()->whereHas('part', function ($query) {
            return $query->whereHas('inventoryItems', function ($query) {
                return $query->isValid();
            });
        });
	}

	public function scopePriceList($query, $priceList, $currency, $quantityLevel = 1) {
		return $query
			->leftJoin('PriceLists', 'PriceLists.part_no', '=', 'WebParts.part_number')
			->where('price_list', $priceList)
			->where('currency', $currency)
			->where('quantity_level', '>=', $quantityLevel)
			->select('WebParts.*', 'PriceLists.price')
		;
	}

	public function scopePriceRange($query, $min, $max) {
		return $query->whereHas('part', function($part) use ($min, $max){
			$part->whereBetween('list_price', [$min, $max]);
		});
	}

	public function scopeManagedBy($query, $manager) {
		return $query->whereHas('part', function ($query) use ($manager) {
			return $query->where('pf_prod_mgr', $manager);
		});
	}

	public function scopeInFamily($query, $webFamilyId) {
		return $query->whereHas('webFamily', function ($query) use ($webFamilyId) {
			return $query->where('webfamily_id', $webFamilyId);
		});
	}

    public function scopeInCollection($query, $webCollectionId) {
        return $query->whereHas('webFamily', function ($query) use ($webCollectionId) {
            return $query->inCollection($webCollectionId);
        });
    }

    public function scopeInCategory($query, $webCategoryId) {
        return $query->whereHas('webFamily', function ($query) use ($webCategoryId) {
            return $query->inCategory($webCategoryId);
        });
    }

	public function scopeInGroup($query, $webGroupId) {
		return $query->whereHas('webFamily', function ($query) use ($webGroupId) {
			return $query->inGroup($webGroupId);
		});
	}

	public function scopeHasVisibleChildren(Builder $query, $activeWebSilo = false, $activeCountry = false, $activePlant = false) {
	    return $query
            ->whereHas('part', function ($query) use ($activeWebSilo, $activeCountry, $activePlant) {
                return $query->visibleToActiveUser($activeWebSilo, $activeCountry, $activePlant);
            })
        ;
    }

    public function scopeHasVisibleParents(Builder $query, $activeWebSilo = false) {
        return $query
            ->whereHas('webFamily', function ($query) use ($activeWebSilo) {
                return $query->hasVisibleParents($activeWebSilo);
            })
            ;
    }

    public function scopeIsVisible(Builder $query) {
        return $query->where('WebParts.is_visible', '1');
    }

	public function scopeWebVisible(Builder $query) {
	    return $query
            ->isVisible()
            ->hasVisibleParents()
            ->hasVisibleChildren()
        ;
    }

    public function scopeWebSiloApproved(Builder $query) {
        return $query
            ->whereHas('part', function ($partQuery) {
                return $partQuery->webSiloApproved();
            })
        ;
    }

	public function scopeVisible(Builder $query, $webSiloId = false) {
	    if ($webSiloId === false) {
	        $webSiloId = b2b()->activeWebSilo()->id;
        }
		return $query
			->where('WebParts.is_visible', '1')
			->whereHas('part', function ($query) use ($webSiloId) {

                if (!b2b()->activeWebSilo()->is_default) {
                    $query->whereHas('webSilos', function ($query) {
                        return $query->where('websilo_id', '=', b2b()->activeWebSilo()->id);
                    });
                }

                if (!b2b()->activeWebSilo()->allow_private_label) {
//                    $query->doesntHave('assortments');
                    $query->where('private_lbl', '0');
                }

//				return $query->whereHas('inventoryItems', function ($query) use ($webSiloId) {
//                    $activeCountry = b2b()->activeCountry();
//                    if ($activeCountry == 'US') $query->inPlant('PLAINVILLE');
//                    return $query->inSilo($webSiloId);
//				});
			})
		;
	}
// *********************************************************************************************************************

    public function visibleInSilo($webSilo, $processDownChain = true, $processUpChain = true) {
        $inSilo = true;
        $isFamilyVisible = !$processUpChain;

        if (!$webSilo instanceof WebSilo) $webSilo = WebSilo::find($webSilo);

        if($processDownChain){
            if (isset($webSilo) && $webSilo->limit_parts) {
                $found = false;
                foreach (array_get($this->part, 'webSilos', []) as $partWebSilo) {
                    if (array_get($partWebSilo, 'id') == $webSilo->id) {
                        $found = true;
                    }
                }
                if (!$found) {
                    $inSilo = false;
                }
            }

            if (isset($webSilo) && !$webSilo->allow_private_label &&
                isset($this->part) && /*$this->part->assortments()->count()*/ $this->part->private_lbl > 0) {

                $inSilo = false;
            }
        }

        if($processUpChain){
            if ($this->webFamily->visibleInSilo($webSilo, false, true)){
                $isFamilyVisible = true;
            }
        }

        return ($inSilo && $isFamilyVisible);
    }


    public static function fetchAllItemsStockInfo($partNumbers, $warehouse = null, $customer = null) {
	    
	    if ($customer == null) {
	        $customer = b2b()->activeCustomer();
        }
        
        $stockInfoReturn = [];
        if (auth()->user()) {
            $guzzleClient = null;
            $guzzleM3Request = new M3Request();
            if (empty($guzzleM3Request->client)) {
                Log::error(
                    'Exception Caught while performing ' . __FUNCTION__ .
                    ': Guzzle Http Client does not exist. Base URI may be missing!');
                return Part::UNKNOWN;
            } else {
                $guzzleClient = $guzzleM3Request->client;
            }

            if (is_null($warehouse)) {
                $warehouse = $customer->activeSegment->def_ship_from;
            }
            
            $getStockInfoRequestData = [];
            foreach ($partNumbers as $partNumber) {
                $getStockInfoRequestData[$partNumber] = [
                    'item_number' => $partNumber,
                    'warehouse' => $warehouse
                ];
            }

            // Make GetItmWhsBal calls in chunks of 10
            $chunks = array_chunk($getStockInfoRequestData, 10, true);
            $getStockInfoResponseObj['atps'] = []; // we aren't doing ATP anymore, this is getItemWarehouseBalance
            foreach ($chunks as $chunk) {
                /*
                 * Expecting a chunk response that resembles:
                 * [
                 *  <part number> => [
                 *      apiVersion" => "v4",
                 *      status" => "success",
                 *      command" => "getItemWarehouseBalance",
                 *      totalTime" => 731,
                 *      details" => [
                 *          Program" => "MMS200MI",
                 *          Transaction" => "GetItmWhsBal",
                 *          Metadata" => null,
                 *          MIRecord" => [
                 *              0 => [
                 *                  NameValue => [
                 *                      0 => [....],
                 *                      1 => [
                 *                          Name => "AV01",
                 *                          Value => "<allocatable net quantity>"
                 *                      ],
                 *                      2 => [.....],
                 *                      3 => [.....],
                 *                      ....
                 *                  ],
                 *                  "RowIndex" => 0
                 *              ]
                 *          ]
                 *      ]
                 *  ],
                 *  <part number> => [....],
                 *  ....
                 * ]
                 *
                 * NOTE 1: AV01 appearing in index 1 in the NameValue array above is just an example
                 * NOTE 2: A kitted item (i.e., a part with item_type KIT and PartBOM rows) will have a details array
                 * that only has AV01:
                 * details => [
                 *      MIRecord => [
                 *          0 => [
                 *              "NameValue" => array:1 [
                 *                  0 => [
                 *                      Name => "AV01",
                 *                      Value => "<total # of possible kits that can be made based on the AV01 values of all its component parts>"
                 *                  ]
                 *              ]
                 *          ]
                 *      ]
                 * ]
                 */
                $chunkResponse = json_decode(
                    $guzzleClient
                        ->post(
                            'v4/getItemWarehouseBalance',
                            [
                                'form_params' => [
                                    'requestFrom' => 'b2b',
                                    'webuser' => auth()->user()->id,
                                    'customer_number' => b2b()->activeCustomer()->cust_no,
                                    'itemsATPRequestInfo' => $chunk
                                ],
                                'headers' => [
                                    'Accept' => 'application/json'
                                ]
                            ]
                        )
                        ->getBody()
                        ->getContents(),
                    true)['atps']
                ;

                //not doing array_merge here because we need to preserve our keys
                $getStockInfoResponseObj['atps'] = $getStockInfoResponseObj['atps'] + $chunkResponse;
            }

            // Process all responses
            // TODO: this logic should be moved entirely to the API
            foreach ($getStockInfoResponseObj['atps'] as $partNo => $stockInfoData) {
                if (isset($stockInfoData['details'])) {
                    $details = $stockInfoData['details'];

                    if (array_has($details, 'MIRecord') && array_has($details['MIRecord'][0], 'NameValue')) {

                        $nameValuePairs = $details['MIRecord'][0]['NameValue']; // TODO: or, if we do nothing else, how about only returning the AV01 NameValue pair?
                        foreach ($nameValuePairs as $nameValuePair) {
                            if ($nameValuePair['Name'] == 'AV01') {
                                $numStock = trim($nameValuePair['Value'], ' ');
                                $stockInfoReturn[$partNo]['stock'] = !empty($numStock) ? round($numStock, 0) : 0;
                                $stockInfoReturn[$partNo]['stockable'] = true; // the fact that we got any warehouse response at all == stockable

                                $part = Part::wherePartNo($partNo)->first();
                                $inventoryItem = $part->inventoryItems()->inPlant($warehouse)->first();

                                if (!empty($inventoryItem)) {
                                    if ($inventoryItem->part_stat == 'DoNotReorder') {
                                        $stockInfoReturn[$partNo]['status'] = Part::LOW_STOCK;
                                    } else if ($stockInfoReturn[$partNo]['stock'] > 0) {
                                        $stockInfoReturn[$partNo]['status'] = Part::IN_STOCK;
                                    } else {
                                        $stockInfoReturn[$partNo]['status'] = Part::OUT_OF_STOCK;
                                    }
                                } else {
                                    $stockInfoReturn[$partNo]['status'] = Part::UNKNOWN;
                                }
                            }
                        }
                    } else {
                        if (array_has($details, 'Message')) {
                            Log::debug(__FUNCTION__ . " received Message " . $details['Message'] . " for item $partNo.");
                            $stockInfoReturn[$partNo] = [
                                'status' => Part::UNKNOWN,
                                'stock' => 0,
                                'stockable' => false
                            ];
                        }
                    }

                } else {
                    Log::debug(__FUNCTION__ . " received empty or nonexistent 'details' field from API for item $partNo.");
                    $stockInfoReturn[$partNo] = [
                        'status' => Part::UNKNOWN,
                        'stock' => 0,
                        'stockable' => false
                    ];
                }
            }
        }

        return $stockInfoReturn;
    }

    public function getPrimaryImageAltText() {
        $webAsset = $this->getPrimaryImage();

        if (!isset($webAsset) || !isset($webAsset->pivot)) {
            return $this->name;
        }

        $overriddenAltText = $webAsset->pivot->alt_text;

        if ($overriddenAltText == null || $overriddenAltText == '') {
            return $this->name;
        } else {
            return $overriddenAltText;
        }
    }

    /*
     * Made this function only to find out that there was a simpler way to solve the problem with was designed to solve.
     * Necessary to ensure quick order and multi line quick order enforces Web Silo restrictions, but probably going to
     *  want to rotate that functionality for scope visibile functions.
     */
    public function isWebSiloAccessible($webSilo = null){
        if(is_null($webSilo)){
            $webSilo = b2b()->activeWebSilo();
        }
        if(!($webSilo->limit_parts)){
            return true;
        }
        $webSiloParts = $webSilo->getParts();
        $id = $this->part->id;
        $validWebPart = $webSiloParts->filter(function($item) use ($id){
                return $item->part_id === $id;
            })->first();
        if(!is_null($validWebPart)){
            return true;
        }
        return false;
    }

    public function getQuantityPresetsArrayAttribute() {
	    $presets = $this->quantity_presets;
	    if (strlen($presets) < 1) return [];

	    $array = explode(',', $presets);
	    return array_combine($array, $array);
    }

    public function slugUrl() {
        return route('family.slug', [$this->webFamily->slug, $this->id]);
    }


    use HasAssets, RevisionableTrait, HasCommitSequence, HasModelTranslations;
}
