<?php

namespace Hilco\Models;
use DB;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Support\Collection;
use Venturecraft\Revisionable\RevisionableTrait;

class WebCollection extends WebModel implements HasLandingPageInterface {
    protected $table = "WebCollections";
    protected $fillable = ['name', 'slug', 'webCategoryIds', 'is_visible', 'detail', 'sort_priority'];
    protected $context_attributes = ['weblayout_id'];
    protected $context_table = 'WebContext_WebCollection';
    protected $context_foreign_key = 'webcollection_id';
    protected $casts = [
        ['is_visible' => 'boolean']
    ];

    public function webCategories() {
        return $this->belongsToMany(WebCategory::class, 'WebCollection_WebCategory', 'webcollection_id', 'webcategory_id')->withPivot('display_order');
    }

    /**
     * @return BelongsToMany
     */
    public function webFamilies() {
        return $this->belongsToMany(WebFamily::class, 'WebFamily_WebCollection', 'webcollection_id', 'webfamily_id')->withPivot(['is_featured', 'display_order']);
    }

    public function visibleWebFamilies() {
        return $this->belongsToMany(WebFamily::class, 'WebFamily_WebCollection', 'webcollection_id', 'webfamily_id')
            ->withPivot(['is_featured', 'display_order'])
            ->visible()
            ->displayOrder()
        ;
    }

    public function scopeDisplayOrder(Builder $query) {
        return $query->orderBy('pivot_display_order', 'asc')->orderBy('name', 'asc');
    }

    public function scopeAlphabetical(Builder $query) {
        return $query->orderBy('name', 'asc');
    }

    public function featuredChildren() {
        return $this->webFamilies()->wherePivot('is_featured', 1);
    }

    public function scopeInCategory($query, $webCategoryId) {
        return $query->whereHas('webCategories', function ($query) use ($webCategoryId) {
            $query->where('webcategory_id', $webCategoryId);
        });
    }

    public function scopeHavingFamily($query, $webFamilyId) {
        return $query->whereHas('webFamilies', function ($query) use ($webFamilyId) {
            $query->where('webfamily_id', $webFamilyId);
        });
    }

    public function setWebCategoryIdsAttribute($ids) {
        $this->save();
        $this->webCategories()->sync($ids);
    }

//    public function webAttributeFilters() {
//        return $this->hasMany(WebAttributeFilter::class, 'webcategory_id', 'id');
//    }

//    public function familiesFeaturedInContext($contextId, $isFeatured = true) {
//        return $this->webFamilies()->whereHas('contexts', function ($query) use ($contextId, $isFeatured) {
//            return $query
//                ->where('webcontext_id', $contextId)
//                ->where('is_featured', $isFeatured)
//            ;
//        });
//    }

//    public function getFiltersArrayAttribute() {
//        $array = [];
//        $this
//            ->webAttributeFilters()
//            ->with('webAttribute', 'webAttributeFilterValues')
//            ->get()
//            ->each(function ($item) use (&$array) {
//                $array[$item->webattribute_id]['attribute_name'] = $item->webAttribute->attribute_name;
//                $array[$item->webattribute_id]['values'][$item->id] = $item->filter_display;
//            })
//        ;
//        return $array;
//    }
//
//    public function getWebLayoutAttribute() {
//        if (is_null($this->weblayout_id)) {
//            return $this->context->webLayout()->with('webLayoutComponents')->first();
//        } else {
//            return WebLayout::with('webLayoutComponents')->find($this->weblayout_id);
//        }
//    }
//
//    use SupportsDataTablesTrait;
//
//    public function createAttributeFilter($attributeId, $filterDisplay) {
//        $this->webAttributeFilters()->create(['webattribute_id' => $attributeId, 'filter_display' => $filterDisplay]);
//    }

    public function getHierarchyPathsAttribute() {
        $paths = [];
        foreach ($this->webCategories as $webCategory) {
            foreach ($webCategory->webGroups as $webGroup) {
                foreach ($webGroup->webSilos as $webSilo) {
                    $paths[] = [
                        'webSilo' => $webSilo,
                        'webGroup' => $webGroup,
                        'webCategory' => $webCategory,
                    ];
                }
            }
        }
        return $paths;
    }

    public function scopeInCollection($query, $webCollection) {
        return $query->whereHas('webCollections', function ($query) use ($webCollection) {
            $query->where('webcollection_id', $webCollection->id);
        });
    }

    public function scopeInGroup($query, $webGroupId) {
        return $query->whereHas('webCategories', function ($query) use ($webGroupId) {
            return $query->inGroup($webGroupId);
        });
    }

    public function getAssetPrefix() {
        return "webCollections/{$this->id}";
    }

    public function scopeVisible(Builder $query) {
        return $query
            ->where('WebCollections.is_visible', '1')
//            ->whereHas('webFamilies', function ($query) {
//                return $query->visible();
//            })
        ;
    }

    public function slugUrl() {
        return route('collection.slug', [$this->slug]);
    }

//    public function thumbnailUrl($avoidCache=false) {
//        $url = assetImage($this->getAssetPrefix() . '/featured');;
//        return ($avoidCache)?\ImgHelper::avoidCache($url):$url;
//    }

    public function getParentRelationName() {
        return 'webCategories';
    }

    public function getPrimaryImage() {
        $asset = $this->assetsByType('primary')->first();
        if (!is_null($asset)) return $asset;

        $webFamily = $this->visibleWebFamilies->first();
        if ($webFamily) return $webFamily->getPrimaryImage();
    }


    use HasAssets, HasSlug, HasLandingPage, RevisionableTrait;
}
