<?php

namespace Hilco\Models;
use DB;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Venturecraft\Revisionable\RevisionableTrait;
use Illuminate\Support\Facades\Input;

/**
 * Class WebCategory
 * @package Hilco\Models
 * @method static WebCategory firstOrNew(array $attributes)
 */
class WebCategory extends WebModel implements HasLandingPageInterface, HasWebSiloOverridesInterface {
    protected $table = "WebCategories";
    protected $fillable = ['name', 'slug', 'webGroupIds', 'is_visible', 'detail', 'sort_priority'];
    protected $casts = [
        ['is_visible' => 'boolean']
    ];
    protected $with = ['webSiloOverride'];
    protected $perPage = 24;
    protected $hierarchyParent = WebGroup::class;

//    use KeyWordsFunctional;

    public function getPerPage()
    {
        $tmpPerPage =Input::get('perPage');
        return (isset($tmpPerPage) && $tmpPerPage > 0)?$tmpPerPage:$this->perPage;
    }

    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function webGroups() {
        return $this->belongsToMany(WebGroup::class, 'WebCategory_WebGroup', 'webcategory_id', 'webgroup_id')->withPivot('display_order');
    }

    public function webCollections() {
        return $this->belongsToMany(WebCollection::class, 'WebCollection_WebCategory', 'webcategory_id', 'webcollection_id')->withPivot('display_order');
    }

    public function visibleWebCollections() {
        return $this->belongsToMany(WebCollection::class, 'WebCollection_WebCategory', 'webcategory_id', 'webcollection_id')
            ->withPivot('display_order')
            ->webVisible()
            ->displayOrder()
        ;
    }

    public function scopeDisplayOrder(Builder $query) {
        return $query->orderBy('pivot_display_order', 'asc')->orderBy('name', 'asc');
    }

    public function scopeAlphabetical(Builder $query) {
        return $query->orderBy('name', 'asc');
    }

    public function getDisplayNameForUACAttribute() {
        return $this->attributes['name'];
    }

    public function getNameAttribute() {
        if (config('hilco.ignoreActiveWebSilo')) {
            return $this->getTranslation('name', 'en', $this->attributes['name']);
        } else {
            return (isset($this->webSiloOverride) && $this->webSiloOverride->is_visible == 1) ?
                $this->webSiloOverride->name :
                $this->getTranslation('name');
        }
    }

    public function getNameForEditAttribute() {
        return $this->getTranslation('name', 'en', $this->attributes['name']);
    }

    public function getDetailAttribute() {
        if (config('hilco.ignoreActiveWebSilo')) {
            return $this->getTranslation('detail', 'en', $this->attributes['detail']);
        } else {
            return (isset($this->webSiloOverride) && $this->webSiloOverride->is_visible == 1) ?
                $this->webSiloOverride->detail :
                $this->getTranslation('detail')
                ;
        }
    }

    public function getFeaturedChildrenAttribute() {
        return $this->visibleWebCollections()->wherePivot('is_featured', 1)->get();
    }

    public function getChildrenAttribute() {
        $children = $this->visibleWebCollections();
        if (request()->get('ap', false) == 1) $children->webSiloApproved();

        return $children->get();
    }

    public function scopeInGroup($query, $groupId) {
        return $query->whereHas('webGroups', function ($query) use ($groupId) {
            $query->where('webgroup_id', $groupId);
        });
    }

    public function scopeInSilo($query, $webSiloId) {
        return $query->whereHas('webGroups', function ($query) use ($webSiloId) {
            return $query->inSilo($webSiloId);
        });
    }

    public function setWebGroupIdsAttribute($ids) {
        $this->save();
        $this->createRevisionRecord('webGroupIds', $this->webGroups->pluck('id')->toJson(), json_encode($ids));
        $this->webGroups()->sync($ids);
    }

    public function getHierarchyPathsAttribute() {
        $paths = [];
        foreach ($this->webGroups as $webGroup) {
            foreach ($webGroup->webHierarchies as $webHierarchy) {
                $paths[] = [
                    'webHierarchy' => $webHierarchy,
                    'webGroup' => $webGroup,
                ];
            }
        }
        return $paths;
    }

    public function scopeHasVisibleChildren(Builder $query, $activeWebSilo = false, $activeCountry = false, $activePlant = false) {
        return $query
            ->whereHas('webCollections', function ($query) use ($activeWebSilo, $activeCountry, $activePlant) {
                return $query->isVisible()->hasVisibleChildren($activeWebSilo, $activeCountry, $activePlant);
            })
        ;
    }

    public function scopeHasVisibleParents(Builder $query, $activeWebSilo = false) {
        return $query
            ->whereHas('webGroups', function ($query) use ($activeWebSilo) {
                return $query->hasVisibleParents($activeWebSilo);
            })
        ;
    }

    public function scopeIsVisible(Builder $query) {
        return $query
            ->where('is_visible', '1')
            ->where(function ($where) {
                $where
                    ->whereDoesntHave('webSiloOverride')
                    ->orWhereHas('webSiloOverride', function ($query) {
                        return $query->where('WebSiloOverrides.is_visible', '=', 1);
                    })
                ;
            })
        ;
    }

    public function scopeWebVisible(Builder $query) {
        return $query
            ->where('is_visible', '1')
            ->where(function ($where) {
                $where
                    ->whereDoesntHave('webSiloOverride')
                    ->orWhereHas('webSiloOverride', function ($query) {
                        return $query->where('WebSiloOverrides.is_visible', '=', 1);
                    })
                ;
            })
            ->whereHas('WebCollections', function ($webCollectionsQuery) {
                return $webCollectionsQuery->webVisible();
            })
            ;
    }

    public function scopeWebSiloApproved(Builder $query) {
        return $query
            ->whereHas('WebCollections', function ($webCollectionsQuery) {
                return $webCollectionsQuery->webSiloApproved();
            })
            ;
    }

    public function scopeVisible(Builder $query) {
        return $query
            ->where('WebCategories.is_visible', '1')
            ->whereHas('webGroups', function ($query) {
                return $query->visible();
            })
        ;
    }

    public function slugUrl() {
        return route('category.slug', [$this->slug,
            'g' => session()->get('breadcrumb.groupId') ? session()->get('breadcrumb.groupId') : $this->webGroups->first()->id,
            'c' => $this->id]);
    }
    public function getAssetPrefix() {
        return "webCategories/{$this->id}/webContexts";
    }

    public function getPrimaryImage() {
        $asset = array_get($this, 'webSiloOverride.primaryImage', null);
        if (!is_null($asset)) return $asset;

        $asset = $this->assetsByType('primary')->first();
        if (!is_null($asset)) return $asset;

        $webCollection = $this->visibleWebCollections->first();
        if ($webCollection) return $webCollection->getPrimaryImage();
    }

    public function getIsSolrVisibleAttribute() {
        return static::solrVisible()->where('id', $this->id)->count() > 0;
    }

    public function scopeSolrVisible(Builder $query) {
        return
            $query
                ->isVisible()
                ->hasSolrVisibleParents()
                ->hasSolrVisibleChildren();
    }

    public function scopeHasSolrVisibleChildren(Builder $query) {
        return $query
            ->whereHas('webCollections', function ($query) {
                return $query->hasSolrVisibleChildren();
            });
    }

    public function getAdvancedSolrRecord() {
        $record = [
            'id' => 'webCategory-' . $this->id,
            'web_model_id' => $this->id,
            'web_silo_ids' => [],
            'web_group_ids' => [],
            'name' => $this->sanitizeSolrValue($this->name),
            'slug' => $this->slug,
            'search_only' => $this->search_only,
            'sort_priority' => $this->sort_priority,
            'label' => $this->label,
            'web_model' => 'webCategory',
            'b2b_web_visible' => $this->is_solr_visible,
        ];

        WebSilo::visible()->each(function ($webSilo) use (&$record) {
            if ($this->visibleInSilo($webSilo)) $record['web_silo_ids'][] = $webSilo->id;
        });

        $this->webGroups->each(function ($webGroup) use (&$record) {
            $record['web_group_ids'][$webGroup->id] = $webGroup->id;
            $record['web_group_names'][$webGroup->id] = $this->sanitizeSolrValue($webGroup->name);
        });

//        $hasNonRx = false;
//        $this->webCollections->each(function ($webCollection) use (&$record, &$hasNonRx) {
//            $webCollection->webFamilies->each(function ($webFamily) use (&$record, &$hasNonRx) {
//                $webFamily->webParts->each(function ($webPart) use (&$record, &$hasNonRx) {
//                    if (!$hasNonRx && $webPart->part && !$webPart->part->is_rx) $hasNonRx = true;
//                });
//            });
//        });
//        $record['is_rx'] = !$hasNonRx;

        return $record;
    }


    use HasAssets, HasSlug, HasLandingPage, RevisionableTrait, ManyToManyRevisionableTrait, HasWebSiloOverrides, HasCommitSequence, HasModelTranslations, SolrTrait;

    public function getParentRelationName() {
        return 'webGroups';
    }

    public function scopeHasAlgoliaVisibleParents(Builder $query) {
        return $query
            ->whereHas('webGroups', function ($query) {
                return $query->hasAlgoliaVisibleParents();
            })
        ;
    }

    public function scopeHasSolrVisibleParents(Builder $query) {
        return $query
            ->whereHas('webGroups', function ($query) {
                return $query->hasSolrVisibleParents();
            })
            ;
    }

    public function visibleInSilo($webSilo, $processDownChain = true, $processUpChain = true) {
        $isCollectionVisible = !$processDownChain;
        $isGroupVisible = !$processUpChain;

        if (!$webSilo instanceof WebSilo) $webSilo = WebSilo::find($webSilo);

        if($processDownChain){
            foreach ($this->webCollections as $webCollection) {
                if ($webCollection->visibleInSilo($webSilo, true, false)){
                    $isCollectionVisible = true;
                    break;
                }
            }
        }

        if($processUpChain){
            foreach ($this->webGroups as $webGroup) {
                if ($webGroup->visibleInSilo($webSilo, false, true)){
                    $isGroupVisible = true;
                    break;
                }
            }
        }

        return ($isCollectionVisible && $isGroupVisible);
    }
}
