<?php

namespace Hilco\Models;
use DB;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Input;
use Venturecraft\Revisionable\RevisionableTrait;
use Storage;

/**
 * Class WebFamily
 * @package Hilco\Models
 * @method static WebFamily firstOrNew(array $attributes)
 */
class WebFamily extends WebModel
{
	protected $table = "WebFamilies";
	protected $fillable = ['name', 'slug', 'detail', 'webCollectionIds', 'webAttributeIds', 'is_visible', 'sort_priority', 'best_seller'];
	protected $context_attributes = ['is_latest', 'is_featured', 'is_visible'];
	protected $context_table = 'WebContext_WebFamily';
	protected $context_foreign_key = 'webfamily_id';
	protected $perPage = 24;
	protected $casts = [
		['is_visible' => 'boolean']
	];
    protected $hierarchyParent = WebCollection::class;

	public function getPerPage()
	{
		$tmpPerPage =Input::get('perPage');
		return (isset($tmpPerPage) && $tmpPerPage > 0)?$tmpPerPage:$this->perPage;
	}
	/**
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function webParts() {
		return $this->hasMany(WebPart::class, 'webfamily_id', 'id');
	}

	public function webDetailTabs() {
		return $this->belongsToMany(WebDetailTab::class, 'WebDetailTab_WebFamily', 'webfamily_id', 'webdetailtab_id')->withPivot('is_visible');
	}
	public function visibleWebDetailTabs() {
		return $this->webDetailTabs()->wherePivot('is_visible',1);
	}

	/**
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function visibleWebParts() {
		return $this
			->hasMany(WebPart::class, 'webfamily_id', 'id')
			->visible()
			->with(['part' => function ($query) {
			    $query->with(['inventoryItems' => function ($query) {
			        $query->inSilo();
                }]);
            }])
            ->with(['assets' => function ($query) {
                return $query->wherePivot('assetable_join_type', 'primary');
            }])
//			->with('images')
			->with(['webAttributes'])
			->orderBy('name');
	}

    public function scopeDisplayOrder(Builder $query) {
        return $query->orderBy('pivot_display_order', 'asc')->orderBy('name', 'asc');
    }

    public function scopeAlphabetical(Builder $query) {
        return $query->orderBy('name', 'asc');
    }

    /**
     * @return BelongsToMany
     */
	public function webCollections() {
		return $this->belongsToMany(WebCollection::class, 'WebFamily_WebCollection', 'webfamily_id', 'webcollection_id')->withPivot(['is_featured', 'display_order']);
	}

	/**
	 * @return BelongsToMany
     */
	public function webAttributes() {
		return $this
            ->belongsToMany(WebAttribute::class, 'WebAttribute_WebFamily', 'webfamily_id', 'webattribute_id')
            ->withPivot(['is_visible', 'display_order', 'is_thumbnail', 'is_filterable', 'is_select'])
            ->orderBy('pivot_display_order');
	}

	public function getWebAttributeIdsAttribute() {
		return $this->webAttributes->pluck('id')->toArray();
	}

    public function getWebCollectionIdsAttribute() {
        return $this->webCollections->pluck('id')->toArray();
    }

	public function setWebAttributeIdsAttribute($ids) {
		$this->save();
		$this->webAttributes()->sync($ids);
	}

	public function setWebCollectionIdsAttribute($ids) {
		$this->save();
		$this->webCollections()->sync($ids);
	}

//	public function getThumbnailsAttribute() {
//		$attributes = [];
//		foreach ($this->visibleWebParts as $webPart) {
//			foreach ($webPart->webAttributes as $partAttribute) {
//				if ($partAttribute->id == $this->thumbnail_attribute_id) {
//					$attributes[$partAttribute->pivot->attribute_value] = $webPart;
//				}
//			}
//		}
//		return $attributes;
//	}
//
	public function getPriceRangeAttribute() {
		if (is_null($this->visibleWebParts)) return null;
		$min = PHP_INT_MAX;
		$max = 0;
		foreach ($this->visibleWebParts as $webPart) {
			$min = min($min, $webPart->part->list_price);
			$max = max($max, $webPart->part->list_price);
		}
		return [$min, $max];
	}

	public function getPriceAttribute() {
	    try {
            if (is_null($this->visibleWebParts)) return null;
            return $this->visibleWebParts->first()->part->list_price;
        } catch (\Exception $e) {
            return null;
        }
	}
//

//	public function setWebCategoryIdsAttribute($value) {
//		$this->webCategories()->sync($value);
//	}


//	public function scopeLatest($query, $take = 8) {
//		return $query->whereHas('context', function($query) {
//			return $query->where('is_featured', true);
//		})->take($take);
//	}

//	public function scopeFeatured($query, $isFeatured = true) {
//		return $query->whereHas('context', function($query) use ($isFeatured) {
//			return $query->where('is_featured', $isFeatured);
//		});
//	}
	public function scopeHasAttributes($query, $filters){
		if(!isset($filters) || is_null($filters)) return $query;
		$attrs = [];
		foreach ($filters as $filter){
			if(!isset($attrs[$filter->webattribute_id])){
				$attrs[$filter->webattribute_id] = [];
			}
			$attrs[$filter->webattribute_id] = array_merge($attrs[$filter->webattribute_id],$filter->webAttributeFilterValues->pluck('filter_value')->toArray());
		}
		return $query->where(function ($query) use ($attrs) {
			return $query->whereHas('webParts', function ($query) use ($attrs) {
				foreach ($attrs as $id=>$values) {
					$query = $query->whereHas('webAttributes', function ($query) use ($id, $values) {
						return $query->where('webattribute_id', $id)->whereIn('attribute_value', $values);
					});
				}
				return $query;
			});
		});
	}
	public function scopeManagedBy($query, $product_manager) {
		return $query->whereHas('webParts', function($query) use ($product_manager) {
			return $query->managedBy($product_manager);
		});
	}

//	public function getOwnImage($force = false, $prefix='auto'){
//		if($prefix == 'auto' && isset($this->pivot) && $this->pivot->is_featured == 1){
//			$prefix = 'featured';
//		}elseif ($prefix == 'auto' && isset($this->webCollections) && $this->webCollections->first()->pivot->is_featured == 1){
//			$prefix = 'featured';
//		}elseif ($prefix == 'auto'){$prefix = '';}
//
//		$assetsDiskPath = ($prefix!="")?"web_families/$prefix/".$this->id:"web_families/".$this->id;
//		if($force || (($prefix!='featured' && $this->has_image > 0 ) || ($prefix =='featured' && $this->has_feature_image > 0 ))){
//			return Storage::disk('assets')->url($assetsDiskPath);
//		}
//		return false;
//	}


	public function scopeInCollection($query, $webCollection) {
		$webCollectionId = ($webCollection instanceof WebCollection) ? $webCollection->id : $webCollection;
		return $query->whereHas('webCollections', function ($query) use ($webCollectionId) {
			return $query->where('webcollection_id', $webCollectionId);
		});
	}

	public function scopeInCategory($query, $webCategory) {
		$webCategoryId = ($webCategory instanceof WebCategory) ? $webCategory->id : $webCategory;
		return $query->whereHas('webCollections', function ($query) use ($webCategoryId) {
			return $query->inCategory($webCategoryId);
		});
	}

	public function scopeInGroup($query, $webGroup) {
		$webGroupId = ($webGroup instanceof WebGroup) ? $webGroup->id : $webGroup;
		return $query->whereHas('webCollections', function ($query) use ($webGroupId) {
			return $query->inGroup($webGroupId);
		});
	}

	public static function in($model) {
		if ($model instanceof WebGroup) {return self::inGroup($model);}
		else if ($model instanceof WebCategory) return self::inCategory($model);
        else if ($model instanceof WebCollection) return self::inCollection($model);
        else return self::newQuery();
	}
//
	public function scopeVisible(Builder $query) {
//		return $query->availableForUser()
        return $query
			->where('WebFamilies.is_visible', '1')
			->whereHas('webParts', function ($query) {
				return $query->visible();
			})
		;
	}

	public function scopeForDisplay(Builder $query, $slug) {
		return $query
			->visible()
			->with('visibleWebParts')
            ->with('webAttributes')
			->slug($slug)
		;
	}

	public function scopeForBreadcrumbs($query) {
        $query->with([
            'webCollections' => function ($lQuery) {
                if (session('breadcrumb.collectionId')) {
                    $lQuery->where('WebCollections.id', '=', session('breadcrumb.collectionId'));
                }
                $lQuery->with(['webCategories' => function ($cQuery) {
                    if (session('breadcrumb.categoryId')) {
                        $cQuery->where('WebCategories.id', '=', session('breadcrumb.categoryId'));
                    }
                    $cQuery->with(['webGroups' => function ($gQuery) {
                        if (session('breadcrumb.groupId')) {
                            $gQuery->where('WebGroups.id', '=', session('breadcrumb.groupId'));
                        }
                    }]);
                }]);
            }
        ]);
        return $query;
    }

	public function scopeForLandingPage($query, HasLandingPageInterface $lpModel) {
	    $query
            ->visible()
            ->join('WebFamily_WebCollection as WFWC', 'WFWC.webfamily_id', '=', 'WebFamilies.id')
            ->join('WebCollection_WebCategory as WLWC', 'WLWC.webcollection_id', '=', 'WFWC.webcollection_id')
            ->join('WebCategory_WebGroup as WCWG', 'WCWG.webcategory_id', '=', 'WLWC.webcategory_id')
            ->join('WebGroup_WebSilo as WGWS', 'WGWS.webgroup_id', '=', 'WCWG.webgroup_id')
            ->join('WebGroups', 'WebGroups.id', '=', 'WGWS.webgroup_id')
            ->join('WebCategories', 'WebCategories.id', '=', 'WCWG.webcategory_id')
            ->join('WebCollections', 'WebCollections.id', '=', 'WLWC.webcollection_id')
            ->orderBy('WebFamilies.sort_priority', 'desc')
            ->orderBy('WGWS.display_order', 'asc')
            ->orderBy('WebGroups.name', 'asc')
            ->orderBy('WCWG.display_order', 'asc')
            ->orderBy('WebCategories.name', 'asc')
            ->orderBy('WLWC.display_order', 'asc')
            ->orderBy('WebCollections.name', 'asc')
            ->orderBy('WFWC.display_order', 'asc')
            ->orderBy('WebFamilies.name', 'asc')
            ->select(DB::raw('WebFamilies.*'))
            ->with(['visibleWebParts' => function ($query) {
                $query->with('assets');
            }, 'assets'])
            ->groupBy('WebFamilies.id')
        ;
        if ($lpModel instanceof WebGroup) {
            $query->where('WGWS.webgroup_id', $lpModel->id);
        } else if ($lpModel instanceof WebCategory) {
            $query->where('WCWG.webcategory_id', $lpModel->id);
        } else if ($lpModel instanceof WebCollection) {
            $query->where('WLWC.webcollection_id', $lpModel->id);
        }
        return $query;
    }

	public static function getImageUrl($partNumber, $width = null, $height = null) {
		$url = config('hilco.images.url');
		if(!isset($url)) $url = config('hilco-b2b.images.url');
		$url .= '/' . $partNumber;
		if ($width) $url .= "/$width";
		if ($height) $url .= "/$height";
		return $url;
	}

	public function scopeJoinWebPart(Builder $query){
		return $query->leftJoin('WebParts', 'WebParts.webfamily_id', '=', 'WebFamilies.id');
	}

    public function slugUrl() {
        return route('family.slug', [$this->slug]);
    }

//	public function getDetailsAttribute() {
//		$familyAttributes = [];
//		$parts = new Collection;
//		$thumbnails = new Collection;
//		$thumbnailAttribute = false;
//		foreach ($this->webAttributes as $familyAttribute) {
//			$familyAttributes[$familyAttribute->id] = $familyAttribute->attribute_name;
//			if (!$thumbnailAttribute && $familyAttribute->is_thumbnail) $thumbnailAttribute = $familyAttribute->id;
//		}
//		foreach ($this->visibleWebParts as $webPart) {
//			$partArray = ['webPart' => $webPart];
//
//			foreach ($familyAttributes as $id => $attribute_name) {
//				$partArray['webPartAttributes'][$attribute_name] = $webPart->webAttributeValue($id);
//				if ($id == $thumbnailAttribute) {
//					$thumbnails->put($webPart->webAttributeValue($id), $webPart);
//				}
//			}
//			$parts->put($webPart->id, $partArray);
//		}
//		return ['thumbnails' => $thumbnails, 'parts' => $parts];
//	}

//
//	public function getImageLink($width = null, $height = null, $avoidCache=false) {
//		$part = $this->visibleWebParts->where('is_family_image',1)->first();
//		if(is_null($part) || $part->images->count() == 0){
//			$ownUrl = $this->getOwnImage(false,'auto');
//			if($ownUrl) return ($avoidCache)?\ImgHelper::avoidCache($ownUrl):$ownUrl;
//			$part = $this->visibleWebParts->first();
//		}
//
//		if (is_null($part) || $part->images->count() == 0) {
//            $url = config('hilco.images.url');
//			if(!isset($url)) $url = config('hilco-b2b.images.url');
//            $url .= '/nopic';
//            if ($width) $url .= "/$width";
//            if ($height) $url .= "/$height";
//            return $url;
//        }
//
//		return $part->getImageLink($width, $height, $avoidCache);
//	}

    public function scopeBestSellers(Builder $query) {
        return $query->where('best_seller', 1);
    }

    public function getPrimaryImage() {
        $asset = $this->assetsByType('primary')->first();
        if (!is_null($asset)) return $asset;

        $part = $this->visibleWebParts->where('is_family_image', 1)->first();
        if (!is_null($part)) {
            $asset = $part->getPrimaryImage();
            if (!is_null($asset)) return $asset;
        }

        $part = $this->visibleWebParts->first();
        if (!is_null($part)) {
            $asset = $part->getPrimaryImage();
            if (!is_null($asset)) return $asset;
        }

        return null;
    }

	use HasAssets, HasSlug, RevisionableTrait;
}
