<?php

namespace Hilco\Models;
use DB;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Venturecraft\Revisionable\RevisionableTrait;
use Illuminate\Support\Facades\Input;

/**
 * Class WebCategory
 * @package Hilco\Models
 * @method static WebCategory firstOrNew(array $attributes)
 */
class WebCategory extends WebModel implements HasLandingPageInterface, HasWebSiloOverridesInterface {
    protected $table = "WebCategories";
    protected $fillable = ['name', 'slug', 'webGroupIds', 'is_visible', 'detail', 'sort_priority'];
    protected $casts = [
        ['is_visible' => 'boolean']
    ];
    protected $with = ['webSiloOverride'];
    protected $perPage = 24;
    protected $hierarchyParent = WebGroup::class;

//    use KeyWordsFunctional;

    public function getPerPage()
    {
        $tmpPerPage =Input::get('perPage');
        return (isset($tmpPerPage) && $tmpPerPage > 0)?$tmpPerPage:$this->perPage;
    }

    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function webGroups() {
        return $this->belongsToMany(WebGroup::class, 'WebCategory_WebGroup', 'webcategory_id', 'webgroup_id')->withPivot('display_order');
    }

    public function webCollections() {
        return $this->belongsToMany(WebCollection::class, 'WebCollection_WebCategory', 'webcategory_id', 'webcollection_id')->withPivot('display_order');
    }

    public function visibleWebCollections() {
        return $this->belongsToMany(WebCollection::class, 'WebCollection_WebCategory', 'webcategory_id', 'webcollection_id')
            ->withPivot('display_order')
            ->webVisible()
            ->displayOrder()
        ;
    }

    public function scopeDisplayOrder(Builder $query) {
        return $query->orderBy('pivot_display_order', 'asc')->orderBy('name', 'asc');
    }

    public function scopeAlphabetical(Builder $query) {
        return $query->orderBy('name', 'asc');
    }

    public function getNameAttribute() {
        return (isset($webSiloOverride) && $webSiloOverride->is_visible == 1) ?
            $this->webSiloOverride->name :
            $this->getTranslation('name')
            ;
    }

    public function getDisplayNameForUACAttribute() {
        return $this->name;
    }

    public function getDetailAttribute () {
        return (isset($webSiloOverride) && $webSiloOverride->is_visible == 1) ?
            $this->webSiloOverride->detail :
            $this->getTranslation('detail')
            ;
    }

    public function getNameWithoutOverrideAttribute() {
        return $this->getTranslation('name');
    }

    public function getDetailWithoutOverrideAttribute() {
        return $this->getTranslation('detail');
    }

    public function getFeaturedChildrenAttribute() {
        return $this->visibleWebCollections()->wherePivot('is_featured', 1)->get();
    }

    public function getChildrenAttribute() {
        $children = $this->visibleWebCollections();
        if (request()->get('ap', false) == 1) $children->webSiloApproved();

        return $children->get();
    }

    public function scopeInGroup($query, $groupId) {
        return $query->whereHas('webGroups', function ($query) use ($groupId) {
            $query->where('webgroup_id', $groupId);
        });
    }

    public function scopeInSilo($query, $webSiloId) {
        return $query->whereHas('webGroups', function ($query) use ($webSiloId) {
            return $query->inSilo($webSiloId);
        });
    }

    public function setWebGroupIdsAttribute($ids) {
        $this->save();
        $this->createRevisionRecord('webGroupIds', $this->webGroups->pluck('id')->toJson(), json_encode($ids));
        $this->webGroups()->sync($ids);
    }

    public function getHierarchyPathsAttribute() {
        $paths = [];
        foreach ($this->webGroups as $webGroup) {
            foreach ($webGroup->webHierarchies as $webHierarchy) {
                $paths[] = [
                    'webHierarchy' => $webHierarchy,
                    'webGroup' => $webGroup,
                ];
            }
        }
        return $paths;
    }

    public function scopeHasVisibleChildren(Builder $query, $activeWebSilo = false, $activeCountry = false, $activePlant = false) {
        return $query
            ->whereHas('webCollections', function ($query) use ($activeWebSilo, $activeCountry, $activePlant) {
                return $query->hasVisibleChildren($activeWebSilo, $activeCountry, $activePlant);
            })
        ;
    }

    public function scopeHasVisibleParents(Builder $query, $activeWebSilo = false) {
        return $query
            ->whereHas('webGroups', function ($query) use ($activeWebSilo) {
                return $query->hasVisibleParents($activeWebSilo);
            })
        ;
    }

    public function scopeIsVisible(Builder $query) {
        return $query
            ->where('is_visible', '1')
            ->where(function ($where) {
                $where
                    ->whereDoesntHave('webSiloOverride')
                    ->orWhereHas('webSiloOverride', function ($query) {
                        return $query->where('WebSiloOverrides.is_visible', '=', 1);
                    })
                ;
            })
        ;
    }

    public function scopeWebVisible(Builder $query) {
        return $query
            ->where('is_visible', '1')
            ->where(function ($where) {
                $where
                    ->whereDoesntHave('webSiloOverride')
                    ->orWhereHas('webSiloOverride', function ($query) {
                        return $query->where('WebSiloOverrides.is_visible', '=', 1);
                    })
                ;
            })
            ->whereHas('WebCollections', function ($webCollectionsQuery) {
                return $webCollectionsQuery->webVisible();
            })
            ;
    }

    public function scopeWebSiloApproved(Builder $query) {
        return $query
            ->whereHas('WebCollections', function ($webCollectionsQuery) {
                return $webCollectionsQuery->webSiloApproved();
            })
            ;
    }

    public function scopeVisible(Builder $query) {
        return $query
            ->where('WebCategories.is_visible', '1')
            ->whereHas('webGroups', function ($query) {
                return $query->visible();
            })
        ;
    }

    public function slugUrl() {
        return route('category.slug', [$this->slug,
            'g' => session()->get('breadcrumb.groupId') ? session()->get('breadcrumb.groupId') : $this->webGroups->first()->id,
            'c' => $this->id]);
    }
    public function getAssetPrefix() {
        return "webCategories/{$this->id}/webContexts";
    }

    public function getPrimaryImage() {
        $asset = array_get($this, 'webSiloOverride.primaryImage', null);
        if (!is_null($asset)) return $asset;

        $asset = $this->assetsByType('primary')->first();
        if (!is_null($asset)) return $asset;

        $webCollection = $this->visibleWebCollections->first();
        if ($webCollection) return $webCollection->getPrimaryImage();
    }


    use HasAssets, HasSlug, HasLandingPage, RevisionableTrait, ManyToManyRevisionableTrait, HasWebSiloOverrides, HasCommitSequence, HasModelTranslations;

    public function getParentRelationName() {
        return 'webGroups';
    }
}
