<?php

namespace Hilco\Models;

use Illuminate\Auth\Authenticatable;
use Illuminate\Auth\Passwords\CanResetPassword;
use Illuminate\Contracts\Auth\Access\Authorizable as AuthorizableContract;
use Illuminate\Contracts\Auth\Authenticatable as AuthenticatableContract;
use Illuminate\Contracts\Auth\CanResetPassword as CanResetPasswordContract;
use Illuminate\Foundation\Auth\Access\Authorizable;
use Venturecraft\Revisionable\RevisionableTrait;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

/**
 * Hilco\Models\WebUser
 *
 * @property integer $id
 * @property string $email
 * @property string $password
 * @property string $name
 * @property integer $customer_id
 * @property string $remember_token
 * @property string $authy_status
 * @property string $authy_id
 * @property string $date_created
 * @property string $date_modified
 * @property string $date_uploaded
 * @property string $deleted_at
 * @property string $last_login
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereId($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereEmail($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser wherePassword($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereName($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereCustomerId($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereRememberToken($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereDateCreated($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereDateModified($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereDateUploaded($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereDeletedAt($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereLastLogin($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereAuthyStatus($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereAuthyId($value)
 * @method static WebUser firstOrNew($attributes)
 * @mixin \Eloquent
 * @property-read \Hilco\Models\Customer $customer
 */
class WebUser extends WebModel implements AuthenticatableContract, AuthorizableContract, CanResetPasswordContract {
	use Authenticatable, Authorizable, CanResetPassword;

	protected $table = "WebUsers";
	protected $fillable = ['email', 'password', 'username', 'name', 'customer_id', 'is_enabled', 'webRoles'];
	protected $hidden = ['password', 'remember_token'];

	public function customer() {
		return $this->belongsTo(Customer::class, 'customer_id', 'id');
	}

	public function payment_profile()
	{
		return $this->hasOne(UserPaymentProfile::class, 'user_id', 'id');
	}

	public function webRoles()
    {
        return $this->belongsToMany(WebRole::class, 'WebRole_WebUser', 'webuser_id', 'webrole_id');
    }

    public function webPermissions() {
        return $this->belongsToMany(WebPermission::class, 'WebPermission_WebUser', 'webuser_id', 'webpermission_id');
    }

	public function can($permissionSlug, $arguments = []) {
        return $this->checkPermission($permissionSlug);
    }

    public function hasRole($roleSlug) {
        return $this->webRoles->where('slug', $roleSlug)->count() > 0;
    }

    public function hasDirectPermission($permissions) {
        if ($permissions instanceof WebPermission) $permissions = $permissions->slug;
        if (!is_array($permissions)) $permissions = [$permissions];
        return $this->webPermissions->whereIn('slug', $permissions)->count() > 0;
    }

    public function hasRolePermission($permissions) {
        if ($permissions instanceof WebPermission) $permissions = $permissions->slug;
        if (!is_array($permissions)) $permissions = [$permissions];
        return $this->webRoles->pluck('webPermissions')->flatten()->whereIn('slug', $permissions)->count() > 0;
    }

	protected function checkPermission($permissionSlug) {
        return $this->effective_permissions->where('slug', $permissionSlug)->count() > 0;
    }

	public function getEffectivePermissionsAttribute() {
	    $rolePemissions = $this->webRoles->pluck('webPermissions')->flatten()->keyBy('id');
        $userPermissions = $this->webPermissions->keyBy('id');
        return $rolePemissions->merge($userPermissions);
	}

	public function setPasswordAttribute($value) {
		$this->attributes['password'] = bcrypt($value);
	}

	public function userPhone() {
		$this->hasOne(CustomerPhone::class, 'customer_id', 'customer_id');
	}

	public function getActiveCustomerAttribute() {
	    if (session()->get('alias.customerId', false)) {
	        $customer = Customer::find(session()->get('alias.customerId'));
        } else {
            $customer = $this->customer;
        }
        return $customer;
    }

    public function hasActiveCustomer() {
        return !is_null($this->active_customer);
    }

    public function setWebRolesAttribute($webRoleIds) {
        $this->save();
        $this->webRoles()->sync($webRoleIds);
    }

    public function setWebRoleIdsAttribute($ids) {
        return $this->setWebRolesAttribute($ids);
    }

    public function getWebRoleIdsAttribute() {
        return $this->webRoles->pluck('id')->toArray();
    }

    public static function aliasAs($customerId = false) {
        if ($customerId) {
            session()->put('alias.customerId', $customerId);
        } else {
            session()->remove('alias.customerId', $customerId);
        }
    }

    public static function isAliased() {
        return (session()->get('alias.customerId', false));
    }

    use RevisionableTrait;
}
