<?php

namespace Hilco\Models;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use LogicException;

class PromotionTrigger extends WebModel {
    protected $table = 'PromotionTriggers';
    protected $fillable = ['sequence', 'trigger_type', 'commit_sequence'];

    use HasCommitSequence;

    // Overrides Model::boot(), used to define custom 'created' and 'deleted' states
    public static function boot() {
        parent::boot();
        // Create a row for the actual trigger before creating the polymorphic relationship row in PromotionTriggers
        static::created(function (PromotionTrigger $trigger) {
            $className = $trigger->getActualClassNameForMorph($trigger->trigger_type);
            $details = $className::create();
            $trigger->details()->associate($details);
            $trigger->save();
        });
        // Delete the row for the actual trigger when deleting the polymorphic relationship row in PromotionTriggers
        static::deleted(function (PromotionTrigger $trigger) {
            $details = $trigger->details;
            $details->delete();
        });
    }

    // 'Sub-category' of promotion triggers related to determining a customer's eligibility for the promotion
    protected static $eligibilityTriggerTypes = [
        'customerTrigger',
        'sourceTrigger'
    ];

    // 'Sub-category' of promotion triggers related to determining an order's qualification for the promotion
    protected static $qualificationTriggerTypes = [
        'orderQuantityTrigger',
        'orderValueTrigger'
    ];

    /**
     * Get the relationship to the Promotion object tied to this trigger
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function promotion() {
        return $this->belongsTo(Promotion::class, 'promotion_id', 'id');
    }

    /**
     * Get the relationship to the actual class type for the trigger
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function details() {
        return $this->belongsTo($this->getActualClassNameForMorph($this->trigger_type), 'trigger_id', 'id');
    }

    /**
     * Get list of distinct trigger_type values, called by PromotionTableCreator to be used in filters
     * @return \Illuminate\Database\Eloquent\Collection|PromotionTrigger[]
     */
    public static function selectList() {
        return self::select('trigger_type')->distinct()->get();
    }


    //Scopes

    /**
     * @param $query
     * @return mixed
     */
    public function scopeOnlyEligibility($query) {
        return
            $query
                ->whereIn('trigger_type', self::$eligibilityTriggerTypes)
            ;
    }

    /**
     * @param $query
     * @return mixed
     */
    public function scopeOnlyQualification($query) {
        return
            $query
                ->whereIn('trigger_type', self::$qualificationTriggerTypes)
            ;
    }

}
