<?php

namespace Hilco\M3Helper;

use GuzzleHttp\Client;
use Hilco\M3Helper\RequestModels\GetItmWhsBalRequestModel;
use Hilco\M3Helper\ResponseModels\ErrorResponseModel;
use Hilco\M3Helper\ResponseModels\GenericM3ResponseModel;
use Hilco\M3Helper\ResponseModels\MessageResponseModel;
use Hilco\M3Helper\ResponseModels\MIRecordResponseModel;
use Hilco\Models\Part;
use League\OAuth2\Client\Provider\Exception\IdentityProviderException;
use League\OAuth2\Client\Provider\GenericProvider;
use League\OAuth2\Client\Token\AccessToken;
use Log;
use UnexpectedValueException;

/**
 * Class M3Wrapper
 * @package Hilco\M3Helper
 */
class M3Wrapper {

    /**
     * @var Client
     */
    private $m3Client;

    /**
     * @var string
     */
    private $activeToken;

    /**
     * @var string
     */
    private $company;

    /**
     * @var M3RequestOptions
     */
    private $defaultOptions;

    const OAUTH_TOKEN_FILE = "oauth2/token.txt";

    /**
     * M3Wrapper constructor.
     * @param string $apiType
     * @throws \Exception
     */
    public function __construct($apiType = 'm3') {
        $host = config('m3api-oauth.host', null);
        $env = config('m3api-oauth.environment', null);
        $type = config("m3api-oauth.endpoints.$apiType", null);
        $company = config('m3api-oauth.cono', null);

        if (is_null($host) || is_null($env) || is_null($type) || is_null($company)) {
            throw new \Exception ("Missing required config(s), unable to create client object.");
        }

        $uri = "${host}/${env}${type}";
        $this->m3Client = new Client(['base_uri' => $uri]);
        $this->loadActiveToken();
        $this->company = $company;
    }

    /**
     * Sets activeToken (Bearer token for Authorization header) if it doesn't already exist or if it's expired
     * @throws \Exception
     */
    private function loadActiveToken() {
        $oauthToken = null;

        if (file_exists(storage_path(self::OAUTH_TOKEN_FILE))) {
            $oauthToken = new AccessToken(json_decode(file_get_contents(storage_path(self::OAUTH_TOKEN_FILE)), true));
        }

        if (is_null($oauthToken) || $oauthToken->getExpires() < time()) {
            $oauthToken = $this->getNewAccessToken();
        }

        $this->activeToken = $oauthToken;

        $this->updateDefaultOptions();
    }

    /**
     * Use M3/ION API client/user credentials to get a new access token
     * @throws \Exception
     */
    private function getNewAccessToken() {
        $m3ClientCredentials = config('m3api-oauth.clientCredentials', null);
        $m3UserCredentials = config('m3api-oauth.userCredentials', null);

        if (is_null($m3ClientCredentials) || is_null($m3UserCredentials)) {
            throw new \Exception ("Missing required credentials info, unable to authenticate with M3 API");
        }

        $providerObj = new GenericProvider($m3ClientCredentials);
        try {
            $oauthToken = $providerObj->getAccessToken('password', $m3UserCredentials);
            $tokenFile = fopen(storage_path(self::OAUTH_TOKEN_FILE), 'w');
            fwrite($tokenFile, json_encode($oauthToken));
            fclose($tokenFile);
        } catch (IdentityProviderException $ipe) {
            throw new \Exception (
                "IdentityProviderException while retrieving OAuth2 access token: " . $ipe->getMessage(),
                $ipe->getCode(),
                $ipe
            );
        } catch (UnexpectedValueException $uve) {
            throw new \Exception (
                "UnexpectedValueException while retrieving OAuth2 access token: " . $uve->getMessage(),
                $uve->getCode(),
                $uve
            );
        }
        return $oauthToken;
    }

    /**
     * Reset the default request options with the latest active access token
     */
    private function updateDefaultOptions() {
        $this->defaultOptions = new M3RequestOptions(
            [
                'Authorization' => 'Bearer ' . $this->activeToken,
                'Accept' => 'application/json'
            ]
        );
    }

    /**
     * Helper method, returns valid JSON object with cURL request error message
     * @param $errorNumber
     * @return string - valid JSON
     */
    private function createCurlErrorJSON ($errorNumber) {
        return <<<JSON
{
    "Error": "An error occurred with this cURL request, error code $errorNumber"
}
JSON;
    }

    /**
     * Get allocatable net quantity values per warehouse for an item
     * @param $itemNumber
     * @param $warehouseCodes
     * @param $isKit
     * @return array [
     *      {
     *          'warehouse' => <warehouse>,
     *          'av01'      => <allocatable net qty>
     *      },
     *      ...
     * ]
     */
    public function getAllocatableNetQuantity ($itemNumber, $warehouseCodes) {
        $returnObj = [];

        $item = Part::where('part_no', $itemNumber)->with('partBOMs')->first();

        if (isset($item) && count($warehouseCodes) > 0) {
            if ($item->item_type == 'KIT') {
                $kittedItemResponses = $this->getItmWhsBalKittedMultiWHLO($item->partBOMs, $warehouseCodes);
                foreach ($kittedItemResponses as $warehouse => $kittedItemResponseInfos) {
                    $av01 = "";
                    $allInStock = true;
                    $allStockable = true;
                    $numKitsPerComp = [];
                    foreach ($kittedItemResponseInfos as $compNo => $kittedItemResponseInfo) {
                        $responseModel = $kittedItemResponseInfo['response'];
                        $quantity = $kittedItemResponseInfo['quantity'];

                        switch (true) {
                            case $responseModel instanceof MIRecordResponseModel: {
                                $miRecord = $responseModel->getMIRecord();
                                $numStock = $miRecord->getValueForName('AV01');
                                if (empty($numStock) || $numStock < $quantity) {
                                    $allInStock = false;
                                    break;
                                } else {
                                    $numKitsPerComp[] = floor($numStock / $quantity);
                                }
                                break;
                            }
                            case $responseModel instanceof MessageResponseModel:
                            case $responseModel instanceof ErrorResponseModel:
                            default: {
                                $allStockable = false;
                            }
                        }
                    }

                    if ($allStockable) {
                        if ($allInStock) {
                            sort($numKitsPerComp);
                            $av01 = $numKitsPerComp[0];
                        } else {
                            $av01 = "0";
                        }
                    } else {
                        $av01 = "ERROR";
                    }

                    $returnObj[] = [
                        'warehouse' => $warehouse,
                        'av01' => $av01
                    ];
                }
            } else {
                $responses = $this->getItmWhsBalMultiWHLO($itemNumber, $warehouseCodes);
                foreach ($responses as $warehouse => $responseModel) {
                    $av01 = "";
                    switch (true) {
                        case $responseModel instanceof MIRecordResponseModel: {
                            $miRecord = $responseModel->getMIRecord();
                            $av01 = $miRecord->getValueForName('AV01');
                            break;
                        }
                        case $responseModel instanceof MessageResponseModel: {
                            $av01 = "0";
                            break;
                        }
                        case $responseModel instanceof ErrorResponseModel:
                        default: {
                            $av01 = "ERROR";
                        }
                    }

                    $returnObj[] = [
                        'warehouse' => $warehouse,
                        'av01' => $av01
                    ];
                }
            }
        }

        return $returnObj;
    }

    /**
     * Perform single GetItmWhsBal request for given item and warehouse
     * @param $itno
     * @param $whlo
     * @return GenericM3ResponseModel
     */
    private function getItmWhsBal ($itno, $whlo) {
        $getItmWhsBalModel = new GetItmWhsBalRequestModel($this->company, $itno, $whlo);
        $requestURL = $getItmWhsBalModel->getRequestURL();
        try {
            $this->loadActiveToken();
            $responseBody = $this->m3Client->get($requestURL, $this->defaultOptions->getOptionsForGuzzleRequest())->getBody();
        } catch (\Exception $e) {
            $responseBody = '{"Error": "' . $e->getMessage() . '"}';
        }

        return GenericM3ResponseModel::createResponseModel(json_decode($responseBody, true));
    }

    /**
     * Perform multiple asynchronous GetItmWhsBal requests for a list of items in a warehouse
     * @param $itnoArr
     * @param $whlo
     * @return array of items (keys) and GenericM3ResponseModels (values)
     */
    private function getItmWhsBalMultiITNO ($itnoArr, $whlo) {
        $asyncRequests = [];
        $curlHandles = [];
        $curlMulti = curl_multi_init();

        foreach ($itnoArr as $itno) {
            $getItmWhsBalModel = new GetItmWhsBalRequestModel($this->company, $itno, $whlo);
            $requestURL = $getItmWhsBalModel->getRequestURL();

            $curlHandle = curl_init($this->m3Client->getConfig('base_uri') . $requestURL);
            $curlHandles[] = $curlHandle;
            $asyncRequests[$itno] = [
                'curlHandle' => $curlHandle,
                'request' => $requestURL
            ];

            curl_setopt_array($curlHandle, $this->defaultOptions->getOptionsForCurlRequest());
            curl_multi_add_handle($curlMulti, $curlHandle);
        }

        $isCurlMultiStillRunning = null;
        do {
            curl_multi_exec($curlMulti, $isCurlMultiStillRunning);
        } while ($isCurlMultiStillRunning > 0);

        foreach ($curlHandles as $curlHandle) {
            curl_multi_remove_handle($curlMulti, $curlHandle);
        }
        curl_multi_close($curlMulti);

        $returnObj = [];

        foreach ($asyncRequests as $itno => $requestInfo) {
            $curlHandle = $requestInfo['curlHandle'];

            $errorNo = curl_errno($curlHandle);
            if ($errorNo) {
                $errorJSON = $this->createCurlErrorJSON($errorNo);
                $returnObj[$itno] = GenericM3ResponseModel::createResponseModel(json_decode($errorJSON, true));
            } else {
                $returnObj[$itno] = GenericM3ResponseModel::createResponseModel(json_decode(curl_multi_getcontent($curlHandle), true));
            }
        }

        return $returnObj;
    }

    /**
     * Perform multiple asynchronous GetItmWhsBal requests for a given item and a list of warehouses
     * @param $itno
     * @param $whloArr
     * @return array of warehouses (keys) and GenericM3ResponseModels (values)
     */
    private function getItmWhsBalMultiWHLO ($itno, $whloArr) {
        try {
            $this->loadActiveToken();
        } catch (\Exception $e) {
            $tokenError = '{"Error": "' . $e->getMessage() . '"}';
            foreach($whloArr as $whlo) {
                $errorResponse[$whlo] = GenericM3ResponseModel::createResponseModel($tokenError);
            }
            return $errorResponse;
        }

        $asyncRequests = [];
        $curlHandles = [];
        $curlMulti = curl_multi_init();

        foreach ($whloArr as $whlo) {
            $getItmWhsBalModel = new GetItmWhsBalRequestModel($this->company, $itno, $whlo);
            $requestURL = $getItmWhsBalModel->getRequestURL();

            $curlHandle = curl_init($this->m3Client->getConfig('base_uri') . $requestURL);
            $curlHandles[] = $curlHandle;
            $asyncRequests[$whlo] = [
                'curlHandle' => $curlHandle,
                'request' => $requestURL
            ];

            curl_setopt_array($curlHandle, $this->defaultOptions->getOptionsForCurlRequest());
            curl_multi_add_handle($curlMulti, $curlHandle);
        }

        $isCurlMultiStillRunning = null;
        do {
            curl_multi_exec($curlMulti, $isCurlMultiStillRunning);
        } while ($isCurlMultiStillRunning > 0);

        foreach ($curlHandles as $curlHandle) {
            curl_multi_remove_handle($curlMulti, $curlHandle);
        }
        curl_multi_close($curlMulti);

        $returnObj = [];

        foreach ($asyncRequests as $whlo => $requestInfo) {
            $curlHandle = $requestInfo['curlHandle'];

            $errorNo = curl_errno($curlHandle);
            if ($errorNo) {
                $errorJSON = $this->createCurlErrorJSON($errorNo);
                $returnObj[$whlo] = GenericM3ResponseModel::createResponseModel(json_decode($errorJSON, true));
            } else {
                $returnObj[$whlo] = GenericM3ResponseModel::createResponseModel(json_decode(curl_multi_getcontent($curlHandle), true));
            }
        }

        return $returnObj;
    }

    /**
     * @param $partBOMs
     * @param $whloArr
     * @return array of warehouses (keys) and associative arrays (values)
     * comprised of component item numbers (keys) and their GenericM3ResponseModels
     */
    private function getItmWhsBalKittedMultiWHLO ($partBOMs, $whloArr) {
        try {
            $this->loadActiveToken();
        } catch (\Exception $e) {
            $tokenError = '{"Error": "' . $e->getMessage() . '"}';
            foreach($whloArr as $whlo) {
                $errorResponse[$whlo] = GenericM3ResponseModel::createResponseModel($tokenError);
            }
            return $errorResponse;
        }

        $asyncRequests = [];
        $curlHandles = [];
        $curlMulti = curl_multi_init();

        foreach ($whloArr as $whlo) {
            foreach ($partBOMs as $partBOM) {
                $compNo = $partBOM->comp_no;
                $getItmWhsBalModel = new GetItmWhsBalRequestModel($this->company, $compNo, $whlo);
                $requestURL = $getItmWhsBalModel->getRequestURL();

                $curlHandle = curl_init($this->m3Client->getConfig('base_uri') . $requestURL);
                $curlHandles[] = $curlHandle;

                $asyncRequests[$whlo][$compNo] = [
                    'curlHandle' => $curlHandle,
                    'request' => $requestURL,
                    'quantity' => $partBOM->quantity,
                ];

                curl_setopt_array($curlHandle, $this->defaultOptions->getOptionsForCurlRequest());
                curl_multi_add_handle($curlMulti, $curlHandle);
            }
        }

        $isCurlMultiStillRunning = null;
        do {
            curl_multi_exec($curlMulti, $isCurlMultiStillRunning);
        } while ($isCurlMultiStillRunning > 0);

        foreach ($curlHandles as $curlHandle) {
            curl_multi_remove_handle($curlMulti, $curlHandle);
        }
        curl_multi_close($curlMulti);

        $returnObj = [];

        foreach ($asyncRequests as $whlo => $requestInfos) {
            foreach ($requestInfos as $compNo => $requestInfo) {
                $curlHandle = $requestInfo['curlHandle'];

                $errorNo = curl_errno($curlHandle); // cURL code 0 is considered 'success'
                if ($errorNo) {
                    $errorJSON = $this->createCurlErrorJSON($errorNo);
                    $returnObj[$whlo][$compNo] = [
                        'response' => GenericM3ResponseModel::createResponseModel(json_decode($errorJSON, true)),
                        'quantity' => $requestInfo['quantity']
                    ];
                } else {
                    $returnObj[$whlo][$compNo] = [
                        'response' => GenericM3ResponseModel::createResponseModel(json_decode(curl_multi_getcontent($curlHandle), true)),
                        'quantity' => $requestInfo['quantity']
                    ];
                }
            }
        }

        return $returnObj;
    }

}