<?php

namespace Hilco\Models;

use Auth;
use DB;
use Debugbar;
use Illuminate\Database\Eloquent\Model;

class WebCart extends WebModel
{
    protected $table = 'WebCarts';
    protected $fillable = ['quantity', 'comment'];

    public function webPart() {
        return $this->hasOne(WebPart::class, 'part_number', 'part_number');
    }

    public function webUser() {
        return $this->hasOne(WebUser::class, 'webuser_id', 'id');
    }

    public function priceList() {
//        return $this->hasOne(PriceList::class)
    }

    public function scopeForUser($query, $webuserId) {
        return $query->where('webuser_id', $webuserId);
    }

    public function scopePartNumber($query, $partNumber) {
        return $query->where('part_number', $partNumber);
    }

    public function getExtendedPriceAttribute() {
        return $this->webPart->getCustomerPrice($this->quantity) * $this->quantity;
    }

    public static function add(WebPart $webPart, $quantity = 1)
    {
        $instance = new static;
        $instance->webuser_id = Auth::id();
        $instance->part_number = $webPart->part_number;
        $instance->quantity = $quantity;
        $instance->save();
        return $instance;
    }

    public static function remove($userId, $partNumber)
    {
        $instance = new static;
        return $instance::where(['part_number' => $partNumber, 'webuser_id' => $userId])->delete();
    }

    /**
     * @param $webUser
     * @return array|WebCart[]|\Illuminate\Database\Eloquent\Builder[]|\Illuminate\Database\Eloquent\Collection
     */
    public static function getWebCartItems($webUser = null) {
        if (is_null($webUser)) $webUser = auth()->user();
        return
            self
            ::where('webuser_id', ($webUser->id))
            ->orderBy('part_number', 'asc')
            ->with(['webPart' => function ($query) {
                return $query->with(['part' => function ($query) {
                    return $query->with('inventoryItems')->with('productFamily.productCategory.productCategoryGroup');
                }]);
            }])
            ->get()
        ;
    }

    /* Calculates and keeps track of the overall status of the active user's cart, including all discounts and pricing */
    public static function cartStatus(WebSilo $webSilo = null, WebUser $webUser = null, Customer $soldToCustomer = null) {
        if (is_null($webSilo)) $webSilo = b2b()->activeWebSilo();
        if (is_null($webUser)) $webUser = auth()->user();
        if (is_null($soldToCustomer)) $soldToCustomer = b2b()->activeCustomer();

        if (is_null($webSilo) || is_null($webUser) || is_null($soldToCustomer)) return [];

        // Initialize 'status' object
        $status = [
            'itemsCount' => 0,
            'quantityCount' => 0,
            'baseTotal' => 0,
            'discountedTotal' => 0,
            'discounts' => [
                'rewards' => 0,
                'customer' => 0,
            ],
            'productCategoryTotals' => [],
            'rewardsUpgradeSpending' => 0,
            'rewardsTier' => rewards()->tier($soldToCustomer),
            'taxAmount' => 0,
            'shippingAmount' => 0,
            'shippingDiscountAmount' => 0,
            'shippingTaxAmount' => 0,
            'requireApproval' => false,
            'priceList' => ''
        ];

        $items = [];

        $customerPriceListName = $soldToCustomer->getDefaultPriceListAttribute();
        $currencyCode = $soldToCustomer->currency;
        $siloPriceListName = $webSilo->priceList;

        // get all current items in the web cart - [array of WebCart objects]
        $webCartItems = self::getWebCartItems($webUser);


//        // set up list of part ids to their total quantities on the order
//        $distinctPartQuantities = [];
//        foreach ($webCartItems as $webCartItem) array_increment($distinctPartQuantities, $webCartItem->webPart->part->id, $webCartItem->quantity);

        // get customer-based price list to use on the order
        $priceListNameToUse = self::getPriceListNameToUse($customerPriceListName, $webCartItems);
        $status['priceList'] = $priceListNameToUse;

        // match part ids to relevant info (base price, total qty, price list, is_contract)
        // From this we get two lists; one with the individual lines, one with the altogether parts.
        $distincts = self::getDistinctPartInfo($priceListNameToUse, $siloPriceListName, $currencyCode, $webCartItems);
        $distinctPartInfo = $distincts['distinctPartInfo'];
        $distinctLineInfo = $distincts['distinctLineInfo'];
//        $distinctPartInfo = self::getDistinctPartInfo($priceListNameToUse, $siloPriceListName, $currencyCode, $webCartItems);

//        // get all part prices based on $priceListNameToUse
//        $customerPrices = PriceList
//            ::where('price_list', '=', $priceListNameToUse)
//            ->where('currency', '=', $currencyCode)
//            ->whereIn('part_id', array_keys($distinctPartQuantities))
//            ->orderBy('quantity_level', 'DESC')
//            ->get();
//
//        // get all the part prices based on the default silo price list just in case
//        $siloPrices = PriceList
//            ::where('price_list', '=', $siloPriceListName)
//            ->where('currency', '=', $currencyCode)
//            ->whereIn('part_id', array_keys($distinctPartQuantities))
//            ->orderBy('quantity_level', 'DESC')
//            ->get();

        // get all "almost qualifying" promotions (used to conditionally show cart and checkout headers)
        $almostQualifyingPromotionsInfo = self::getAlmostQualifiedPromotionsInfo($soldToCustomer, $webCartItems, $distinctPartInfo);
        $status['almostQualifiedPromotionsInfo'] = $almostQualifyingPromotionsInfo;

        // get relevant discounts and other info from all triggered promotions
        $triggeredPromotionsInfo = self::getTriggeredPromotionsInfo($soldToCustomer, $webCartItems, $distinctPartInfo);

        // set individual promo info objects and lists for use later
        $orderHeaderDiscounts = $triggeredPromotionsInfo['orderHeaderDiscounts'];
        $triggeredPromotions = $triggeredPromotionsInfo['triggeredPromotions'];
        $discountNotes = $triggeredPromotionsInfo['discountNotes'];
        $freeItemActionsInfo = $triggeredPromotionsInfo['freeItemActionsInfo'];

        // get the current cheapest item in the cart that qualifies for a free-item promotion
//        $cheapestItemAndPromo = self::getCheapestWebCartItemsByProduct($freeItemActionsInfo, $webCartItems, $customerPrices, $siloPrices, $distinctPartQuantities, $priceListNameToUse, $soldToCustomer);
//        $cheapestItem = $cheapestItemAndPromo['cheapestItem'];
//        $freeItemPromoUsed = $cheapestItemAndPromo['promotionUsed'];
//        $hasAppliedFreeItem = false;

        // Using $webCartItems, prepare items[] array to be included in $status
//        $partInfo = [
//            'partId' => $partId,
//            'basePrice' => $partPriceInfo['basePrice'],
//            'isContractPrice' => $partPriceInfo['isContractPrice'],
//            'priceListUsed' => $partPriceInfo['priceListUsed'],
//            'totalQuantity' => $partQuantity
//        ];
        foreach ($webCartItems as $key=>$webCartItem) {
            $part = $webCartItem->webPart->part;

            $lineInfo = $distinctLineInfo[$key];

//            $partTotalQuantity = array_get($distinctPartQuantities, $part->id, 0);

            // Determine appropriate pricing for the part based on the price list(s)
//            $partInfo = array_get($distinctPartInfo, $part->id);//self::getPartPriceInfo($customerPrices, $siloPrices, $part, $partTotalQuantity, $priceListNameToUse, $siloPriceListName, $currencyCode);
            $partInfo = $lineInfo;
            $basePrice = $partInfo['basePrice'];
            $isContractPrice = $partInfo['isContractPrice'];
            $priceListUsed = $partInfo['priceListUsed'];

            $discountPercent = 0;
            $baseDiscountAmount = 0;
            $extendedDiscountAmount = 0;
            $discounts = [];
            $discountedPrice = $basePrice;
            $extendedBasePrice = $basePrice * $webCartItem->quantity;

            // As long as this is non-contract pricing, calculate all the applicable discounts for the part
            if (!$isContractPrice) {
                // if there are still free item actions to process..
                if (count($freeItemActionsInfo)) {
                    // loop thru each action's list of cheapest parts...
                    foreach ($freeItemActionsInfo as $key => $value) {
                        $freeItemAction = $value['freeItemAction'];
                        $cheapestPartList = $value['cheapestPartList'];
                        // if they still have parts to discount...
                        if (count($cheapestPartList)) {
                            foreach ($cheapestPartList as $partId => $numFree) {
                                // if this part is still on the list to discount...
                                if ($numFree > 0) {
                                    // and the current item matches...
                                    if ($part->id == $partId) {
                                        // if the number of times this item should be set to free is less than the current
                                        // qty on this cart line, then use that number, otherwise, just use the cart line qty
                                        $numToDiscount = ($numFree < $webCartItem->quantity) ? $numFree : $webCartItem->quantity;
                                        // calculate the actual discount percentage for the line to set that number of items as 'free'
                                        $freePercent = $numToDiscount * 100 * (1 / $webCartItem->quantity);
                                        // make sure the cart line's discount percentage is capped at 100%
                                        if ($discountPercent + $freePercent >= 100) {
                                            $freePercent = 100 - $discountPercent;
                                        }
                                        // if there is still a discount to be applied..
                                        if ($freePercent !== 0) {
                                            // keep track of current discount percentage
                                            $discountPercent += $freePercent;
                                            // increment the percent, amt, and extAmt values for the promo disc type
                                            $discountAmount = $basePrice * ($freePercent / 100);
                                            array_increment($discounts, "promotion.percent", $freePercent);
                                            array_increment($discounts, "promotion.baseAmount", $discountAmount);
                                            array_increment($discounts, "promotion.extendedAmount", $discountAmount * $webCartItem->quantity);
                                            array_increment($discounts, "promotion.isUsed", 1);
                                            // make sure the promo that had this free item action is updated to 'applied'
                                            $triggeredPromotions = self::updateAppliedPromotions($triggeredPromotions, $freeItemAction->promotion()->first());
                                            // decrement the number on the cheapest part list to keep track...
                                            $cheapestPartList[$partId] = $numFree - $numToDiscount;
                                            // ...and remove the item from the list once they've all been discounted
                                            if ($cheapestPartList[$partId] == 0) {
                                                unset($cheapestPartList[$partId]);
                                            }
                                            $freeItemActionsInfo[$key] = [
                                                'freeItemAction' => $freeItemAction,
                                                'cheapestPartList' => $cheapestPartList
                                            ];
                                        } else {
                                            // otherwise stop calculating for this free item actions part list, it's already at 100%
                                            break;
                                        }
                                    }
                                } else {
                                    unset($cheapestPartList[$partId]);
                                    $freeItemActionsInfo[$key] = [
                                        'freeItemAction' => $freeItemAction,
                                        'cheapestPartList' => $cheapestPartList
                                    ];
                                }
                            }
                        } else {
                            unset($freeItemActionsInfo[$key]);
                        }
                    }
                }

                if ($discountPercent < 100) {
                    $itemDiscounts = $part->getApplicableDiscounts($soldToCustomer, $triggeredPromotions);
                    $hasCustomerDiscounts = false;
                    $hasRewardsDiscounts = false;

                    foreach ($itemDiscounts as $discount) {
                        $dPercent = $discount['percent'];
                        // keep track of customer and rewards discounts (see if-check after for-loop)
                        if ($discount['type'] === 'customer') {
                            $hasCustomerDiscounts = true;
                        } else if ($discount['type'] === 'rewards') {
                            $hasRewardsDiscounts = true;
                        } else {
                            $discountPercent += $dPercent;
                        }

                        $dAmount = round($basePrice * ($dPercent / 100), 2);

                        array_increment($discounts, "$discount[type].percent", $dPercent);
                        array_increment($discounts, "$discount[type].baseAmount", $dAmount);
                        array_increment($discounts, "$discount[type].extendedAmount", $dAmount * $webCartItem->quantity);
                        array_increment($discounts, "$discount[type].isUsed", 1);
                    }

                    // if the item has both customer and rewards discounts, only add the greater of the two, not both
                    if ($hasCustomerDiscounts && $hasRewardsDiscounts) {
                        if ($discounts['customer']['percent'] > $discounts['rewards']['percent']) {
                            $discountPercent += $discounts['customer']['percent'];
                            $discounts['rewards']['isUsed'] = 0;
                        } else {
                            $discountPercent += $discounts['rewards']['percent'];
                            $discounts['customer']['isUsed'] = 0;
                        }
                    } else if ($hasCustomerDiscounts) {
                        $discountPercent += $discounts['customer']['percent'];
                    } else if ($hasRewardsDiscounts) {
                        $discountPercent += $discounts['rewards']['percent'];
                    }
                }


                if ($discountPercent >= 100) $discountPercent = 100;

                $baseDiscountAmount = round($basePrice * ($discountPercent/100), 2);
                $extendedDiscountAmount = round($extendedBasePrice * ($discountPercent/100), 2);
                $discountedPrice = $basePrice - $baseDiscountAmount;
            }

            $extendedPrice = $discountedPrice * $webCartItem->quantity;

            // If applicable, calculate the tax amount
            $itemTaxAmount = self::calculateTaxAmount($part, $extendedPrice);

            // If any part requires approval to be ordered, set the status of the cart to require approval
            $itemRequiresApproval = $webCartItem->webPart->requireApproval;
            if ($itemRequiresApproval) {
                $status['requireApproval'] = true;
            }

            // Add the part and its relevant details to the $items list
            $items[$webCartItem->id] = collect([
                'id' => $webCartItem->id,
                'webCart' => $webCartItem,
                'webPart' => $webCartItem->webPart,
                'part' => $webCartItem->webPart->part,
                'quantity' => $webCartItem->quantity,
                'basePrice' => $basePrice,
                'discountedPrice' => $discountedPrice,
                'baseDiscountAmount' => $baseDiscountAmount,
                'discountPercent' => $discountPercent,
                'extendedBasePrice' => $extendedBasePrice,
                'extendedPrice' => $extendedPrice,
                'extendedDiscountAmount' => $extendedDiscountAmount,
                'discounts' => $discounts,
                'isContractPrice' => $isContractPrice,
                'priceListUsed' => $priceListUsed,
                'taxAmount' => $itemTaxAmount,
                'requireApproval' => $itemRequiresApproval,
            ]);
            array_increment($status, 'itemsCount');
            array_increment($status, 'quantityCount', $webCartItem->quantity);
            array_increment($status, 'baseTotal', $extendedBasePrice);
            array_increment($status, 'taxAmount', $itemTaxAmount);

            foreach ($discounts as $discountType => $discountInfo) {
                if ($discountInfo['isUsed']) {
                    array_increment($status, "discounts.$discountType", $discountInfo['extendedAmount']);
                }
            }

            $productCategoryGroupId = $part->productFamily->productCategory->productCategoryGroup->id;
            array_increment($status, "productCategoryTotals.$productCategoryGroupId", $extendedPrice);
        }

        // set the ones that won't change and are necessary for other uses in $status
        $status['triggeredPromotions'] = $triggeredPromotions;
        $status['discountNotes'] = $discountNotes;
        $status['orderHeaderDiscounts'] = $orderHeaderDiscounts;

        // Check to see if this order will qualify the active user's customer account for a Rewards Program tier upgrade
        if (rewards()->isRegistered() && b2b()->isBillingDirect() && rewards()->isEligible($soldToCustomer, false)) {
            $rewardsUpgradeSpending = rewards()->calculateUpgradeSpending($status, $soldToCustomer);
            $status['rewardsUpgradeSpending'] = $rewardsUpgradeSpending;

            // If the order will cause a tier upgrade, recalculate the 'discounts.rewards' discount type in $status
            if ($status['rewardsUpgradeSpending'] < 0) { //Tier Upgrade
                $status['baseRewardsTier'] = $status['rewardsTier'];
                $tierJump = $status['rewardsUpgradeSpending'] * -1;
                $status['rewardsTier'] = min(4, rewards()->tier() + $tierJump);
                $status['upgradedTier'] = min(4, rewards()->tier() + $tierJump);

                $rewardsDiscountTotal = 0;
                foreach ($items as $itemKey => $item) {
//                if (array_has($item, 'discounts.rewards')) {
                    $discountPercent = rewards()->discountForPartTier($item['part']->id, $status['upgradedTier'], $soldToCustomer);
                    $discountAmount = round($item['basePrice'] * ($discountPercent / 100), 2);
                    $extendedDiscountAmount = $discountAmount * $item['quantity'];
                    $baseDiscountAmount = round($item['basePrice'] * ($discountPercent / 100), 2);
                    $discountedPrice = $item['basePrice'] - $baseDiscountAmount;
                    $extendedPrice = $discountedPrice * $item['quantity'];
                    $rewardsDiscountTotal += $extendedDiscountAmount;
                    //I don't think this is right...this shouldn't be applied until we know for sure whether the reward discounts should be applied

                    /**
                     *  The old method didn't actually add any objects that would be considered in the checkout controller.
                     * It might never have actually worked.
                     * This new method will actually inject the new discount values (replacing them because these are
                     * discounts acquainted with the new upgrade tier.
                     */
                    $discounts = $item->get('discounts');
                    $discounts['rewards']['percent'] = $discountPercent;
                    $discounts['rewards']['baseAmount'] = $discountAmount;
                    $discounts['rewards']['extendedAmount'] = $extendedDiscountAmount;
                    $item->put('discounts', $discounts);

                    // pretty sure these fields are vestigial and unused now, leaving until we clean up web cart someday
                    $item->put('baseDiscountAmount', $discountAmount);
                    $item->put('extendedDiscountAmount', $extendedDiscountAmount);
                    $item->put('discountedPrice', $discountedPrice);
                    $item->put('extendedPrice', $extendedPrice);
                }
                $status['discounts']['rewards'] = $rewardsDiscountTotal;
            }
        }

        // Calculate any "header-level" discounts applied via promotion(s)
        $calculatedOrderHeaderDiscountMax = 0;
        foreach ($orderHeaderDiscounts as $orderHeadDiscount) {
            $discountMax = array_get($orderHeadDiscount, 'maximum_dollar_amount');
            $calculatedOrderHeaderDiscountMax = max($calculatedOrderHeaderDiscountMax, $discountMax);
        }

        foreach ($orderHeaderDiscounts as $headerDiscount) {
            $discountPercent = array_get($headerDiscount, 'discount_percent');
            $discountMaxAmount = array_get($headerDiscount, 'maximum_dollar_amount');
            $discountAmount = round($status['baseTotal'] * ($discountPercent/100), 2);
            if ($discountMaxAmount > 0) {
                $discountAmount = min($discountMaxAmount, $discountAmount);
            }
            array_increment($status, "discounts.promotion", $discountAmount);
        }

        if ($calculatedOrderHeaderDiscountMax > 0) {
            $status['discounts']['promotion'] = min(array_get($status, 'discounts.promotion', 0), $calculatedOrderHeaderDiscountMax);
        }


        // Assign 'discountedTotal' to be the combined values of all discount types in $status
        $status['discountedTotal'] = array_get($status, 'baseTotal');
        foreach (array_get($status, 'discounts') as $type => $amt) {
            $status['discountedTotal'] = array_get($status, 'discountedTotal') - $amt;
        }
        if($status['discountedTotal'] < 0 ){
            $status['discountedTotal'] = 0;
        }

        // Rate Shop the order
        $rate = b2b()->activeRate();
        if ($rate && !$rate->isError()) {
            $status['shippingAmount'] = $rate->rate();
            $status['shippingDiscountAmount'] = $rate->discountAmount();
            $status['shippingTaxAmount'] = $rate->taxAmount();
        }

        // Get the actual order total (discounted total plus any taxes)
        $status['orderTotal'] = $status['discountedTotal'] + $status['taxAmount'];

        // Final check -- if user is on a microsite that has spending and/or minimum order limits, set the order to
        // require approval if the order violates any of those rules
        $webUserWebSilo = $webUser->webSilos()->where('websilo_id', b2b()->activeWebSilo()->id)->first();
        $spendingLimit = array_get($webUserWebSilo, 'pivot.spending_limit', false);
        $minimumOrder = array_get($webUserWebSilo, 'pivot.minimum_order', false);
        if ($spendingLimit === false) $spendingLimit = array_get($webSilo, 'default_spending_limit', false);
        if ($minimumOrder === false) $minimumOrder = array_get($webSilo, 'default_minimum_order', false);
        if ($spendingLimit && $spendingLimit > 0 && $status['orderTotal'] > $spendingLimit) $status['requireApproval'] = true;
        if ($minimumOrder && $minimumOrder > 0 && $status['orderTotal'] < $minimumOrder) $status['requireApproval'] = true;

        $status['items'] = $items;

        return $status;
    }

    /**
     * Get the list of all triggered promotions and their relevant info (discounts, etc)
     * @param $customer
     * @param $webCartItems
     * @param $distinctPartInfo
     * @return array
     * [
     *     'triggeredPromotions' => [array of Promotion objects, with extra 'applied' field],
     *     'discountNotes'=> [array of Discount Notes - DEPRECATED],
     *     'orderHeaderDiscounts' => [array of the following arrays, for each Order Discount action: ['discount_percent' => ?, 'maximum_dollar_amount' => ?]],
     *     'freeItemActions' => [array of PromotionFreeItemAction objects]
     * ]
     */
    public static function getTriggeredPromotionsInfo($customer, $webCartItems, $distinctPartInfo) {
        $promotions = Promotion::getActivePromotions();
        $orderHeaderDiscounts = [];
//        $priceListActions = [];
        $triggeredPromotions = array();
        $discountNotes = array();
        $freeItemActionsInfo = [];

        foreach($promotions as $promotion) {
            $triggeredAllTriggers = true;
            $triggers = $promotion->triggers;
            foreach($triggers as $trigger) {
                $triggeredAllTriggers = $triggeredAllTriggers && $trigger->details->isTriggered($customer, $webCartItems, $distinctPartInfo);
            }

//            $itemDiscountActions = $promotion->itemDiscountActions;

            if($triggeredAllTriggers) {
//                foreach ($itemDiscountActions as $discountAction) {
//                    if(strcmp($discountAction->discount_note, "") != 0) {
//                        $discountNotes[] = $discountAction->discount_note;
//                    }
//                }
                foreach ($promotion->orderDiscountActions as $discountAction) {
                    $orderHeaderDiscounts[] = [
                        'discount_percent' => array_get($discountAction, 'discount_percent'),
                        'maximum_dollar_amount' => array_get($discountAction, 'maximum_dollar_amount'),
                    ];
                }

                foreach ($promotion->freeItemActions as $freeItemAction) {
                    $freeItemActionsInfo[] = [
                        'freeItemAction' => $freeItemAction,
                        'cheapestPartList' => self::getCheapestWebCartItemsForAFreeItemPromo($freeItemAction, $distinctPartInfo)
                    ];
                }
//                foreach ($promotion->priceListActions as $priceListAction) {
//                    $priceListActions[] = [
//                        'price_list' => array_get($priceListAction, 'price_list')
//                    ];
//                }

                if (count(array_get($promotion, 'orderDiscountActions', [])) ||
                    count(array_get($promotion, 'shippingDiscountActions', [])) ||
                    count(array_get($promotion, 'orderCommentActions', []))) {
                    $promotion['applied'] = true;
                }

                $triggeredPromotions[] = $promotion;
            }
        }

        // if multiple price list actions are set across multiple promotions
//        if (count($priceListActions) !== 1) $priceListActions = [];

        return
        [
            'triggeredPromotions' => $triggeredPromotions,
            'discountNotes'=> $discountNotes,
            'orderHeaderDiscounts' => $orderHeaderDiscounts,
//            'priceListActions' => $priceListActions,
            'freeItemActionsInfo' => $freeItemActionsInfo
        ];
    }

    /**
     * @param $customer
     * @param $webCartItems
     * @param $distinctPartInfo
     * @return array
     */
    public static function getAlmostQualifiedPromotionsInfo($customer, $webCartItems, $distinctPartInfo) {
        /**
         *  Addendum [9-14-2018]:
         *  Promotions with Almost Qualified Triggers should only be considered if all of their other non-AQ triggers
         *      (if they have any others) have been satisfied/triggered.
         *  For example, if the promotion has a code trigger, then unless that code trigger has been activated, any
         *  AlmostQualified conditions will be disregarded.
         */
        $promotions = Promotion::getActivePromotions();
        $almostQualifyingPromotionsInfo = [];
//        TODO: Get rid the indexing, replace with basic appending
        $i = 0;
        foreach($promotions as $promotion) {
            $triggers = $promotion->triggers;
            $almostQualifiedValid = true; // This determines if there is a non-AQ trigger that is not satisfied.

            $j = 0;
            $promo = []; // Stores the Promotion and List of Triggers assosciated.
            $trigList = []; // Stores the Triggers w/ Qualifying Fields.
            foreach($triggers as $trigger) {

                if ($trigger->details->hasAlmostQualifyingField()) {
                    $trigDetails = [];
//                    For every promotion that has at least one trigger for an almost qualifying field, we create an object
//                      that stores the promotion, the triggerdetails, and the current cart status in terms of the relevant amounts.
                    $amountLeft = $trigger->details->isAlmostQualifying($customer, $webCartItems, $distinctPartInfo);
                    if ($amountLeft > 0) {
                        $trigDetails['trigger'] = $trigger; // Stores individual trigger and the amount progress their tied to.
                        if($trigger->details->minimum_quantity > 0 ) {
                            $trigDetails['type'] = 'minimum_quantity';
                        }else if($trigger->details->minimum_dollar > 0){
                            $trigDetails['type'] = 'minimum_dollar';
                        }
                        $trigDetails['amountLeft'] = $amountLeft;
                        $trigList[$j] = $trigDetails;        //Stores all Triggers that are tied to the individual promotion.
                        $j++;
                    }
                }else{
                    if(!$trigger->details->isTriggered($customer, $webCartItems, $distinctPartInfo)){
                        $almostQualifiedValid = false;
                    }
                }
            }

            if(count($trigList) and $almostQualifiedValid){
                $promo['triggers'] = $trigList;
                $promo['promotion'] = $promotion;
                $almostQualifyingPromotionsInfo[$i] = $promo;
                $i++;
            }
        }
        return $almostQualifyingPromotionsInfo;
    }

    /**
     * @param $customerPriceListName
     * @param $siloPriceListName
     * @param $currencyCode
     * @param $webCartItems
     * @return array
     * [
     *     part_id => [
     *         'baseprice' => base price of the part from the price list used,
     *         'isContractPrice' => boolean value from PriceLists->contract_flag,
     *         'priceListUsed' => name of the price list that was used,
     *         'totalQuantity' => total quantity of this part in the cart
     *     ]
     * ]
     */
    private static function getDistinctPartInfo($customerPriceListName, $siloPriceListName, $currencyCode, $webCartItems) {
        $distinctPartQuantities = [];
        foreach ($webCartItems as $webCartItem) {
            array_increment($distinctPartQuantities, $webCartItem->webPart->part->id, $webCartItem->quantity);
        }

        $distinctLineQuantities = [];

        foreach($webCartItems as $webCartItem){
            $lineInfo = [
                'partId' => $webCartItem->webPart->part->id,
                'quantity' => $webCartItem->quantity
            ];
            array_push($distinctLineQuantities, $lineInfo);
        }

        // get all part prices based on $priceListNameToUse
        $customerPrices = self::getPartPricesByPriceList($customerPriceListName, $currencyCode, array_keys($distinctPartQuantities));

        // get all the part prices based on the default silo price list just in case
        $siloPrices = self::getPartPricesByPriceList($siloPriceListName, $currencyCode, array_keys($distinctPartQuantities));
//        dd($webCartItems);

        $distinctPartInfo = [];
        foreach ($distinctPartQuantities as $partId => $partQuantity) {
            $part = Part::find($partId);

            // Determine appropriate pricing for the part based on the price list(s)
            $partPriceInfo = self::getPartPriceInfo($customerPrices, $siloPrices, $part, $partQuantity, $customerPriceListName, $siloPriceListName, $currencyCode);

            $distinctPartInfo[$partId] = [
                'basePrice' => $partPriceInfo['basePrice'],
                'isContractPrice' => $partPriceInfo['isContractPrice'],
                'priceListUsed' => $partPriceInfo['priceListUsed'],
                'totalQuantity' => $partQuantity
            ];
        }

//        $distinctPartInfo = [];
        $distinctLineInfo = [];
        foreach($distinctLineQuantities as $lineInfo){
            $partId = $lineInfo['partId'];
            $quantity = $lineInfo['quantity'];
            $part = Part::find($partId);

            $partPriceInfo = self::getPartPriceInfo($customerPrices, $siloPrices, $part, $quantity, $customerPriceListName, $siloPriceListName, $currencyCode);
            $partInfo = [
                'partId' => $partId,
                'basePrice' => $partPriceInfo['basePrice'],
                'isContractPrice' => $partPriceInfo['isContractPrice'],
                'priceListUsed' => $partPriceInfo['priceListUsed'],
                'totalQuantity' => $partQuantity
            ];
            array_push($distinctLineInfo, $partInfo);

        }
        $distincts = [
            'distinctLineInfo' => $distinctLineInfo,
            'distinctPartInfo' => $distinctPartInfo
        ];

        return $distincts;
    }

    /**
     * @param $priceListName
     * @param $currencyCode
     * @param $partIds
     * @return array|PriceList[]|\Illuminate\Database\Eloquent\Collection
     */
    private static function getPartPricesByPriceList($priceListName, $currencyCode, $partIds) {
        return PriceList::where('price_list', '=', $priceListName)
                        ->where('currency', '=', $currencyCode)
                        ->whereIn('part_id', $partIds)
                        ->orderBy('quantity_level', 'DESC')
                        ->get();
    }

//    /* Returns the cheapest item in the web cart that matches the product_type hierarchy */
//    private static function getCheapestWebCartItemsByProduct($freeItemActions, $webCartItems, $customerPrices, $siloPrices, $partTotalQuantities, $priceListName, $soldToCustomer) {
//        $cheapestPrice = -1;
//        $cheapestItem = 0;
//        $promotionUsed = null;
//        foreach ($webCartItems as $webCartItem) {
//            $part = $webCartItem->webPart->part;
//
//            $qualifiesForFreeItem = false;
//            foreach ($freeItemActions as $freeItemAction) {
//                if ($freeItemAction->isApplicable($part)) {
//                    $qualifiesForFreeItem = true;
//                    $promotionUsed = $freeItemAction->promotion()->first();
//                    break;
//                }
//            }
//
//            if ($qualifiesForFreeItem) {
//                $partTotalQuantity = array_get($partTotalQuantities, $part->id, 0);
//                $currencyCode = $soldToCustomer->currency;
//                $siloPriceListName = b2b()->activeWebSilo()->priceList;
//                $partPriceInfo = self::getPartPriceInfo($customerPrices, $siloPrices, $part, $partTotalQuantity, $priceListName, $siloPriceListName, $currencyCode);
//                $basePrice = $partPriceInfo['basePrice'];
//                $cheapestPrice = $cheapestPrice == -1 ? $basePrice : min($cheapestPrice, $basePrice);
//                if ($cheapestPrice === $basePrice) $cheapestItem = $webCartItem;
//            }
//        }
//        return [
//            'cheapestItem' => $cheapestItem,
//            'promotionUsed' => $promotionUsed
//        ];
//    }

    /**
     * Determine the cheapest parts that are applicable to a free item promo action,
     * and how many of them can be marked as 'free' in the current cart.
     * @param $freeItemAction
     * @param $distinctPartInfo
     * @return array
     * [
     *     <part_id> => <number to mark as free>
     * ]
     */
    private static function getCheapestWebCartItemsForAFreeItemPromo($freeItemAction, $distinctPartInfo) {
        $cheapestList = [];
        $cheapestItemsFound = 0;

        $applicableItems = [];
        foreach ($distinctPartInfo as $partId => $partInfo) {
            $part = Part::find($partId);
            if ($freeItemAction->isApplicable($part)) {
                $applicableItems[] = [
                    'partId' => $partId,
                    'partInfo' => $partInfo
                ];
            }
        }

        usort($applicableItems, array('\Hilco\Models\WebCart', 'compareWebCartItemByPrice'));

        $maxFree = $freeItemAction->num_free;
        foreach($applicableItems as $key => $value) {
            $partId = $value['partId'];
            $partInfo = $value['partInfo'];
            if ($cheapestItemsFound < $maxFree) {
                $partQuantity = $partInfo['totalQuantity'];
                $quantityToDiscount = ($cheapestItemsFound + $partQuantity <= $maxFree ? $partQuantity : $maxFree - $cheapestItemsFound);
                array_increment($cheapestList, $partId, $quantityToDiscount);
                $cheapestItemsFound += $quantityToDiscount;
            }
        }

        return $cheapestList;
    }

    /**
     * Comparator function for sorting web cart items by their base price
     * Only used in getCheapestWebCartItemsForAFreeItemPromo() right now
     * @param $item1
     * @param $item2
     * @return int
     */
    static function compareWebCartItemByPrice($item1, $item2) {
        $price1 = array_get($item1, 'partInfo.basePrice');
        $price2 = array_get($item2, 'partInfo.basePrice');

        if ($price1 === $price2)
            return 0;

        return ($price1 < $price2) ? -1 : 1;
    }

    /**
     * Returns general price info for a given part, including its base price, whether it's under
     * contract pricing, and which price list was used. Tries to get the customer's price list first,
     * and uses the default WebSilo price list otherwise.
     * @param $customerPrices
     * @param $siloPrices
     * @param $part
     * @param $partTotalQuantity
     * @param $customerPriceListName
     * @param $siloPriceListName
     * @param $currencyCode
     * @return array
     * [
     *     'basePrice' => base price of the part from the price list used,
     *     'isContractPrice' => boolean value from PriceLists->contract_flag,
     *     'priceListUsed' => name of the price list that was used
     * ]
     */
    private static function getPartPriceInfo($customerPrices, $siloPrices, $part, $partTotalQuantity, $customerPriceListName, $siloPriceListName, $currencyCode) {
        $customerPriceList = PriceList::filterForPartQuantity($customerPrices, $part->id, $partTotalQuantity);
        if ($customerPriceList) {
            $basePrice = $customerPriceList->price;
            $isContractPrice = $customerPriceList->contract_flag;
            $priceListUsed = $customerPriceListName;
        } else {
            $siloPriceList = PriceList::filterForPartQuantity($siloPrices, $part->id, $partTotalQuantity);
            if ($siloPriceList) {
                $basePrice = $siloPriceList->price;
                $isContractPrice = false;
                $priceListUsed = $siloPriceListName;
            } else {
                $basePrice = $part->getListPriceForCurrencyCode($currencyCode);
                $isContractPrice = false;
                $priceListUsed = 'PART_TABLE_LIST_PRICE';
            }
        }
        return [
            'basePrice' => $basePrice,
            'isContractPrice' => $isContractPrice,
            'priceListUsed' => $priceListUsed
        ];
    }

    /**
     * Update the triggered promotions list to mark one as 'applied'
     * @param $triggeredPromotions
     * @param $promoToUpdate
     * @return array
     */
    private static function updateAppliedPromotions($triggeredPromotions, $promoToUpdate) {
        $promos = array();
        foreach ($triggeredPromotions as $promo) {
            if ($promo->id === $promoToUpdate->id) {
                $promo['applied'] = true;
            }
            $promos[] = $promo;
        }
        return $promos;
    }

    /**
     * Calculate the part's tax amount, if applicable, based on the current customer and the calculated extended price
     * @param $part
     * @param $extendedPrice
     * @return float|int
     */
    private static function calculateTaxAmount($part, $extendedPrice) {
        $itemTaxAmount = 0;
        if (b2b()->isCustomerTaxable()) {
            $shippingAddress = b2b()->activeShippingAddress();
            $defaultShipFromId = array_get($shippingAddress, 'default_shipfrom_id', false);
            $isPartTaxable = false;

            if ($defaultShipFromId) {
                foreach (array_get($part, 'inventoryItems', []) as $inventoryItem) {
                    if (array_get($inventoryItem, 'plant_id') == $defaultShipFromId) {
                        $isPartTaxable = array_get($inventoryItem, 'part_taxable', false);
                    }
                }
            }

            if ($isPartTaxable) {
                $taxes = auth()->user()->getTaxes($shippingAddress);
                foreach($taxes as $taxPart) {
                    $itemTaxAmount += round($extendedPrice * ($taxPart / 100), 2);
                }

                if (b2b()->activeCountry() == 'GB') {
                    $vatTax = array_get($part, 'VATTax.tax_percentage', 0);
                    $itemTaxAmount += round($extendedPrice * ($vatTax / 100), 2);
                }
            }
        }
        return $itemTaxAmount;
    }

    /**
     * Get the name of the Price List to use on the order -- exists because Croakies Adventures logic can "override"
     * a customer's default price list (see CroakiesHelper.php for more details)
     * @param $customerPriceListName
     * @param $webCartItems
     * @return false|int|null|string
     */
    public static function getPriceListNameToUse($customerPriceListName, $webCartItems) {
        // check Croakies Adventures Race logic if customer is qualifying...
        $priceListNameToUse = croakies()->getPriceListToUse($webCartItems);

        // DISABLED, DEFUNCT AS OF EARLY 2018
//         ...and check if there's a promotion that applies a price list...
//        $priceLists = $triggeredPromotionsInfo['priceListActions'];
//        if (count($priceLists)) $priceListNameToUse = array_values($priceLists)[0];//reset($priceLists);

        //...otherwise just use sold-to customer's default price list
        if (!$priceListNameToUse) $priceListNameToUse = $customerPriceListName;

        return $priceListNameToUse;
    }
}
