<?php

namespace Hilco\Models;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;
use Mailchimp\Mailchimp;

/**
 * Class Promotion
 * @package Hilco\Models
 *
 * @method static Builder active()
 */
class Promotion extends WebModel {
    protected $table = 'Promotions';
    protected $fillable = [
        'name',
        'start_date', 'end_date',
        'enabled',
        'cart_header', 'checkout_header',
        'cart_header_title', 'checkout_header_title',
        'homepage_banner_text', 'homepage_banner_link',
        'almost_header', 'almost_header_title',
        'task_excluded', 'task_priority',
        'mailchimp_campaign_id',
        'description',
        'commit_sequence'
    ];
    protected $casts = [
        'enabled' => 'boolean',
    ];

    protected static $morphMap = [
        'codeTrigger' => PromotionCodeTrigger::class,
        'customerTrigger' => PromotionCustomerTrigger::class,
        'currentOrderTrigger' => PromotionCurrentOrderTrigger::class,
        'currencyTrigger' => PromotionCurrencyTrigger::class,
        'itemQuantityTrigger' => PromotionItemQuantityTrigger::class,
        'sourceTrigger' => PromotionSourceTrigger::class,
        'shippingDiscountAction' => PromotionShippingDiscountAction::class,
        'freeItemAction' => PromotionFreeItemAction::class,
        'itemDiscountAction' => PromotionItemDiscountAction::class,
        'orderCommentAction' => PromotionOrderCommentAction::class,
        'orderDiscountAction' => PromotionOrderDiscountAction::class,
        'priceListAction' => PromotionPriceListAction::class
    ];


    public static function boot() {
        parent::boot();

        static::deleted(function (Promotion $promotion) {
            foreach ($promotion->triggers as $trigger) {
                $trigger->delete();
            }
            foreach ($promotion->actions as $action) {
                $action->delete();
            }
        });
    }

    public static function getMorphMap() {
        return self::$morphMap;
    }

    public static function getActivePromotions() {
        return Promotion::active()->get();
    }

    public static function getMailChimpCampaigns() {
        $mailchimp = new Mailchimp(config('mailchimp.apikey'), []);
        try {
            $newArray = array();
            $newArray[''] = '';
            $campaignsCollect = collect($newArray);
            $offset = 0;
            $limit = 10;
            do {
                $campaigns = $mailchimp->request('campaigns', [
                    'fields' => 'campaigns.id,campaigns.settings.title',
                    'count' => $limit,
                    'offset' => $offset
                ]);
                $campaignsArray = $campaigns->toArray();

                $newArray = array();
                foreach ($campaignsArray as $key => $value) {
                    $newArray[$value->id] = $value->settings->title;
                    unset($campaignsArray[$key]);
                }
                natcasesort($newArray);
                $campaignsCollect = $campaignsCollect->union(collect($newArray));

                $offset += $limit;
            }
            while (count($campaigns) > 0);
            $arrayForSort = $campaignsCollect->toArray();
            natcasesort($arrayForSort);
            $campaignsCollect = collect($arrayForSort);
            return $campaignsCollect;
        } catch (\Exception $e) {
            echo ('could not return mailchimp campaigns');
            return collect();
        }
    }


    public function triggers() {
        return $this->hasMany(PromotionTrigger::class);
    }

    public function actions() {
        return $this->hasMany(PromotionAction::class);
    }

    public function codeTriggers() {
        return $this->morphedByMany(PromotionCodeTrigger::class, 'trigger', 'PromotionTriggers','promotion_id');
    }

    public function customerTriggers() {
        return $this->morphedByMany(PromotionCustomerTrigger::class, 'trigger', 'PromotionTriggers', 'promotion_id');
    }

    public function freeItemActions() {
        return $this->morphedByMany(PromotionFreeItemAction::class, 'action', 'PromotionActions', 'promotion_id');
    }

    public function itemDiscountActions() {
        return $this->morphedByMany(PromotionItemDiscountAction::class, 'action', 'PromotionActions', 'promotion_id')
            ->withPivot(['sequence'])
            ;
    }

    public function orderCommentActions() {
	return $this->morphedByMany(PromotionOrderCommentAction::class, 'action', 'PromotionActions', 'promotion_id');
    }

    public function orderDiscountActions() {
        return $this->morphedByMany(PromotionOrderDiscountAction::class, 'action', 'PromotionActions', 'promotion_id')
            ->withPivot(['sequence'])
            ;
    }

    public function priceListActions() {
        return $this->morphedByMany(PromotionPriceListAction::class, 'action', 'PromotionActions', 'promotion_id');
    }

    public function shippingDiscountActions() {
        return $this->morphedByMany(PromotionShippingDiscountAction::class, 'action', 'PromotionActions', 'promotion_id')
            ->withPivot(['sequence'])
            ;
    }

    public function orders() {
        return $this->belongsToMany(Order::class,
            'Promotion_SalesOrder',
            'promotion_id',
            'salesorder_id')
            ->wherePivot('applied', '=', 1);
    }


    public function getStartDateDatepickerAttribute() {
        $date = ($this->attributes['start_date'] == '0000-00-00') ? Carbon::create(2017, 1, 1) : Carbon::createFromFormat('Y-m-d', $this->attributes['start_date']);

        return $date->format('Y-m-d');
    }

    public function getEndDateDatepickerAttribute() {
        $date = ($this->attributes['end_date'] == '0000-00-00') ? Carbon::now()->addYear(1) : Carbon::createFromFormat('Y-m-d', $this->attributes['end_date']);

        return $date->format('Y-m-d');
    }

    /**
     * @deprecated pretty sure nothing uses this, can probably be deleted
     * @return \Illuminate\Support\Collection
     */
    public function getAllActionsAttribute() {
        return collect([
            $this->itemDiscountActions->flatten(),
        ])->flatten();
    }


    public function scopeActive($query) {
        $query
            ->where('enabled', 1)
            ->where(function ($query) {
                return $query
                    ->where('start_date', '=', '0000-00-00 00:00:00')
                    ->orWhere('start_date', '<=', DB::raw('DATE(NOW())'))
                    ;
            })
            ->where(function ($query) {
                return $query
                    ->where('end_date', '=', '0000-00-00 00:00:00')
                    ->orWhere('end_date', '>=', DB::raw('DATE(NOW())'))
                    ;
            })
        ;
    }


    public function mailchimpCampaignName() {
        if ($this->mailchimp_campaign_id) {
            $mailchimp = new Mailchimp(config('mailchimp.apikey'));
            $campaign = $mailchimp->get('campaigns/' . $this->mailchimp_campaign_id);
            $fieldArray = $campaign->toArray();
            return $fieldArray['settings']->title;
        } else {
            return '';
        }
    }

    public function mailchimpCampaign() {
        if ($this->mailchimp_campaign_id) {
            $mailchimp = new Mailchimp(config('mailchimp.apikey'));
            try {
                $campaign = $mailchimp->get('campaigns/' . $this->mailchimp_campaign_id);
                $fieldArray = $campaign->toArray();
                return $fieldArray;
            } catch (Exception $e) {
                return [];
            }
        } else {
            return [];
        }
    }


    use HasCommitSequence;
}
