<?php

namespace Hilco\Models;

use DB;
use Hilco\Models\SelectsRelatedCount;
use Venturecraft\Revisionable\RevisionableTrait;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class WebAttribute extends Model
{
	use \Hilco\SoftDeletes;

	const CREATED_AT = 'date_created';
	const UPDATED_AT = 'date_modified';

	protected $table = 'WebAttributes';
	protected $fillable = ['attribute_name', 'searchbucket_type', 'global_search_facet'];

	public function webParts() {
		return $this->belongsToMany(WebPart::class, 'WebAttribute_WebPart', 'webattribute_id', 'webpart_id')->withPivot(['attribute_value']);
	}

	public function webFamilies() {
		return $this->belongsToMany(WebFamily::class, 'WebAttribute_WebFamily', 'webattribute_id', 'webfamily_id');
	}

	public function values() {
		return $this->hasMany(WebAttribute_WebPart::class, 'webattribute_id', 'id');
	}
	
	public function searchBuckets(){
	    if($this->searchbucket_type == "static"){
	        return $this->hasMany(SearchBucketStatic::class, 'webattribute_id', 'id');
        }else if($this->searchbucket_type == "range"){
            return $this->hasMany(SearchBucketRange::class, 'webattribute_id', 'id');
        }else{
	        return null;
        };
    }
    
	public function getValuesInCategoryAttribute() {
		return $this->webParts->sortBy('pivot.attribute_value')->pluck('pivot.attribute_value', 'pivot.attribute_value');
	}

	public function scopeInCollection($query, $webCollection) {
		$webCollectionId = ($webCollection instanceof WebCollection) ? $webCollection->id : $webCollection;
		return $query->whereHas('webFamilies', function ($query) use ($webCollectionId) {
			return $query->inCollection($webCollectionId);
		});
	}

	public function scopeInCategory($query, $webCategory) {
		$webCategoryId = ($webCategory instanceof WebCategory) ? $webCategory->id : $webCategory;
		return $query->whereHas('webFamilies', function ($query) use ($webCategoryId) {
			return $query->inCategory($webCategoryId);
		});
	}

	public function scopeInGroup($query, $webGroup) {
		$webGroupId = ($webGroup instanceof WebGroup) ? $webGroup->id : $webGroup;
		return $query->whereHas('webFamilies', function ($query) use ($webGroupId) {
			return $query->inGroup($webGroupId);
		});
	}

	public static function in($model) {
		if ($model instanceof WebGroup) {return self::inGroup($model);}
		else if ($model instanceof WebCategory) return self::inCategory($model);
		else if ($model instanceof WebCollection) return self::inCollection($model);
	}

	public function getAttributeNameAttribute() {
	    $key = "attributes." . $this->attributes['attribute_name'];
            $translation = $this->getTranslation('attribute_name');
            if(isset($translation)){
                return $translation;
            }
	    return transDefault($key, $this->attributes['attribute_name']);
    }

    public function getDisplayNameForUACAttribute() {
        return $this->getAttributeNameAttribute();
    }

	use RevisionableTrait, HasModelTranslations;

	public function getFacetName(){
        $name = preg_replace('/\s+/', '_', $this->attribute_name);
        return $name;
    }

    public function scopeAttributesByGroup($query, $groupId) {
        return $query
            ->whereHas('webParts', function ($query) use ($groupId) {
                return $query->whereHas('webFamily', function ($query) use ($groupId) {
                    return $query->whereHas('webCollections', function ($query) use ($groupId) {
                        return $query->whereHas('webCategories', function ($query) use ($groupId) {
                            return $query->whereHas('webGroups', function ($query) use ($groupId) {
                                return $query->where('WebGroups.id', '=', $groupId);
                            });
                        });
                    });
                });
            })
        ;
    }

    public function scopeAttributesByCategory($query, $groupId) {
        return $query
            ->whereHas('webParts', function ($query) use ($groupId) {
                return $query->whereHas('webFamily', function ($query) use ($groupId) {
                    return $query->whereHas('webCollections', function ($query) use ($groupId) {
                        return $query->whereHas('webCategories', function ($query) use ($groupId) {
                            return $query->where('WebCategories.id', '=', $groupId);
                        });
                    });
                });
            })
            ;
    }

    public function scopeAttributesByCollection($query, $groupId) {
        return $query
            ->whereHas('webParts', function ($query) use ($groupId) {
                return $query->whereHas('webFamily', function ($query) use ($groupId) {
                    return $query->whereHas('webCollections', function ($query) use ($groupId) {
                        return $query->where('WebCollections.id', '=', $groupId);
                    });
                });
            })
            ;
    }
}
