<?php

namespace Hilco\Models;
use DB;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Venturecraft\Revisionable\RevisionableTrait;
use Illuminate\Support\Facades\Input;

/**
 * Class WebCategory
 * @package Hilco\Models
 * @method static WebCategory firstOrNew(array $attributes)
 */
class WebCategory extends WebModel implements HasLandingPageInterface {
    protected $table = "WebCategories";
    protected $fillable = ['name', 'slug', 'webGroupIds', 'is_visible', 'detail', 'sort_priority'];
    protected $casts = [
        ['is_visible' => 'boolean']
    ];
    protected $perPage = 24;
    protected $hierarchyParent = WebGroup::class;

//    use KeyWordsFunctional;

    public function getPerPage()
    {
        $tmpPerPage =Input::get('perPage');
        return (isset($tmpPerPage) && $tmpPerPage > 0)?$tmpPerPage:$this->perPage;
    }

    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function webGroups() {
        return $this->belongsToMany(WebGroup::class, 'WebCategory_WebGroup', 'webcategory_id', 'webgroup_id')->withPivot('display_order');
    }

    public function webCollections() {
        return $this->belongsToMany(WebCollection::class, 'WebCollection_WebCategory', 'webcategory_id', 'webcollection_id')->withPivot('display_order');
    }

    public function visibleWebCollections() {
        return $this->belongsToMany(WebCollection::class, 'WebCollection_WebCategory', 'webcategory_id', 'webcollection_id')
            ->withPivot('display_order')
            ->visible()
            ->displayOrder()
        ;
    }

    public function scopeDisplayOrder(Builder $query) {
        return $query->orderBy('pivot_display_order', 'asc')->orderBy('name', 'asc');
    }

    public function scopeAlphabetical(Builder $query) {
        return $query->orderBy('name', 'asc');
    }

    public function getFeaturedChildrenAttribute() {
        return $this->visibleWebCollections()->wherePivot('is_featured', 1)->get();
    }

    public function getChildrenAttribute() {
        return $this->visibleWebCollections()->get();
    }

    public function scopeInGroup($query, $groupId) {
        return $query->whereHas('webGroups', function ($query) use ($groupId) {
            $query->where('webgroup_id', $groupId);
        });
    }

    public function scopeInSilo($query, $webSiloId) {
        return $query->whereHas('webGroups', function ($query) use ($webSiloId) {
            return $query->inSilo($webSiloId);
        });
    }

    public function setWebGroupIdsAttribute($ids) {
        $this->save();
        $this->createRevisionRecord('webGroupIds', $this->webGroups->pluck('id')->toJson(), json_encode($ids));
        $this->webGroups()->sync($ids);
    }

    public function getHierarchyPathsAttribute() {
        $paths = [];
        foreach ($this->webGroups as $webGroup) {
            foreach ($webGroup->webSilos as $webSilo) {
                $paths[] = [
                    'webSilo' => $webSilo,
                    'webGroup' => $webGroup,
                ];
            }
        }
        return $paths;
    }

    public function scopeVisible(Builder $query) {
        return $query
            ->where('WebCategories.is_visible', '1')
            ->whereHas('webGroups', function ($query) {
                return $query->visible();
            })
        ;
    }

    public function slugUrl() {
        return route('category.slug', [$this->slug]);
    }
    public function getAssetPrefix() {
        return "webCategories/{$this->id}/webContexts";
    }

    public function getPrimaryImage() {
        $asset = $this->assetsByType('primary')->first();
        if (!is_null($asset)) return $asset;

        $webCollection = $this->visibleWebCollections->first();
        if ($webCollection) return $webCollection->getPrimaryImage();
    }


    use HasAssets, HasSlug, HasLandingPage, RevisionableTrait, ManyToManyRevisionableTrait;

    public function getParentRelationName() {
        return 'webGroups';
    }
}
