<?php

namespace Hilco;

use Debugbar;
use Hilco\Models\AustraliaFreightPostCode;
use Hilco\Models\Carrier;
use Hilco\Models\Customer;
use Hilco\Models\CustomerCreditProfile;
use Hilco\Models\Part;
use Hilco\Models\Plant;
use Hilco\Models\VATTax;
use Hilco\Models\WebCategory;
use Hilco\Models\WebCollection;
use Hilco\Models\WebFamily;
use Hilco\Models\WebFeedbackResponse;
use Hilco\Models\WebGroup;
use Hilco\Models\WebPart;
use Hilco\Models\WebRole;
use Hilco\Models\WebSilo;
use Hilco\Models\WebUrl;
use Hilco\Shipments\Rate;
use HilcoB2B\AuthorizeNetWrapper;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Hilco\Facades\RateShop;

class B2BHelper {
    protected $activeCustomer;
    protected $activeWebSilo;
    protected $activeShippingAddress;
    protected $activeBillingCustomer;
    protected $activeCarrier;
    protected $activeRate;

    public function __construct() {
        $this->activeShippingAddress = null;
        $this->activeBillingCustomer = null;
        $this->activeCarrier = null;
        $this->activeRate = null;
        $this->loadActiveCustomer();
        if ($this->activeCustomer) {
            $this->loadActiveBillingCustomer();
            $this->loadActiveShippingAddress();
            $this->loadActiveCarrier();
            $this->loadActiveRate();
        }

        $this->loadLocale();
        $this->loadActiveWebSilo();
    }

    protected function loadActiveCustomer() {
        $this->activeCustomer = false;

        $aliasedCustomerId = session()->get('b2b.aliasedCustomerId', false);
        if ($aliasedCustomerId) {
            $this->activeCustomer = Customer::with('segments.address.plant', 'discounts', 'availableShippingAddresses', 'webSilo')->find($aliasedCustomerId);
        } else {
            $user = auth()->user();
            if ($user) $this->activeCustomer = $user->customer;
        }
    }

    protected function loadActiveBillingCustomer() {
        $this->activeBillingCustomer = false;

        $activeBillingCustomerId = session()->get('b2b.activeBillingCustomerId', false);
        if ($activeBillingCustomerId) {
            $this->activeBillingCustomer = Customer::with('segments.address.plant', 'discounts', 'availableShippingAddresses')->find($activeBillingCustomerId);
        } else {
            $this->activeBillingCustomer = array_get($this->activeCustomer, 'billToCustomer', false);
        }
    }

    protected function loadActiveShippingAddress() {
        $this->activeShippingAddress = $this->activeCustomer->default_shipping_address;

        $activeShippingAddressId = session()->get('b2b.activeShippingAddressId', false);
        if ($activeShippingAddressId) {
            foreach ($this->activeCustomer->availableShippingAddresses as $address) {
                if ($address->id == $activeShippingAddressId) {
                    $this->activeShippingAddress = $address;
                    break;
                }
            }
        }
    }

    public function loadLocale() {
        $locale = 'en';
        switch ($this->activeCountry()) {
            case 'GB':
                $locale = 'en_GB';
                break;
        }

        $oldLocale = App::getLocale();
        if ($oldLocale != $locale) {
            Debugbar::addMessage("Setting locale from $oldLocale to $locale");
            App::setLocale($locale);
        }
    }


    protected function loadActiveCarrier() {
        $this->activeCarrier = array_get($this->activeShippingAddress, 'defaultCarrier', false);

        $activeCarrierId = session()->get('b2b.activeCarrierId', false);
        if ($activeCarrierId) {
            $this->activeCarrier = Carrier::find($activeCarrierId);
        }
    }

    protected function loadActiveRate() {
        $this->activeRate = session()->get('b2b.activeRate', false);
    }

    protected function loadActiveWebSilo() {
        $this->activeWebSilo = false;

        $activeWebSiloId = session()->get('b2b.activeWebSiloId', false);

        if (!$activeWebSiloId) {
            $user = auth()->user();
            if ($user) {
                $userSilo = $user->defaultWebSilo;
                if ($userSilo) {
                    $activeWebSiloId = $userSilo->id;
                } else if ($this->activeCustomer) {
                    $defaultWebSilo = $this->activeCustomer->webSilo;
                    if ($defaultWebSilo) {
                        $activeWebSiloId = $defaultWebSilo->id;
                    }
                }
            }
            if (!$activeWebSiloId) {
                $query = WebSilo::with('webLandingPageWebSiloJoins');

                if ($this->activeCustomer) {
                    $cust_cat_summary = $this->activeCustomer->customerCategory->cust_cat_summary;
                    $divisions = $this->activeCustomer->divisions->pluck('id');
                    $query->whereHas('divisionRules', function ($query) use ($divisions) {
                        $query->whereIn('Divisions.id', $divisions);
                    })->whereHas('customerCategorySummaryRules', function ($query) use ($cust_cat_summary) {
                        $query->where('cust_cat_summary', $cust_cat_summary);
                    });
                } else {
                    $webUrl = WebUrl::current()->first();
                    if (!is_null($webUrl)) {
                        $query->whereHas('webUrl', function ($query) use ($webUrl) {
                            $query->where('id', '=', $webUrl->id);
                        });
                    }
                }
                $result = $query->first();
                if (!is_null($result)) $activeWebSiloId = $result->id;
            }

            if (!$activeWebSiloId) $activeWebSiloId = WebSilo::first()->id;
        }
        $this->activeWebSilo = WebSilo::with('webLandingPageWebSiloJoins')->find($activeWebSiloId);
        session()->put('b2b.activeWebSiloId', $activeWebSiloId);

        return $this->activeWebSilo;
    }

    public function aliasAs($customerId = false) {
        session()->forget('b2b');

        if ($customerId) {
            session()->put('b2b.aliasedCustomerId', $customerId);
        }

        $this->loadActiveCustomer();
    }

    public function isAliased() {
        return (session()->get('b2b.aliasedCustomerId', false));

    }

    public function stopAliasing() {
        $this->aliasAs(false);
    }

    public function setActiveShippingAddress($activeShippingAddressId = false) {
        if ($activeShippingAddressId) {
            session()->put('b2b.activeShippingAddressId', $activeShippingAddressId);
        } else {
            session()->forget('b2b.activeShippingAddressId');
        }
        $this->loadActiveShippingAddress();
    }

    public function setActiveBillingCustomer($billingCustomerId = false) {
        if ($billingCustomerId) {
            session()->put('b2b.activeBillingCustomerId', $billingCustomerId);
        } else {
            session()->forget('b2b.activeBillingCustomerId');
        }
        $this->loadActiveBillingCustomer();
    }

    public function setActiveCarrier($activeCarrierId = false) {
        if ($activeCarrierId) {
            session()->put('b2b.activeCarrierId', $activeCarrierId);
        } else {
            session()->forget('b2b.activeCarrierId');
        }
        $this->loadActiveCarrier();
    }

    public function setActiveRate($activeRate = false) {
        if ($activeRate) {
            session()->put('b2b.activeRate', $activeRate);
        } else {
            session()->forget('b2b.activeRate');
        }
        $this->activeRate = $activeRate;
    }

    public function activeCustomer() {
        return $this->activeCustomer;
    }

    public function activeBillingCustomer() {
        return $this->activeBillingCustomer;
    }

    public function activeShippingAddress() {
        return $this->activeShippingAddress;
    }

    public function activeCarrier() {
        return $this->activeCarrier;
    }

    /**
     * @return Rate
     */
    public function activeRate() {
        return $this->activeRate;
    }

    public function activeCountry() {
        if ($this->activeShippingAddress) return $this->activeShippingAddress()->country;

        return null;
    }

    public function activePriceList() {
        $activePriceList = 'Catalog';
        if ($this->activeCustomer) {
            $customerPriceList = $this->activeCustomer->default_price_list;
            if ($customerPriceList) $activePriceList = $customerPriceList;
        }

        return $activePriceList;
    }

    public function activeCurrency() {
        $activeCurrency = 'USD';

        if ($this->activeCustomer) {
            $activeCurrency = $this->activeCustomer->currency;
        }

        return $activeCurrency;
    }

    public function activeWebSilo() {
        return $this->activeWebSilo;
    }

    public function currencySymbol($currencyCode = false) {
        if ($currencyCode === false) $currencyCode = $this->activeCurrency();
        if ($currencyCode == 'GBP') return '£';

        return '$';
    }

    public function formatPrice($price, $decimals = 2, $symbol = false) {
        if ($symbol === false) $symbol = $this->currencySymbol();
        if ($decimals == -1) {
            if ($price == round($price)) $decimals = 0;
            else $decimals = 2;
        }
        return $symbol . number_format($price, $decimals);
    }

    public function dateFormatString($division = false) {
        if ($division === false) $division = $this->activeCustomer()->activeSegment->division;

        $format = 'm/d/Y';
        switch ($division) {
            case 'HILCO-UK':
                $format = 'd/m/Y';
                break;
            case 'HILCO-US':
            case 'WILSON-US':
                $format = 'm/d/Y';
        }

        return $format;
    }

    public function isOrderable(Model $model) {
        if ($model instanceof WebFamily) return $this->isWebFamilyOrderable($model);
        else if ($model instanceof WebPart) return $this->isWebPartOrderable($model);
        else if ($model instanceof Part) return $this->isPartOrderable($model);

        return false;
    }

    public function isWebFamilyOrderable(WebFamily $webFamily) {
        $isOrderable = true;
        foreach ($webFamily->visibleWebParts as $webPart) {
            if (!$this->isWebPartOrderable($webPart)) $isOrderable = false;
        }
        return $isOrderable;
    }

    public function isWebPartOrderable(WebPart $webPart) {
        $part = $webPart->part;
        if (is_null($part)) return false;
        return $this->isPartOrderable($part);
    }

    public function isPartOrderable(Part $part) {
        if ($this->activeCountry() != 'US' && $this->activeCountry() != 'UK') {
            if ($part->isRX) return false;
        }
        return true;
    }

    public function allowCreditCardOrders() {
        return (bool) $this->activeWebSilo->allow_credit_cards && $this->isBillingDirect() && strlen(array_get($this->authorizeNetCredentials(), 'clientKey', ''));
    }

    public function allowDirectBilling() {
        return (bool) $this->activeWebSilo->allow_direct_billing;
    }

    public function allowGroupBilling() {
        return (bool) $this->activeWebSilo->allow_group_billing;
    }

    public function requireApprovalSiloParts() {
        return (bool) array_get($this->activeCustomer,'webSilo.pivot.require_approval_silo_parts', false);
    }

    public function requireApprovalHilcoParts() {
        return (bool) array_get($this->activeCustomer,'webSilo.pivot.require_approval_hilco_parts', false);
    }

    public function allowPOOrders() {
        $terms = array_get($this->activeBillingCustomer, 'activeSegment.def_pmt_terms');
        return !in_array($terms, ['CR CARD', 'CR CARD+BAL']);
    }

    public function authorizeNetCredentials() {
        $clientKey = '';
        $apiLoginID = '';

        try {
            $credentials = AuthorizeNetWrapper::getCredentials($this->activeBillingCustomer->activeSegment->division, $this->activeBillingCustomer->currency);
            $clientKey = $credentials['CLIENT_KEY'];
            $apiLoginID = $credentials['API_LOGIN'];
        } catch (\AuthorizeNetException $e) {

        }

        return ['clientKey' => $clientKey, 'apiLoginId' => $apiLoginID];
    }

    public function creditProfiles() {
        $creditProfiles = [];
        try {
            $authorizeNet = AuthorizeNetWrapper::create($this->activeBillingCustomer->activeSegment->division, $this->activeBillingCustomer->currency);
            $customerCreditProfile = CustomerCreditProfile::where('customer_id', $this->activeCustomer->id)
                ->whereNotNull('profileId')
                ->whereRaw('LENGTH(`profileId`) > 0')
                ->where('deleted_at', '0000-00-00 00:00:00')
                ->first();
            if ($customerCreditProfile) {
                foreach ($authorizeNet->getCreditProfiles($authorizeNet->getCustomerProfile($customerCreditProfile->profileId)) as $creditProfile) {
                    $creditProfiles[] = [
                        'id' => $creditProfile->getCustomerPaymentProfileId(),
                        'text' => $creditProfile->getPayment()->getCreditCard()->getCardNumber(). ' - ' .$creditProfile->getPayment()->getCreditCard()->getExpirationDate(),
                        'last4' => substr($creditProfile->getPayment()->getCreditCard()->getCardNumber(), -4),
                        'expDate' => $creditProfile->getPayment()->getCreditCard()->getExpirationDate(),
                        'cardType' => $creditProfile->getPayment()->getCreditCard()->getCardType(),
                    ];
                }
                $creditProfiles[] = ['id' => 0, 'text' => 'Use new Card'];
            }
        } catch (\AuthorizeNetException $e) {

        }

        return $creditProfiles;
    }

    public function isRewardsEnabled() {
        if (!config('rewards.rewardsEnabled')) return false;
        return (bool) $this->activeWebSilo->allow_rewards;
    }

    public function isBillingDirect() {
        return $this->activeCustomer->id == $this->activeBillingCustomer->id;
    }

    public function isCustomerTaxable() {
        return array_get($this->activeBillingCustomer, 'activeSegment.cust_txbl', false);
    }

    public function availableShippingAddresses() {
        $defaultAddress = $this->activeCustomer->default_shipping_address;
        return $this->activeCustomer->availableShippingAddresses->sort(function ($a, $b) use ($defaultAddress) {
            if ($a->id == $defaultAddress->id) return -1;
            if ($b->id == $defaultAddress->id) return 1;

            return strcmp($a->cust_name, $b->cust_name);
        });
    }

    /**
     * @return Collection|Rate[]
     */
    public function rateShop() {
        if (b2b()->activeWebSilo()->allow_rate_shopping) {
            $oldRates = $this->oldRateShop();
        } else {
            $oldRates = [];
        }

        $carrierCodes = [];
        foreach ($oldRates as $oldRate) {
            $code = array_get($oldRate, 'hilco_carrier_code', false);
            if ($code) {
                $carrierCodes[$code] = $code;
            }
        }
        $carriers = Carrier::codes($carrierCodes)->get()->keyBy('carrier_code');

        $activeRate = $this->activeRate();
        $activeRateCode = ($activeRate) ? $activeRate->code() : false;
        $wasRateSelected = false;

        $rates = new Collection();
        foreach ($oldRates as $oldRate) {
            $code = array_get($oldRate, 'hilco_carrier_code', false);
            if ($code === false) continue;
            $carrier = array_get($carriers, $code);
            $rate = array_get($oldRate, 'rate', 0);
            $isDefault = array_get($oldRate, 'is_default', false);
            $discount = array_get($oldRate, 'discount_amount', 0);
            $tax = array_get($oldRate, 'tax_amount', 0);

            $rate = new Rate($carrier, $rate, $isDefault, $discount, $tax);
            $rate->setText(array_get($oldRate, 'carrier_desc'));

            if ($code == $activeRateCode) {
                $rate->selected = true;
                $wasRateSelected = true;
            }
            $rates->put($code, $rate);
        }

        $rates = $rates->sort(function (Rate $a, Rate $b) {
            $aSpecial = $a->isDefault() || $a->isDiscounted();
            $bSpecial = $b->isDefault() || $b->isDiscounted();

            if ($aSpecial && !$bSpecial) return -1; //a is special but b isn't, a goes first
            if (!$aSpecial && $bSpecial) return 1; //b is special but a isn't, b goes first

            //both or neither are special
            if ($a->rate() > $b->rate()) return 1;
            if ($a->rate() < $b->rate()) return -1;

            return 0;
        });

        if ($rates->count() == 0) {
            $rates->put('-1', new Rate(null, -1, true));
        }

        if ($activeRateCode === false || $wasRateSelected === false) {
            $firstRate = $rates->shift();
            $firstRate->selected = true;
            $this->setActiveRate($firstRate);
            $rates->prepend($firstRate, $firstRate->code());
        }

        return $rates;
    }

    protected function oldRateShop() {
        $shippingAddress = $this->activeShippingAddress;
        $countryString = $shippingAddress->country;
        $groupCode = $countryString != null && $countryString == "CA" ? 2 : 3;
        $fromCode = $shippingAddress->plant->plantAddress->zip;
        $defaultCarrierCode = $shippingAddress->def_carr_cd;

        $cartStatus = App::make('cartSingleton')->getCartStatus();
        $cartItems = $cartStatus['items'];
        $cartTotal = 0;
        $weights = [];
        $volume = 0;
        $hideAir = false;
        foreach ($cartItems as $cartItem) {
            $quantity = $cartItem['quantity'];
            $part = $cartItem['part'];

            $key = 0;
            if ($shippingAddress->def_ship_from == 'MONTREAL') {
                $inventoryItem = $part->inventoryItems->where('plant','MONTREAL')->first();
                if (isset($inventoryItem) && $inventoryItem->inStock) {
                    $key = 1;
                }
            }

            if (!isset($weights[$key])) {
                $weights[$key] = $part->weight_lbs * $quantity;
            } else {
                $weights[$key] += $part->weight_lbs * $quantity;
            }

            $volume += $part->vol_ci * $quantity;
            $hideAir = $hideAir || ($part->inventoryItems->where('plant_id', $shippingAddress->default_shipfrom_id)->count() > 0 && $part->inventoryItems->where('plant_id', $shippingAddress->default_shipfrom_id)->first()->hazmat_code === 'ORM-D');
            $cartTotal += $cartItem['extendedPrice'];
        }

        if (count($weights) > 0) {
            if ($shippingAddress->plant->plant == 'AUSTRALIA') {
                $rateArray = [];
                $weightKg = $weights[0] * 0.45359237;
                $roundedWeight = ceil($weightKg);
                if ($weightKg <= 0.5) $roundedWeight = 0.5;

                if ($roundedWeight > 0.5) {
                    //ROAD
                    $destinationPostCode = array_get($shippingAddress, 'postal_cd');
                    $zone = array_get(AustraliaFreightPostCode::postCode($destinationPostCode)->first(), 'australiaFreightZone');

                    if (!is_null($zone)) {
                        $rateType = array_get($zone, 'rate_type');
                        if ($rateType == 'Quantity') {
                            $quantityRate = array_get($zone, 'unit_charge');
                            if ($quantityRate) {
                                $rateArray[] = [
                                    'id' => 'A364',
                                    'api_carrier' => null,
                                    'residential' => null,
                                    'hilco_carrier_code' => 'A364',
                                    'carrier_desc' => 'Australia Road Freight',
                                    'text' => 'Australia Road Freight - ' . $this->formatPrice($quantityRate),
                                    'rate' => $quantityRate,
                                    'rate_string' => number_format($quantityRate, 2),
                                    'service' => null,
                                    'delivery_date' => null,
                                    'delivery_time' => null,
                                    'guaranteed' => null,
                                    'base_rate' => $quantityRate,
                                    'base_rate_string' => number_format($quantityRate, 2),
                                    'tax_amount' => round($quantityRate * .123, 2),
                                ];
                            }
                        } else if ($rateType == 'Weight') {
                            $minimumCharge = array_get($zone, 'min_charge');
                            $baseCharge = array_get($zone, 'base_charge');
                            $unitCharge = array_get($zone, 'unit_charge');
                            $weightCharge = max(($unitCharge * $roundedWeight) + $baseCharge, $minimumCharge);
                            if ($weightCharge) {
                                $rateArray[] = [
                                    'id' => 'A364',
                                    'api_carrier' => null,
                                    'residential' => null,
                                    'hilco_carrier_code' => 'A364',
                                    'carrier_desc' => 'Australia Road Freight',
                                    'text' => 'Australia Road Freight - ' . $this->formatPrice($weightCharge),
                                    'rate' => $weightCharge,
                                    'rate_string' => number_format($weightCharge, 2),
                                    'service' => null,
                                    'delivery_date' => null,
                                    'delivery_time' => null,
                                    'guaranteed' => null,
                                    'base_rate' => $weightCharge,
                                    'base_rate_string' => number_format($weightCharge, 2),
                                    'tax_amount' => round($weightCharge * .123, 2),
                                ];
                            }
                        }
                    }
                }

                if ($roundedWeight <= 5) {
                    if ($roundedWeight <= 0.5) {
                        $overnightRate = 8.34;
                    } else if ($roundedWeight <= 1) {
                        $overnightRate = 9.59;
                    } else if ($roundedWeight <= 3) {
                        $overnightRate = 11.41;
                    } else if ($roundedWeight <= 5) {
                        $overnightRate = 19.08;
                    }
                    $rateArray[] = [
                        'id' => 'A360',
                        'api_carrier' => null,
                        'residential' => null,
                        'hilco_carrier_code' => 'A360',
                        'carrier_desc' => 'Australia Overnight',
                        'text' => 'Australia Overnight - ' . $this->formatPrice($overnightRate),
                        'rate' => $overnightRate,
                        'rate_string' => number_format($overnightRate, 2),
                        'service' => null,
                        'delivery_date' => null,
                        'delivery_time' => null,
                        'guaranteed' => null,
                        'base_rate' => $overnightRate,
                        'base_rate_string' => number_format($overnightRate, 2),
                        'tax_amount' => round($overnightRate * .123, 2),
                    ];
                }

                if ($this->isCustomerTaxable()) {
                    $additionalTaxCustomerSegments = array_get($this->activeSegment(), 'additionalTaxCustomerSegment', []);

                    foreach ($rateArray as $index => $rate) {
                        $rateAmount = array_get($rate, 'rate', 0);
                        $taxAmount = array_get($rate, 'tax_amount', 0);

                        foreach ($additionalTaxCustomerSegments as $additionalTaxCustomerSegment) {
                            foreach (array_get($additionalTaxCustomerSegment, 'additionalTax', []) as $additionalTax) {
                                $tax = round($rateAmount * (array_get($additionalTax, 'tax_percentage', 0) / 100), 2);
                                $taxAmount += $tax;
                            }
                        }
                        $rateArray[$index]['tax_amount'] = $taxAmount;
                    }
                }

                return $rateArray;
            } else if ($countryString == 'GB') {
                if ($weights[0] < 2.20462) { //under 1kg
                    $flatRate = 2.50;
                    $flatCarrierCode = 'UKH1';
                } else if ($weights[0] < 44.0925) { //under 20kg
                    $flatRate = 5.70;
                    $flatCarrierCode = 'UKH2';
                } else {
                    return [['id' => -1, 'text' => 'Unable to calculate shipping rates.']];
                }

                $shippingTaxAmount = 0;
                if ($this->isCustomerTaxable()) {
                    $shippingVatRate = VATTax::standardRate();
                    $shippingTaxAmount = round($flatRate * ($shippingVatRate / 100), 2);
                }

                return [
                    [
                        'id' => $flatCarrierCode,
                        'api_carrier' => null,
                        'residential' => null,
                        'hilco_carrier_code' => $flatCarrierCode,
                        'carrier_desc' => 'Flat Rate Shipping',
                        'text' => 'Flat Rate Shipping - ' . $this->formatPrice($flatRate),
                        'rate' => $flatRate,
                        'rate_string' => number_format($flatRate, 2),
                        'service' => null,
                        'delivery_date' => null,
                        'delivery_time' => null,
                        'guaranteed' => null,
                        'base_rate' => $flatRate,
                        'base_rate_string' => number_format($flatRate, 2),
                        'tax_amount' => $shippingTaxAmount,
                    ]
                ];
            } else if ($groupCode == 3) {
                $carrierCodes = DB::connection()
                    ->table('RateShopGroups')
                    ->where('group_code', '=', $groupCode)
                    ->pluck('hilco_carrier_code');

                $isDefaultNewRow = false;
                $defaultCarrier = null;
                if ($shippingAddress->def_carr_cd != null) {
                    if (!in_array($shippingAddress->def_carr_cd, $carrierCodes)) {
                        $defaultCarrier = DB::connection()
                            ->table('Carriers')
                            ->leftJoin('RateShopAPIDetails', 'Carriers.carrier_code', '=', 'RateShopAPIDetails.hilco_carrier_code')
                            ->where('carrier_code', '=', $shippingAddress->def_carr_cd)
                            ->select()
                            ->first();

                        // Yes this is intentional. No it isn't our data. Yes if they decide to fix it without telling us it would be bad.
                        if ($defaultCarrier->bill == 'Hico' || $defaultCarrier->bill == 'Hilco') {
                            $carrierCodes[] = $shippingAddress->def_carr_cd;
                        } else {
                            $isDefaultNewRow = true;
                        }
                    }
                }

                if ($shippingAddress->def_carr_cd != null && ($shippingAddress->def_carr_cd == '1400' || (isset($defaultCarrier->super_code) && $defaultCarrier->super_code == '1400'))) {
                    if (($key = array_search('1000', $carrierCodes)) !== false) {
                        unset($carrierCodes[$key]);
                    }
                } else {
                    if (($key = array_search('1400', $carrierCodes)) !== false) {
                        unset($carrierCodes[$key]);
                    }
                }

                if ($hideAir) {
                    $airCodes = DB::connection()
                        ->table('RateShopAPIDetails')
                        ->whereIn('hilco_carrier_code', $carrierCodes)
                        ->where('is_air', '=', true)
                        ->select('hilco_carrier_code')
                        ->get();

                    foreach ($airCodes as $airCode) {
                        if (($key = array_search($airCode->hilco_carrier_code, $carrierCodes)) !== false) {
                            unset($carrierCodes[$key]);
                        }
                    }
                }

                try {
                    $shipVia = RateShop::rateShopUSAWithAccount($carrierCodes, $weights[0], $volume, $shippingAddress->state, $shippingAddress->postal_cd, $fromCode);
                } catch (\Exception $e) {
                    Log::critical('Exception caught while getting shipping rates: ' . $e->getMessage(), $e->getTrace());
                }


                if ($isDefaultNewRow) {
                    if (isset($defaultCarrier->super_code) && isset($defaultCarrier->code_type)) {
                        if ($shippingAddress->def_ship_via != null && strlen($shippingAddress->def_ship_via) > 0) {
                            $codes = DB::connection()
                                ->table('RateShopAPIDetails as myCode')
                                ->join('RateShopAPIDetails as otherCode', 'myCode.code_type', '=', 'otherCode.code_type')
                                ->join('Carriers', 'Carriers.carrier_code', '=', 'otherCode.hilco_carrier_code')
                                ->where('myCode.hilco_carrier_code', '=', $shippingAddress->def_carr_cd)
                                ->whereIn('otherCode.super_code', $carrierCodes)
                                ->select(['otherCode.hilco_carrier_code', 'Carriers.carrier_desc', 'otherCode.super_code'])
                                ->get();

                            foreach ($codes as $code) {
                                $shipVia[$code->hilco_carrier_code] = [
                                    'api_carrier' => null,
                                    'residential' => null,
                                    'hilco_carrier_code' => $code->hilco_carrier_code,
                                    'carrier_desc' => $code->carrier_desc,
                                    'rate' => 2,
                                    'text' => $code->carrier_desc . ' - ' . $this->formatPrice(2),
                                    'service' => null,
                                    'delivery_date' => null,
                                    'delivery_time' => null,
                                    'guaranteed' => null,
                                    'def_ship_via' => $shippingAddress->def_ship_via,
                                    'is_default' => $code->hilco_carrier_code == $defaultCarrierCode,
                                ];
                            }
                        }
                    } else {
                        $shipVia[$shippingAddress->def_carr_cd] = [
                            'api_carrier' => null,
                            'residential' => null,
                            'hilco_carrier_code' => $shippingAddress->def_carr_cd,
                            'carrier_desc' => $defaultCarrier->carrier_desc,
                            'rate' => 2,
                            'text' => $defaultCarrier->carrier_desc . ' - ' . $this->formatPrice(2),
                            'service' => null,
                            'delivery_date' => null,
                            'delivery_time' => null,
                            'guaranteed' => null,
                            'def_ship_via' => $shippingAddress->def_ship_via,
                            'is_default' => true,
                        ];
                    }
                }
            } else {
                $shipmentArray = [];
                foreach ($weights as $weight) {
                    $shipmentArray[] = ['weight' => $weight, 'volume' => $volume, 'toCode' => $shippingAddress->postal_cd, 'fromCode' => $fromCode];
                }
                $shipVia = RateShop::rateShopCanada($groupCode, $shipmentArray);
            }
        }

        if(!isset($shipVia)) $shipVia = [];
        foreach ($shipVia as $key => $value) {
            $shipVia[$key]['id'] = $value['hilco_carrier_code'];
            if (isset($value['rate']) && $value['rate'] > 0) {
                $shipVia[$key]['rate_string'] = number_format($value['rate'], 2);
                $shipVia[$key]['text'] = $value['carrier_desc'] . ' - ' . $this->formatPrice($value['rate']);
                $shipVia[$key]['is_default'] = $value['hilco_carrier_code'] == $defaultCarrierCode;
            } else {
                if (is_null($shipVia[$key]['rate'])) unset($shipVia[$key]);
            }
        }
        $baseRate = array_get($shipVia, '1000.rate', 0);
        $rewardsCarrierCode = rewards()->carrierCode($cartTotal, $baseRate, $cartStatus['rewardsTier']);
        if ($rewardsCarrierCode && $baseRate > 0) {
            $shipVia[$rewardsCarrierCode['code']] = [
                'id' => $rewardsCarrierCode['code'],
                'api_carrier' => null,
                'residential' => null,
                'hilco_carrier_code' => $rewardsCarrierCode['code'],
                'carrier_desc' => $rewardsCarrierCode['description'],
                'rate' => $baseRate,
                'rate_string' => number_format($rewardsCarrierCode['rate'], 2),
                'text' => $rewardsCarrierCode['description'] . ' - ' . $this->formatPrice($rewardsCarrierCode['rate']),
                'service' => null,
                'delivery_date' => null,
                'delivery_time' => null,
                'guaranteed' => null,
                'base_rate' => $baseRate,
                'base_rate_string' => number_format($baseRate, 2),
                'discount_amount' => $baseRate - $rewardsCarrierCode['rate'],
                'is_default' => $rewardsCarrierCode['code'] == $defaultCarrierCode,
            ];
        }

        if (count($shipVia) == 0) {
            $shipVia[0] = ['id' => -1, 'text' => 'Unable to calculate shipping rates.'];
        }

        return array_values($shipVia);
    }

    public function hasSentFeedback() {
        $user = auth()->user();
        $has = ($user) ? auth()->user()->webFeedbackResponses()->count() > 0 : false;
        return $has;
    }

    public function getShipmentRates(CustomerShippingAddress $customerShippingAddress) {
        $cartComposer = App::make('cartSingleton');
        $cartStatus = $cartComposer->getCartStatus();
        $rateType = array_get($this->activeWebSilo, 'shipment_rate_type', 'api');
        $rates = new Collection;

        if ($rateType == 'api'){
            $carriers = array_get($this->activeWebSilo, 'carriers', []);
            $package = new Package(array_get($cartStatus, 'totalWeight', 0), array_get($cartStatus, 'totalVolume', 0), $customerShippingAddress->postal_cd, $customerShippingAddress->country);
            foreach ($carriers as $carrier) {

            }
        } else {
            $rates = array_get($this->activeWebSilo, 'shipmentFlatRates', false);
            krsort($rates);
            $cartValue = 0;
            if ($rateType == 'flatByWeight') $cartValue = array_get($cartStatus, 'totalWeight', 0);
            if ($rateType == 'flatByPrice') $cartValue = array_get($cartStatus, 'discountedTotal', 0);
            foreach ($rates as $threshold => $rate) {
                if ($cartValue >= $threshold) {
                    $rates->push([
                        'rate' => $rate,
                    ]);
                    break;
                }
            }
        }

        return $rates;
    }

    public function getAPIShipmentRates($cartStatus, $customerShippingAddress) {

    }

    public function allowBannerCarousel($lpModel) {
        if ($this->activeWebSilo->id == '2') {
            if ($lpModel instanceof WebGroup || $lpModel instanceof WebCategory || $lpModel instanceof WebCollection) {
                return false;
            }
        }
        return true;
    }

    /**
     * Retrieves product models from an array of productType, productId pairs
     *
     * @param $products
     */
    public function retrieveProductModels($products) {
        $idsByType = [];
        foreach ($products as $product) {
            $type = array_get($product, 'productType');
            $id = array_get($product, 'productId');
            array_set($idsByType, "$type.$id", $id);
        }

        $map = Relation::morphMap();
        $models = collect();
        foreach ($idsByType as $type => $ids) {
            $type = normalizeProductType($type);
            $className = array_get($map, $type, false);
            if ($className === false) continue;
            $classModels = $className::findMany($ids);
            foreach ($classModels as $model) {
                $models->push($model);
            }
        }

        return $models;
    }

    public function activeSegment() {
        $activeCustomer = $this->activeCustomer();

        return array_get($activeCustomer, 'activeSegment');
    }

    public function activePlant() {
        $activePlant = null;
        $activeSegment = $this->activeSegment();

        $activePlant = array_get($activeSegment, 'address.plant', Plant::defaultPlant());

        return $activePlant;
    }

    public function defaultRole() {
        $defaultRoleSlug = config('hilco.defaultWebRoleSlug');
        $webRole = WebRole::where('slug', '=', $defaultRoleSlug)->first();

        return $webRole;
    }
}