<?php

namespace Hilco\Models;

use DB;
use Illuminate\Database\Eloquent\Builder;

class WebGroup extends WebModel implements HasLandingPageInterface, HasWebSiloOverridesInterface {
	protected $table = "WebGroups";
	protected $fillable = ['name', 'slug', 'webHierarchyIds' ,'is_visible', 'detail', 'metadata_title', 'metadata_description', 'is_menubar', 'commit_sequence'];
	protected $casts = [
	    'is_visible' => 'boolean',
        'layout' => 'array',
	];
    protected $with = ['webSiloOverride'];

    public static function maxCommit(): int {
        return WebGroup::withTrashed()->max("commit_sequence");
    }

    public static function nextCommit(): int {
        return WebGroup::maxCommit() + 1;
    }

    /**
     * @param array $webgroupData
     * @param array $webhierarchies
     * @param array $webcategories
     * @return WebGroup
     */
    public static function createWebGroup (array $webgroupData, array $webhierarchies = [], array $webcategories = []) {
        if (! isset($webgroupData['commit_sequence'])) {
            $webgroupData['commit_sequence'] = WebGroup::nextCommit();
        }
        $newWebGroup = new WebGroup($webgroupData);
        $newWebGroup->save();
        $newWebGroup->refresh();
        if (count($webhierarchies)) {
            // ok to use sync() here because 1. it's create, and 2. cst doesn't use webgroup_webhierarchy
            $newWebGroup->webHierarchies()->sync($webhierarchies);
        }
        if (count($webcategories)) {
            $wcwgCS = WebCategory_WebGroup::withTrashed()->max('commit_sequence') + 1;
            $wcQuery = WebCategory::whereIn('id', $webcategories)
                ->selectRaw("WebCategories.id AS webcategory_id, $newWebGroup->id AS webgroup_id, $wcwgCS AS commit_sequence");
            WebCategory_WebGroup::insertUsing([
                'webcategory_id', 'webgroup_id', 'commit_sequence'
            ], $wcQuery);
        }
        return $newWebGroup;
    }

    public static function deleteWebGroup ($id) {
        $webGroup = WebGroup::find($id);
        $webGroup->delete();
        WebGroup_WebHierarchy::where('webgroup_id', $id)->delete();
        WebCategory_WebGroup::where('webgroup_id', $id)->delete();
    }

    protected $hierarchyParent = false;

    public function webHierarchies() {
        return $this->belongsToMany(WebHierarchy::class, 'WebGroup_WebHierarchy', 'webgroup_id', 'webhierarchy_id')
            ->wherePivot('deleted_at', '=', '0000-00-00 00:00:00');
    }

    public function webGroupWebHierarchies() {
        return $this->hasMany(WebGroup_WebHierarchy::class, "webgroup_id", "id");
    }

	public function webCategories() {
		return $this->belongsToMany(WebCategory::class, 'WebCategory_WebGroup', 'webgroup_id', 'webcategory_id')
		    ->wherePivot('deleted_at', '=', '0000-00-00 00:00:00')
		    ->withPivot('display_order');
    }

    public function webCategoryWebGroups() {
        return $this->hasMany(WebCategory_WebGroup::class, "webgroup_id", "id");
    }

	public function webTranslations(){
        return $this->hasMany(Translations_WebGroup::class, 'webgroup_id', 'id');
    }

    public function webProductMenuConfiguration(){
        return $this->hasOne(WebProductMenuConfiguration::class, 'product_id', 'id')->whereRaw('product_type = "webGroup"');
    }

    public function visibleWebCategories() {
        return $this->belongsToMany(WebCategory::class, 'WebCategory_WebGroup', 'webgroup_id', 'webcategory_id')
            ->wherePivot('deleted_at', '=', '0000-00-00 00:00:00')
            ->withPivot('display_order', 'is_featured')
            ->webVisible()
            ->displayOrder()
        ;
    }

    private $cachedWebProductMenuConfiguration = false;

    public function getWebProductMenuConfiguration(){
        //first time webProductMenuConfiguration is loaded, cache it so we don't need to re-do the rendering logic and slowdown page loads unnecessarily
        if ($this->cachedWebProductMenuConfiguration === false) {
            if (isset($this->webSiloOverride) &&
                $this->webSiloOverride->is_visible &&
                !is_null($this->webSiloOverride->webproductmenuconfiguration_id)){
                $this->cachedWebProductMenuConfiguration = $this->webSiloOverride->webProductMenuConfiguration->getForCustomer(b2b()->activeCustomer());
            }else{
                $this->cachedWebProductMenuConfiguration = $this->webProductMenuConfiguration;
                if ($this->cachedWebProductMenuConfiguration) $this->cachedWebProductMenuConfiguration = $this->cachedWebProductMenuConfiguration->getForCustomer(b2b()->activeCustomer());
            }    
        }
        return $this->cachedWebProductMenuConfiguration;
    }

    public function scopeDisplayOrder(Builder $query) {
        return $query->orderBy('pivot_display_order', 'asc')->orderBy('name', 'asc');
    }

    public function scopeAlphabetical(Builder $query) {
        return $query->orderBy('name', 'asc');
    }

	public function webCategoriesOrdered() {
        return $this->webCategories()->orderBy('pivot_display_order')->orderBy('name')
            ;
    }

	public function webLandingPages() {
		return $this->belongsToMany(WebLandingPage::class, 'WebGroup_WebLandingPage', 'webgroup_id', 'weblandingpage_id');
	}

    public function webAttributes() {
        return $this->belongsToMany(WebAttribute::class, 'WebAttribute_WebGroup', 'webgroup_id', 'webattribute_id')->wherePivot('deleted_at', '=', '0000-00-00 00:00:00')->withPivot(['display_order']);
    }

	public function getFeaturedChildrenAttribute() {
        return $this->visibleWebCategories()->wherePivot('is_featured', 1)->get();
    }

    public function getChildrenAttribute() {
        $children = $this->visibleWebCategories();
        if (request()->get('ap', false) == 1) $children->webSiloApproved();

        return $children->get();
    }

    public function getDisplayNameForUACAttribute() {
        return $this->attributes['name'];
    }

    public function getNameAttribute() {
        if (config('hilco.ignoreActiveWebSilo')) {
            return $this->getTranslation('name', AvailableLanguage::DEFAULT_LANG_CODE, $this->attributes['name']);
        } else {
            if (isset($this->webSiloOverride) &&
                $this->webSiloOverride->is_visible &&
                !is_null($this->webSiloOverride->name) &&
                trim($this->webSiloOverride->name) != '') {

                return $this->webSiloOverride->name;
            } else {
                return $this->getTranslation('name');
            }
        }
    }

    public function getNameForEditAttribute() {
        return $this->getTranslation('name', AvailableLanguage::DEFAULT_LANG_CODE, $this->attributes['name']);
    }

    public function getDetailAttribute() {
        if (config('hilco.ignoreActiveWebSilo')) {
            return $this->getTranslation('detail', AvailableLanguage::DEFAULT_LANG_CODE, $this->attributes['detail']);
        } else {
            if (isset($this->webSiloOverride) &&
                $this->webSiloOverride->is_visible &&
                !is_null($this->webSiloOverride->detail)) {

                return $this->webSiloOverride->detail;
            } else {
                return $this->getTranslation('detail');
            }
        }
    }

    /**
     * @deprecated this is technically only "ok" to do when creating a new WebGroup, otherwise it won't soft-delete
     */
    public function setWebHierarchyIdsAttribute($ids) {
        $this->webHierarchies()->sync($ids);
	}

	public function scopeHasVisibleChildren(Builder $query, $activeWebSilo = false, $activeCountry = false, $activePlant = false) {
        return $query->where(function ($where) {
            return $where
                ->whereDoesntHave('webSiloOverride')
                ->orWhereHas('webSiloOverride', function ($query) {
                    return $query->whereRaw('WebSiloOverrides.is_visible = 1');
//                        ->where('WebSiloOverrides.websilo_id', '=', b2b()->activeWebSilo()->id)
//                        ->where('WebSiloOverrides.language_code', '=', b2b()->activeLanguage());
                });
            })->whereHas('webCategories', function ($query) use ($activeWebSilo, $activeCountry, $activePlant) {
                return $query->isVisible()->hasVisibleChildren($activeWebSilo, $activeCountry, $activePlant);
            });
    }

    public function scopeIsVisible(Builder $query) {
        return $query
            ->whereRaw('WebGroups.is_visible =1')
            ->where(function ($where) {
                $where
                    ->whereDoesntHave('webSiloOverride')
                    ->orWhereHas('webSiloOverride', function ($query) {
                        return $query->whereRaw('WebSiloOverrides.is_visible = 1');
                    })
                ;
            })
        ;
    }

    public function scopeHasVisibleParents(Builder $query, $activeWebSilo = false) {
        return $query->where(function ($where) {
            return $where
                ->whereDoesntHave('webSiloOverride')
                ->orWhereHas('webSiloOverride', function ($query) {
                    return $query->whereRaw('WebSiloOverrides.is_visible = 1');
//                        ->where('WebSiloOverrides.websilo_id', '=', b2b()->activeWebSilo()->id)
//                        ->where('WebSiloOverrides.language_code', '=', b2b()->activeLanguage());
                });
            })->whereHas('webHierarchies', function ($query) use ($activeWebSilo) {
               return $query->hasVisibleParents($activeWebSilo);
            });
    }

	public function scopeVisible(Builder $query) {
		$query
			->whereRaw('WebGroups.is_visible =1')
            ->has('webHierarchies')
		;
//
//        if (b2b()->activeCountry() != 'US') {
//            $query->where('slug', '!=', 'pharmaceuticals');
//        }

        return $query;
	}


    public function scopeWebVisible(Builder $query) {
        return $query
            ->whereRaw('WebGroups.is_visible =1')
            ->where(function ($where) {
                $where
                    ->whereDoesntHave('webSiloOverride')
                    ->orWhereHas('webSiloOverride', function ($query) {
                        return $query->whereRaw('WebSiloOverrides.is_visible = 1');
                    })
                ;
            })
            ->whereHas('webCategories', function ($webCategoriesQuery) {
                return $webCategoriesQuery->webVisible();
            })
            ->with('webHierarchies')
        ;
    }

    public function scopeWebSiloApproved(Builder $query) {
        return $query
            ->whereHas('webCategories', function ($webCategoriesQuery) {
                return $webCategoriesQuery->webSiloApproved();
            })
            ;
    }

	public function scopeInSilo($query, $webSiloId) {
        return $query->whereHas('webSilos', function ($query) use ($webSiloId) {
            $query->where('websilo_id', $webSiloId);
        });
    }

    public function scopeHasSolrVisibleParents(Builder $query) {
        return $query
            ->whereHas('webHierarchies', function ($query) {
                return $query->hasSolrVisibleParents();
            })
            ;
    }

    public function getHierarchyPathsAttribute() {
        $paths = [];
        foreach ($this->webHierarchies as $webHierarchy) {
            $paths[] = [
                'webHierarchy' => $webHierarchy,
            ];
        }
        return $paths;
    }

    public function visibleInSilo($webSilo, $processDownChain = true, $processUpChain = true) {
        $isCategoryVisible = !$processDownChain;
        $isHierarchyVisible = !$processUpChain;

        if (!$webSilo instanceof WebSilo) $webSilo = WebSilo::find($webSilo);

        if (count($this->webSiloOverrides)) {
            foreach ($this->webSiloOverrides as $webSiloOverride) {
                if ($webSiloOverride->websilo_id == $webSilo->id && $webSiloOverride->is_visible == 0) {
                    return false;
                }
            }
        }

        if($processDownChain){
            foreach ($this->webCategories as $webCategory) {
                if ($webCategory->visibleInSilo($webSilo, true, false)){
                    $isCategoryVisible = true;
                    break;
                }
            }
        }

        if($processUpChain){
            foreach ($this->webHierarchies as $webHierarchy) {
                if ($webHierarchy->id === $webSilo->webhierarchy_id){
                    $isHierarchyVisible = true;
                    break;
                }
            }
        }

        return ($isCategoryVisible && $isHierarchyVisible);
    }

    public function slugUrl($approvedOnly = false) {
	    $params = [$this->slug];
	    if ((request()->get('ap', false) == 1) || $approvedOnly) $params['ap'] = 1;
        return route('group.slug', $params);
    }

    public function getParentRelationName() {
        return false;
    }

    public function getPrimaryImage() {
	    $asset = \Illuminate\Support\Arr::get($this, 'webSiloOverride.primaryImage', null);
        if (!is_null($asset)) return $asset;

        $asset = $this->assetsByType('primary')->first();
        if (!is_null($asset)) return $asset;

        $webCategory = $this->visibleWebCategories->first();
        if ($webCategory) return $webCategory->getPrimaryImage();
    }

    public function getMetadataDescriptionAttribute() {
        $value = $this->getMetadata('description', b2b()->activeLanguage());
        if (strlen($value)) return $value;

        return trim(preg_replace('/(\s*<.*?>\s*)+/', ' ', $this->detail));
    }

    public function getMetadataTitleAttribute() {
        $value = $this->getMetadata('title', b2b()->activeLanguage());
        if (strlen($value)) return $value;

        return $this->name;
    }

    public function scopeInHierarchy ($query, $webhierarchy_id = null) {
        if (is_null($webhierarchy_id)) {
            $webhierarchy_id = b2b()->activeWebHierarchy()->id;
        }

        return $query->whereExists(function ($query) use ($webhierarchy_id) {
            return $query
                ->select(DB::raw('1'))
                ->from('WebGroup_WebHierarchy')
                ->whereRaw('WebGroups.id = WebGroup_WebHierarchy.webgroup_id')
                ->whereRaw('WebGroup_WebHierarchy.deleted_at = 0')
                ->whereRaw("WebGroup_WebHierarchy.webhierarchy_id = $webhierarchy_id");
        });
    }

    use HasAssets, HasSlug, HasLandingPage, HasWebSiloOverrides, HasModelTranslations, HasMetadata;
}