<?php

namespace Hilco\Models;

use Illuminate\Database\Eloquent\Collection;

/**
 * Hilco\Models\WebAttribute
 * @property string $attribute_name
 * @property string $searchbucket_type
 * @property integer $global_search_facet
 * @property boolean $from_syndigo
 * @property string $syndigo_attribute
 * @property Collection|WebPart[] $webParts
 * @property Collection|WebFamily[] $webFamilies
 * @property Collection|WebAttribute_WebPart[] $values
 * @property Collection $searchBuckets
 */
class WebAttribute extends SyndigoAttributeType {
	protected $table = 'WebAttributes';
	protected $fillable = ['attribute_name', 'searchbucket_type', 'global_search_facet', 'from_syndigo', 'syndigo_attribute'];

	public function webParts() {
		return $this->belongsToMany(WebPart::class, 'WebAttribute_WebPart', 'webattribute_id', 'webpart_id')->withPivot(['attribute_value']);
	}

	public function webFamilies() {
		return $this->belongsToMany(WebFamily::class, 'WebAttribute_WebFamily', 'webattribute_id', 'webfamily_id');
	}

	public function values() {
		return $this->hasMany(WebAttribute_WebPart::class, 'webattribute_id', 'id');
	}
	
	public function searchBuckets(){
	    if($this->searchbucket_type == "static"){
	        return $this->hasMany(SearchBucketStatic::class, 'webattribute_id', 'id');
        }else if($this->searchbucket_type == "range"){
            return $this->hasMany(SearchBucketRange::class, 'webattribute_id', 'id');
        }else{
	        return null;
        };
    }
    
	public function getValuesInCategoryAttribute() {
		return $this->webParts->sortBy('pivot.attribute_value')->pluck('pivot.attribute_value', 'pivot.attribute_value');
	}

	public function scopeInCollection($query, $webCollection) {
		$webCollectionId = ($webCollection instanceof WebCollection) ? $webCollection->id : $webCollection;
		return $query->whereHas('webFamilies', function ($query) use ($webCollectionId) {
			return $query->inCollection($webCollectionId);
		});
	}

	public function scopeInCategory($query, $webCategory) {
		$webCategoryId = ($webCategory instanceof WebCategory) ? $webCategory->id : $webCategory;
		return $query->whereHas('webFamilies', function ($query) use ($webCategoryId) {
			return $query->inCategory($webCategoryId);
		});
	}

	public function scopeInGroup($query, $webGroup) {
		$webGroupId = ($webGroup instanceof WebGroup) ? $webGroup->id : $webGroup;
		return $query->whereHas('webFamilies', function ($query) use ($webGroupId) {
			return $query->inGroup($webGroupId);
		});
	}

	public static function in($model) {
		if ($model instanceof WebGroup) {return self::inGroup($model);}
		else if ($model instanceof WebCategory) return self::inCategory($model);
		else if ($model instanceof WebCollection) return self::inCollection($model);
	}

	public function getAttributeNameAttribute() {
        if (config('hilco.ignoreActiveWebSilo')) {
            return $this->getTranslation('attribute_name', AvailableLanguage::DEFAULT_LANG_CODE, $this->attributes['attribute_name']);
        } else {
            return $this->getTranslation('attribute_name');
        }
    }

    public function getOriginalAttributeName() {
        return $this->attributes['attribute_name'];
    }

    public function getDisplayNameForUACAttribute() {
        return $this->getAttributeNameAttribute();
    }

	use HasModelTranslations;

	public function getFacetName(){
        $name = 'attr_webAttribute' . $this->id . '_' . b2b()->activeLanguage();
        return $name;
    }

    public function scopeAttributesByGroup($query, $groupId) {
        return $query
            ->whereHas('webParts', function ($query) use ($groupId) {
                return $query->whereHas('webFamily', function ($query) use ($groupId) {
                    return $query->whereHas('webCollections', function ($query) use ($groupId) {
                        return $query->whereHas('webCategories', function ($query) use ($groupId) {
                            return $query->whereHas('webGroups', function ($query) use ($groupId) {
                                return $query->where('WebGroups.id', '=', $groupId);
                            });
                        });
                    });
                });
            })
        ;
    }

    public function scopeAttributesByCategory($query, $groupId) {
        return $query
            ->whereHas('webParts', function ($query) use ($groupId) {
                return $query->whereHas('webFamily', function ($query) use ($groupId) {
                    return $query->whereHas('webCollections', function ($query) use ($groupId) {
                        return $query->whereHas('webCategories', function ($query) use ($groupId) {
                            return $query->where('WebCategories.id', '=', $groupId);
                        });
                    });
                });
            })
            ;
    }

    public function scopeAttributesByCollection($query, $groupId) {
        return $query
            ->whereHas('webParts', function ($query) use ($groupId) {
                return $query->whereHas('webFamily', function ($query) use ($groupId) {
                    return $query->whereHas('webCollections', function ($query) use ($groupId) {
                        return $query->where('WebCollections.id', '=', $groupId);
                    });
                });
            })
            ;
    }
}
