<?php

namespace Hilco\Models;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Input;
use Venturecraft\Revisionable\RevisionableTrait;

/**
 * Class WebFamily
 * @package Hilco\Models
 * @method static WebFamily firstOrNew(array $attributes)
 */
class WebFamily extends HierarchyModel
{
	protected $table = "WebFamilies";
	protected $fillable = ['name', 'detail', 'slug', 'webCollectionIds', 'webAttributeIds', 'is_visible'];
	protected $context_attributes = ['is_latest', 'is_featured', 'is_visible'];
	protected $context_table = 'WebContext_WebFamily';
	protected $context_foreign_key = 'webfamily_id';
	protected $perPage = 24;
	protected $casts = [
		['is_visible' => 'boolean']
	];

	public function getPerPage()
	{
		$tmpPerPage =Input::get('perPage');
		return (isset($tmpPerPage) && $tmpPerPage > 0)?$tmpPerPage:$this->perPage;
	}
	/**
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function webParts() {
		return $this->hasMany(WebPart::class, 'webfamily_id', 'id');
	}

	/**
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function visibleWebParts() {
		return $this
			->hasMany(WebPart::class, 'webfamily_id', 'id')
			->visible()
			->with('part')
			->with('images')
			->with(['webAttributes'])
			->orderBy('name');
	}

	public function webCollections() {
		return $this->belongsToMany(WebCollection::class, 'WebFamily_WebCollection', 'webfamily_id', 'webcollection_id');
	}

	/**
	 * @return BelongsToMany
     */
	public function webAttributes() {
		return $this
            ->belongsToMany(WebAttribute::class, 'WebAttribute_WebFamily', 'webfamily_id', 'webattribute_id')
            ->withPivot(['is_visible', 'display_order', 'is_thumbnail', 'is_filterable', 'is_select'])
            ->orderBy('pivot_display_order');
	}

	public function getWebAttributeIdsAttribute() {
		return $this->webAttributes->pluck('id')->toArray();
	}

	public function setWebAttributeIdsAttribute($ids) {
		$this->save();
		$this->webAttributes()->sync($ids);
	}

	public function setWebCollectionIdsAttribute($ids) {
		$this->save();
		$this->webCollections()->sync($ids);
	}

//	public function getThumbnailsAttribute() {
//		$attributes = [];
//		foreach ($this->visibleWebParts as $webPart) {
//			foreach ($webPart->webAttributes as $partAttribute) {
//				if ($partAttribute->id == $this->thumbnail_attribute_id) {
//					$attributes[$partAttribute->pivot->attribute_value] = $webPart;
//				}
//			}
//		}
//		return $attributes;
//	}
//
	public function getPriceRangeAttribute() {
		if (is_null($this->webParts)) return null;
		$min = PHP_INT_MAX;
		$max = 0;
		foreach ($this->webParts as $webPart) {
			$min = min($min, $webPart->part->list_price);
			$max = max($max, $webPart->part->list_price);
		}
		return [$min, $max];
	}

	public function getPriceAttribute() {
		if (is_null($this->webParts)) return null;
		return $this->webParts->first()->part->list_price;
	}
//

//	public function setWebCategoryIdsAttribute($value) {
//		$this->webCategories()->sync($value);
//	}


//	public function scopeLatest($query, $take = 8) {
//		return $query->whereHas('context', function($query) {
//			return $query->where('is_featured', true);
//		})->take($take);
//	}

//	public function scopeFeatured($query, $isFeatured = true) {
//		return $query->whereHas('context', function($query) use ($isFeatured) {
//			return $query->where('is_featured', $isFeatured);
//		});
//	}
	public function scopeHasAttributes($query, $filters){
		if(!isset($filters) || is_null($filters)) return $query;
		$attrs = [];
		foreach ($filters as $filter){
			if(!isset($attrs[$filter->webattribute_id])){
				$attrs[$filter->webattribute_id] = [];
			}
			$attrs[$filter->webattribute_id] = array_merge($attrs[$filter->webattribute_id],$filter->webAttributeFilterValues->pluck('filter_value')->toArray());
		}
		return $query->where(function($query)use($attrs){
			return  $query->whereHas('webParts', function ($query) use ($attrs) {
				foreach ($attrs as $id=>$values) {
					$query = $query->whereHas('webAttributes', function ($query) use ($id, $values) {
						return $query->where('webattribute_id', $id)->whereIn('attribute_value', $values);
					});
				}
				return $query;
			});
		});
	}
	public function scopeManagedBy($query, $product_manager) {
		return $query->whereHas('webParts', function($query) use ($product_manager) {
			return $query->managedBy($product_manager);
		});
	}

	public function getImageLink($width = null, $height = null) {
		$part = $this->visibleWebParts()->has('images')->first();
		if (is_null($part)) {
            $url = config('hilco.images.url');
            $url .= '/nopic';
            if ($width) $url .= "/$width";
            if ($height) $url .= "/$height";
            return $url;
        }

		return $part->getImageLink($width, $height);
	}

	public function scopeInCollection($query, $webCollection) {
		$webCollectionId = ($webCollection instanceof WebCollection) ? $webCollection->id : $webCollection;
		return $query->whereHas('webCollections', function ($query) use ($webCollectionId) {
			return $query->where('webcollection_id', $webCollectionId);
		});
	}

	public function scopeInCategory($query, $webCategory) {
		$webCategoryId = ($webCategory instanceof WebCategory) ? $webCategory->id : $webCategory;
		return $query->whereHas('webCollections', function ($query) use ($webCategoryId) {
			return $query->inCategory($webCategoryId);
		});
	}

	public function scopeInGroup($query, $webGroup) {
		$webGroupId = ($webGroup instanceof WebGroup) ? $webGroup->id : $webGroup;
		return $query->whereHas('webCollections', function ($query) use ($webGroupId) {
			return $query->inGroup($webGroupId);
		});
	}

	public static function in($model) {
		if ($model instanceof WebGroup) {return self::visible()->inGroup($model);}
		else if ($model instanceof WebCategory) return self::visible()->inCategory($model);
        else if ($model instanceof WebCollection) return self::visible()->inCollection($model);
        else return self::visible();
	}

	public function scopeVisible(Builder $query) {
		return $query
			->where('is_visible', '1')
			->whereHas('webParts', function ($query) {
				return $query->visible();
			})
		;
	}

	public function scopeForDisplay(Builder $query, $slug) {
		return $query
			->visible()
			->with('visibleWebParts')
            ->with('webAttributes')
			->slug($slug)
		;
	}
	
	public function scopeJoinWebPart(Builder $query){
		return $query->leftJoin('WebParts', 'WebParts.webfamily_id', '=', 'WebFamilies.id');
	}

    public function slugUrl() {
        return route('family.slug', [$this->slug]);
    }

    public function thumbnailUrl($width = null, $height = null) {
        return $this->getImageLink($width, $height);
    }

//	public function getDetailsAttribute() {
//		$familyAttributes = [];
//		$parts = new Collection;
//		$thumbnails = new Collection;
//		$thumbnailAttribute = false;
//		foreach ($this->webAttributes as $familyAttribute) {
//			$familyAttributes[$familyAttribute->id] = $familyAttribute->attribute_name;
//			if (!$thumbnailAttribute && $familyAttribute->is_thumbnail) $thumbnailAttribute = $familyAttribute->id;
//		}
//		foreach ($this->visibleWebParts as $webPart) {
//			$partArray = ['webPart' => $webPart];
//
//			foreach ($familyAttributes as $id => $attribute_name) {
//				$partArray['webPartAttributes'][$attribute_name] = $webPart->webAttributeValue($id);
//				if ($id == $thumbnailAttribute) {
//					$thumbnails->put($webPart->webAttributeValue($id), $webPart);
//				}
//			}
//			$parts->put($webPart->id, $partArray);
//		}
//		return ['thumbnails' => $thumbnails, 'parts' => $parts];
//	}

	use HasSlug, RevisionableTrait;
}
