<?php

namespace Hilco;

use Carbon\Carbon;
use Hilco\Models\Customer;
use Hilco\Models\CustomerRewardsStatus;
use Hilco\Models\CustomerRewardsTier;
use Hilco\Models\Part;
use Hilco\Models\ProductCategoryGroup;
use Hilco\Models\WebGroup;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Request;

class RewardsHelper {

    protected function customer() {
        $customer = b2b()->activeCustomer();
        if ($customer) return $customer->affiliateCustomer;

        return false;
    }

    public function tierMap($id = false) {
        if ($id === false) return config('rewards.tierMap');
        return config("rewards.tierMap.$id", $id);
    }

    public function tierImage(Customer $customer = null) {
        if (is_null($customer)) $customer = $this->customer();

        $tier = $this->tier($customer);

        if ($tier == 2) {
            return '/images/rewards-silver.png';
        } else if ($tier == 3) {
            return '/images/rewards-gold.png';
        } else if ($tier == 4) {
            return '/images/rewards-platinum.png';
        }
        return '';
    }

    public function tierImageX($tier) {
        if ($tier == 2) {
            return '/images/rewards-silver.png';
        } else if ($tier == 3) {
            return '/images/rewards-gold.png';
        } else if ($tier == 4) {
            return '/images/rewards-platinum.png';
        }
        return '';
    }

    public function eligibility(Customer $customer = null, $needsValidCustomerCategory = true) {
        if (is_null($customer)) {
            $customer = $this->customer();
        }

        $eligibility = [];

        $customerCategory = $customer->affiliateCustomer->overrideCustomerCategory;
        $customerCategoryId = is_null($customerCategory) ? null : $customerCategory->id;

        $eligibleCustomersString = config('rewards.eligibleCustomerNumbers', '');
        $eligibleCustomers = [];
        if (strlen($eligibleCustomersString)) $eligibleCustomers = explode('^', $eligibleCustomersString);
        if (count($eligibleCustomers)) {
            $eligibility['validCustomer'] = in_array($customer->cust_no, $eligibleCustomers);
        }

        if($needsValidCustomerCategory) {
            $eligibility['validCustomerCategory'] = in_array($customerCategoryId, config('rewards.eligibleCustomerCategoryIds', []));
        } else {
            $eligibility['validCustomerCategory'] = true;
        }
        $eligibility['validMemberGroup'] = !in_array($customer->member_group, config('rewards.excludedMemberGroups', []));
        $eligibility['validBilling'] = $customer->billsDirect;
        $eligibility['validPriceList'] = in_array(array_get($customer, 'activeSegment.def_price_list'), config('rewards.eligiblePriceLists', []));
        $eligibility['validCurrency'] = in_array($customer->currency, config('rewards.eligibleCurrencyCodes', []));

        $eligibility['validAffiliates'] = true;
        foreach ($customer->affiliateCustomer->affiliateChildren as $affiliate) {
            if (starts_with($affiliate->cust_name, '*')) continue;
            if (is_null($affiliate->activeSegment)) continue;

            if (!$affiliate->billsDirect) $eligibility['validAffiliates'] = false;

            if (!in_array(array_get($affiliate, 'activeSegment.def_price_list'), config('rewards.eligiblePriceLists', []))) $eligibility['validAffiliates'] = false;


        }
        return $eligibility;
    }



    public function isEligible(Customer $customer = null, $needsValidCustomerCategory = true) {
        if (!b2b()->isRewardsEnabled()) return false;

        if (is_null($customer)) $customer = $this->customer();

        $eligibility = $this->eligibility($customer, $needsValidCustomerCategory);

        foreach ($eligibility as $item=>$isEligible) {
            if (!$isEligible) return false;
        }

        return true;
    }

    public function isSelfEligible(Customer $customer = null, $needsValidCustomerCategory = true) {
        if (!b2b()->isRewardsEnabled()) return false;

        if (is_null($customer)) $customer = b2b()->activeCustomer();

        $eligibility = $this->eligibility($customer, $needsValidCustomerCategory);

        foreach ($eligibility as $item=>$isEligible) {
            if ($item == 'validAffiliates') continue;
            if (!$isEligible) return false;
        }

        return true;
    }

    public function isRegistered(Customer $customer = null) {
        if (!b2b()->isRewardsEnabled()) return false;

        if (is_null($customer)) $customer = $this->customer();

        $registration = $customer->rewardsRegistration;
        return !is_null($registration);
    }

    public function model(Customer $customer = null) {
        if (is_null($customer)) $customer = $this->customer();
        $customerCategory = $customer->overrideCustomerCategory;
        $customerCategoryId = is_null($customerCategory) ? null : $customerCategory->id;
        return in_array($customerCategoryId, [7,14]) ? 1 : 2;
    }

    public function modelThresholds($model) {
        if ($model == 1) {
            return [
                0 => [
                    2 => 801,
                    3 => 2001,
                    4 => 5001,
                ],
                1 => [
                    2 => 801,
                    3 => 2001,
                    4 => 5001,
                ],
                3 => [
                    2 => 0,
                    3 => 801,
                    4 => 2001,
                ],
                5 => [
                    2 => 0,
                    3 => 0,
                    4 => 801,
                ],
            ];
        } else {
            return [
                0 => [
                    2 => 2501,
                    3 => 6001,
                    4 => 14001,
                ],
                1 => [
                    2 => 2501,
                    3 => 6001,
                    4 => 14001,
                ],
                3 => [
                    2 => 0,
                    3 => 2501,
                    4 => 6001,
                ],
                5 => [
                    2 => 0,
                    3 => 0,
                    4 => 2501,
                ],
            ];
        }
    }

    public function normalizeMix($mix) {
        if ($mix >= 5) return 5;
        else if ($mix >= 3) return 3;
        else if ($mix >= 1) return 1;
        else return $mix;
    }

    public function threshold($model, $mix, $tier) {
        $mix = $this->normalizeMix($mix);
        $thresholds = $this->modelThresholds($model);
        $tiers = array_get($thresholds, $mix, []);
        return array_get($tiers, $tier, false);
    }

    public function calculateTierForSpending($spending, $mix, Customer $customer = null) {
        $thresholds = $this->modelThresholds($this->model($customer));
        krsort($thresholds);

        $theMix = $this->normalizeMix($mix);

        foreach ($thresholds as $minMix => $tiers) {
            if ($minMix > $theMix) continue;
            arsort($tiers);
            foreach ($tiers as $tier => $minSpend) {
                if ($spending >= $minSpend) return $tier;
            }
        }

        return false;
    }

    public function categoryBonus(Customer $customer = null) {
        if (is_null($customer)) $customer = $this->customer();
        $model = $this->model($customer);

        $thresholds = $this->modelThresholds($model);

        $bonuses = [
            2 => [
                2 => ($thresholds[1][2] - $thresholds[3][2]) / 2,
                3 => ($thresholds[1][3] - $thresholds[3][3]) / 2,
                4 => ($thresholds[1][4] - $thresholds[3][4]) / 2,
            ],
            3 => [
                2 => $thresholds[1][2] - $thresholds[3][2],
                3 => $thresholds[1][3] - $thresholds[3][3],
                4 => $thresholds[1][4] - $thresholds[3][4],
            ],
            4 => [
                2 => ($thresholds[3][2] - $thresholds[5][2]) / 2,
                3 => ($thresholds[3][3] - $thresholds[5][3]) / 2,
                4 => ($thresholds[3][4] - $thresholds[5][4]) / 2,
            ],
            5 => [
                2 => $thresholds[3][2] - $thresholds[5][2],
                3 => $thresholds[3][3] - $thresholds[5][3],
                4 => $thresholds[3][4] - $thresholds[5][4],
            ],
        ];

        return array_get($bonuses, $this->qualifyingCategories() . '.' . $this->tier(), 0);
    }

    public function categoryPercentage(Customer $customer = null) {
        if (is_null($customer)) $customer = $this->customer();
        $thresholds = $this->modelThresholds($this->model($customer));
        $max = $thresholds[1][4];
        return $this->categoryBonus() / $max;
    }


    public function statusPercentage($categories = false, Customer $customer = null) {
        if (is_null($customer)) $customer = $this->customer();
        $thresholds = $this->modelThresholds($this->model($customer));
        $totalSpending = $this->totalSpending($customer);

        $mySpending = $totalSpending;
        $statusPercentage = 0;

        if ($categories === false) {
            $categories = $this->qualifyingCategories($customer);
        }
        $categories = $this->normalizeMix($categories);

        foreach (array_get($thresholds, $categories, []) as $tier => $threshold) {
            if ($mySpending > 0) {
                $percentOf = ($threshold <= 0) ? 1 : min(1, $totalSpending / $threshold);
                $mySpending -= $threshold;
                $statusPercentage += ($percentOf / 3);
            }
        }

        return $statusPercentage;
    }

    public function potential(Customer $customer = null) {
        if (is_null($customer)) $customer = $this->customer();

        $potentials = [
            1 => [
                3 => $this->statusPercentage(3, $customer) - $this->statusPercentage(1, $customer),
                5 => $this->statusPercentage(5, $customer) - $this->statusPercentage(3, $customer),
            ],
            2 => [
                3 => $this->statusPercentage(3, $customer) - $this->statusPercentage(1, $customer),
                5 => $this->statusPercentage(5, $customer) - $this->statusPercentage(3, $customer),
            ],
            3 => [
                5 => $this->statusPercentage(5, $customer) - $this->statusPercentage(3, $customer),
            ],
            4 => [
                5 => $this->statusPercentage(5, $customer) - $this->statusPercentage(3, $customer),
            ],
            5 => [

            ],
        ];

        return array_get($potentials, $this->qualifyingCategories($customer), []);
    }

    public function spendingThreshold(Customer $customer = null) {
        if (is_null($customer)) $customer = $this->customer();

        $threshold = 0;
        $model = $this->model($customer);
        $categories = $this->qualifyingCategories($customer);
        $tier = $this->tier($customer);

        if ($model == 1) {
            if ($tier >= 3) {
                if ($categories >= 5) {
                    $threshold = 800;
                } else if ($categories >= 3) {
                    $threshold = 2000;
                } else {
                    $threshold = 5000;
                }
            } else {
                if ($categories >= 5) {
                    $threshold = 0;
                } else if ($categories >= 3) {
                    $threshold = 801;
                } else {
                    $threshold = 2001;
                }
            }
        } else {
            if ($tier >= 3) {
                if ($categories >= 5) {
                    $threshold = 2500;
                } else if ($categories >= 3) {
                    $threshold = 6000;
                } else {
                    $threshold = 14000;
                }
            } else {
                if ($categories >= 5) {
                    $threshold = 0;
                } else if ($categories >= 3) {
                    $threshold = 2501;
                } else {
                    $threshold = 6001;
                }
            }
        }

        return $threshold;
    }

    public function calculateStatusBarPercent($categorySpending, Customer $customer = null) {
        if (is_null($customer)) $customer = $this->customer();

        $overallPct = min(1, $this->totalSpending($customer) / $this->spendingThreshold($customer));
        $categoryPct = $categorySpending / $this->totalSpending($customer);
        return ($overallPct * $categoryPct) * 100;
    }

    public function qualifyingCategories(Customer $customer = null) {
        return $this->potentialQualifyingCategories($customer);
//        if (is_null($customer)) $customer = $this->customer();
//
//        $rewardsTier = $customer->rewardsTier;
//        return ($rewardsTier) ? $customer->rewardsTier->category_mix : 0;
    }

    public function totalSpending(Customer $customer = null) {
        return $this->potentialSpending($customer);
//        if (is_null($customer)) $customer = $this->customer();
//
//        $rewardsTier = $customer->rewardsTier;
//        return ($rewardsTier) ? $customer->rewardsTier->total_spending : 0;
    }

    public function potentialQualifyingCategories(Customer $customer = null) {
        if (is_null($customer)) $customer = $this->customer();

        $categories = 0;
        foreach (array_get($customer, 'rewardsStatus', []) as $statusRow) {
            if ($statusRow['qualifying']) $categories++;
        }

        return $categories;
    }

    public function potentialSpending(Customer $customer = null) {
        if (is_null($customer)) $customer = $this->customer();

        $spending = 0;
        foreach (array_get($customer, 'rewardsStatus', []) as $statusRow) {
            $spending += $statusRow['spending'];
        }

        return $spending;
    }

    public function className($tier = false) {
        if ($tier === false) $tier = rewards()->tier();

        if ($tier == 2) return 'rewards-silver';
        if ($tier == 3) return 'rewards-gold';
        if ($tier == 4) return 'rewards-platinum';
        return '';
    }

    public function tier(Customer $customer = null) {
        if (is_null($customer)) $customer = $this->customer();

        if (!$this->isRegistered($customer) || !$this->isEligible($customer)) return 0;
        $rewardsTier = $customer->rewardsTier;

        return is_null($rewardsTier) ? 0 : $rewardsTier->tier;
    }

    public function tierName($tier = false, Customer $customer = null) {
        if ($tier === false) $tier = $this->tier($customer);
        return $this->tierMap($tier);
    }

    public function joinDate(Customer $customer = null) {
        if (is_null($customer)) $customer = $this->customer();

        $registration = $customer->rewardsRegistration;
        if (is_null($registration)) return false;

        return $registration->date_created->toDateString();
    }

    public function discountForPart($partId, Customer $customer = null) {
        if (is_null($customer)) $customer = $this->customer();

        if (!$this->isRegistered($customer) || !b2b()->isBillingDirect() || !$this->isEligible($customer, false)) return 0;
        $part = ($partId instanceof Part) ? $partId : Part::find($partId);
        $groupName = $part->productFamily->productCategory->productCategoryGroup->product_category_group;
        $currentTier = $this->tier($customer);
        $discount = config("rewards.productCategoryGroups.$groupName.discountPercentByTier.$currentTier", 0);
        return $discount;
    }

    public function discountForPartTier($partId, $tier = false, Customer $customer = null) {
        if (is_null($customer)) $customer = $this->customer();

        if (!$this->isRegistered($customer)) return 0;
        $part = ($partId instanceof Part) ? $partId : Part::find($partId);
        $groupName = $part->productFamily->productCategory->productCategoryGroup->product_category_group;
        if ($tier === false) $tier = $this->tier($customer);
        $discount = config("rewards.productCategoryGroups.$groupName.discountPercentByTier.$tier", 0);
        return $discount;
    }

    public function discountCode($tier = false) {
        if ($tier == 2) return 'TIER 2';
        else if ($tier == 3) return 'TIER 3';
        else if ($tier == 4) return 'TIER 4';

        return 'DISCOUNT';
    }

    public function freightDiscountMultiplier($orderTotal, Customer $customer = null) {
        if (is_null($customer)) $customer = $this->customer();

        if (!$this->isRegistered($customer)) return 1;
        $tier = $this->tier($customer);

        if ($tier == 4 && $orderTotal >= 100) return 0;
        else if ($tier == 3 && $orderTotal >= 150) return 0;
        else if ($tier == 2 && $orderTotal >= 150) return 0.5;
        else return 1;
    }

    public function carrierCode($orderTotal, $baseRate, $tier) {
        $customer = b2b()->activeCustomer();
        if (!$this->isRegistered($customer)) return false;
        if (!$this->isEligible($customer, false)) return false;
        if (!b2b()->isBillingDirect()) return false;

        if ($tier == 4) {
            if ($orderTotal >= 100) {
                return [
                    'code' => '1407',
                    'description' => 'Platinum Rewards Free Freight',
                    'base_code' => '1000',
                    'discount_percent' => 100,
                    'rate' => 0,
                    'fob' => 'NO CHARGE',
                ];
            } else {
                return [
                    'code' => '1404',
                    'description' => 'Platinum Rewards Priority Processing',
                    'base_code' => '1000',
                    'discount_percent' => 0,
                    'rate' => $baseRate,
                    'fob' => 'PREDEFINED',
                ];
            }
        } else if ($tier == 3 && $orderTotal >= 150) {
            return [
                'code' => '1407',
                'description' => 'Gold Rewards Free Freight',
                'base_code' => '1000',
                'discount_percent' => 100,
                'rate' => 0,
                'fob' => 'NO CHARGE',
            ];
        } else if ($tier == 2 && $orderTotal >= 150) {
            return [
                'code' => '1408',
                'description' => 'Silver Rewards 50% Off Freight',
                'base_code' => '1000',
                'discount_percent' => 50,
                'rate' => round($baseRate / 2, 2),
                'fob' => 'PREDEFINED',
            ];
        }
        return false;
    }

    public function calculateUpgradeSpending($currentOrderSpending, Customer $customer = null) {
        if (is_null($customer)) $customer = $this->customer();
        $currentTier = $this->tier($customer);
        if ($currentTier == 4) return [];

        $nextTier = min($currentTier + 1, 4);
        $newSpending = 0;
        $newQualifying = 0;
        $newCategories = [];

        foreach ($this->productCategoryGroups($customer) as $productCategoryGroup) {
            $id = $productCategoryGroup->productcategorygroup_id;
            $categorySpending = $productCategoryGroup->spending;
            $categorySpending += array_get($currentOrderSpending['productCategoryTotals'], $id, 0);

            if ($categorySpending >= 200) $newQualifying++;

            $newCategories[$id] = $categorySpending;
            $newSpending += $categorySpending;
        }

        arsort($newCategories);

        $newTier = $this->calculateTierForSpending($newSpending, $newQualifying, $customer);

        if ($newTier > $currentTier) return 0;

        $needSpending = $this->threshold($this->model($customer), $newQualifying, $nextTier);
        return  $needSpending - $newSpending;
    }

    public function productCategoryGroups(Customer $customer = null) {
        if (is_null($customer)) $customer = $this->customer();

        $groupsConfig = config('rewards.productCategoryGroups');
        $map = collect();
        foreach ($groupsConfig as $name => $info) {
            $map->put($info['webGroupSlug'], $name);
        }

        $status = $customer->rewardsStatus;
        $groups = ProductCategoryGroup::orderBy('product_category_group')->get()->keyBy('id');
        $webGroups = WebGroup::whereIn('slug', $map->keys())->get();

        $data = collect();
        foreach ($webGroups as $webGroup) {
            $groupName = $map[$webGroup->slug];
            $group = $groups->where('product_category_group', $groupName)->first();
            if (is_null($group)) continue;
            $theStatus = $status->where('productcategorygroup_id', $group->id)->first();
            if (is_null($theStatus)) {
                $theStatus = new CustomerRewardsStatus();
                $theStatus->productcategorygroup_id = $group->id;
                $theStatus->spending = 0;

            }
            $theStatus['displayName'] = $webGroup->name;
            $theStatus['webGroupId'] = $webGroup->id;
            $theStatus['webGroupSlug'] = $webGroup->slug;
            $theStatus['spendingPercent'] = 100 * (min(200, $theStatus['spending']) / 200);
            $theStatus['discounts'] = array_get($groupsConfig, "$groupName.discountPercentByTier");
            $data->put($group->id, $theStatus);
        }

        return $data;
    }

    public function redirectToWelcome(Customer $customer = null) {
        if (is_null($customer)) $customer = $this->customer();
        if (!$customer) return false;
        if (!config('rewards.rewardsEnabled')) return false;
        if(!rewards()->isEligible($customer)) return false;


        $registration = $customer->rewardsRegistration;
        if (array_get($registration, 'hasAnsweredSurvey', 1) == 0) {
            $isVisionSource = $customer->member_group == '5653107';
            if ($isVisionSource) {
                return redirect()->route('rewards.welcomeVS');
            } else {
                return redirect()->route('rewards.welcome');
            }
        }
        return false;
    }

    public function info(Customer $customer) {
        $productCategoryGroups = ProductCategoryGroup::get()->keyBy('id');
        $tierLevel = 0;
        $tierName = rewards()->tierMap(0);

        $registration = $customer->rewardsRegistration;
        $isRegistered = !is_null($registration);

        $rewardsStatus = $customer->rewardsStatus;
        if (!is_null($rewardsStatus)) {
            $spending = $rewardsStatus;
        } else {
            $spending = $productCategoryGroups->map(function ($item) {
                $item->spending = 0;
                $item->qualifying = 0;
            });
        }

        $rewardsTier = $customer->rewardsTier;
        if (!is_null($rewardsTier)) {
            $tierLevel = $rewardsTier->tier;
            $tierName = rewards()->tierMap($rewardsTier->tier);
        }

        return [
            'tierLevel' => $tierLevel,
            'tierName' => $tierName,
            'registered' => $isRegistered,
            'eligible' => $customer->rewardsEligibility == Customer::REWARDS_ELIGIBLE,
            'qualified' => $tierLevel > 1,
            'spending' => $spending,
        ];
    }

    public function tierOverride($customer, $newTierLevel, $reason = 'orderUpgrade') {
        $tierCommitSequence = DB::table('CustomerRewardsTiers')->lockForUpdate()->max('commit_sequence') + 1;
        $newTier = new CustomerRewardsTier();
        $newTier->id = generateUUID();
        $newTier->customer_id = $customer->id;
        $newTier->first_qualified_date = Carbon::now()->toDateString();
        $newTier->latest_qualified_date = Carbon::now()->toDateString();
        $newTier->tier = $newTierLevel;
        $newTier->total_spending = $this->totalSpending($customer);
        $newTier->category_mix = $this->qualifyingCategories($customer);
        $newTier->upgrade_reason = $reason;
        $newTier->commit_sequence = $tierCommitSequence;
        $newTier->save();
    }

    public function tierUpgradeSQL($singleAccount = false, $upgradeReason = 'dailyUpgrade') {
        $singleAccountString = '';
        if ($singleAccount) {
            $singleAccountString = ' AND Customers.id = :customer_id';
        }
        $sql = <<<SQLEND
INSERT IGNORE INTO CustomerRewardsTiers (id, customer_id, first_qualified_date, latest_qualified_date, tier, total_spending, category_mix, commit_sequence)
SELECT
  UuidToBin(UUID()),
  newTier.customer_id,
  IF(
	newTier.tier = (select tier from CustomerRewardsTiers where customer_id = newTier.customer_id and CustomerRewardsTiers.deleted_at = 0 order by first_qualified_date desc limit 1),
	(select first_qualified_date from CustomerRewardsTiers where customer_id = newTier.customer_id and CustomerRewardsTiers.deleted_at = 0 order by first_qualified_date desc, tier desc limit 1),
	DATE(
	  NOW()
	)
   ) as first_qualified_date,
  DATE(NOW()) as latest_qualified_date,
  newTier.tier as tier,
  newTier.total_spending as total_spending,
  newTier.category_mix as category_mix,
  :commit_sequence as commit_sequence
FROM
    (SELECT
        CustomerRewardsStatus.customer_id,
        CASE
        WHEN COALESCE(CustomerOverrides.customercategory_id, Customers.customercategory_id) IN (14, 7)
          THEN (
            CASE
            WHEN (
              (SUM(qualifying) >= 5 AND SUM(spending) >= 801) OR
              (SUM(qualifying) >= 3 AND SUM(spending) >= 2001) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 5000)
            )
              THEN 4
            WHEN (
              (SUM(qualifying) >= 5) OR
              (SUM(qualifying) >= 3 AND SUM(spending) >= 801) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 2001)
            )
              THEN 3
            WHEN (
              (SUM(qualifying) >= 3) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 801)
            )
              THEN IF(Customers.member_group = '5653107', 3, 2)
            ELSE IF(Customers.member_group = '5653107', 3, 1)
            END
          )
        WHEN COALESCE(CustomerOverrides.customercategory_id, Customers.customercategory_id) IN (18)
          THEN (
            CASE
            WHEN (
              (SUM(qualifying) >= 5 AND SUM(spending) >= 2501) OR
              (SUM(qualifying) >= 3 AND SUM(spending) >= 6001) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 14000)
            )
              THEN 4
            WHEN (
              (SUM(qualifying) >= 5) OR
              (SUM(qualifying) >= 3 AND SUM(spending) >= 2501) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 6001)
            )
              THEN 3
            WHEN (
              (SUM(qualifying) >= 3) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 2501)
            )
              THEN IF(Customers.member_group = '5653107', 3, 2)
            ELSE IF(Customers.member_group = '5653107', 3, 1)
            END
          )
        ELSE 0
        END AS tier,
        CASE
        WHEN COALESCE(CustomerOverrides.customercategory_id, Customers.customercategory_id) IN (14, 7, 18)
          THEN SUM(spending)
        ELSE 0
        END AS total_spending,
        CASE
        WHEN COALESCE(CustomerOverrides.customercategory_id, Customers.customercategory_id) IN (14, 7, 18)
          THEN SUM(qualifying)
        ELSE 0
        END AS category_mix,
        CASE WHEN CustomerRewardsRegistration.id IS NOT NULL
          THEN 1
        ELSE 0 END AS isRegistered,
        COALESCE(qualifiedCustomers.qualified, 0) AS isQualified
      FROM CustomerRewardsStatus
        JOIN Customers ON (Customers.id = CustomerRewardsStatus.customer_id)
        JOIN CustomerSegments ON (Customers.id = CustomerSegments.customer_id AND CustomerSegments.active = 1)
        LEFT JOIN CustomerOverrides ON (CustomerOverrides.customer_id = Customers.id)
        LEFT JOIN CustomerRewardsRegistration ON (CustomerRewardsRegistration.customer_id = CustomerRewardsStatus.customer_id)
        LEFT JOIN (
                    SELECT
                      Customers.id AS customer_id,
                      CASE WHEN SUM(affiliateGroups.notDirectBilled) = 0
                        THEN 1
                      ELSE 0 END AS qualified
                    FROM Customers
                      JOIN CustomerRewardsRegistration ON (CustomerRewardsRegistration.customer_id = Customers.id)
                      JOIN CustomerSegments ON (CustomerSegments.customer_id = Customers.id)
                      LEFT JOIN CustomerOverrides ON (CustomerOverrides.customer_id = Customers.id)
                      LEFT JOIN (
                                  SELECT
                                    aff_customer_id,
                                    SUM(CASE WHEN CustomerSegments.billto_customer_id != CustomerSegments.customer_id
                                      THEN 1
                                        ELSE 0 END) notDirectBilled
                                  FROM Customers
                                    JOIN CustomerSegments ON (CustomerSegments.customer_id = Customers.id)
                                  WHERE
                                    CustomerSegments.active = 1 AND
                                    CustomerSegments.billto_customer_id = CustomerSegments.customer_id
                                  GROUP BY aff_customer_id
                               ) affiliateGroups ON (affiliateGroups.aff_customer_id = COALESCE(Customers.aff_customer_id, Customers.id))
                    WHERE
                      CustomerSegments.active = 1 AND
                      Customers.id = Customers.aff_customer_id AND
                      CustomerSegments.billto_customer_id = CustomerSegments.customer_id AND
                      COALESCE(CustomerOverrides.customercategory_id, Customers.customercategory_id) IN (7, 14, 18) AND                      
                      CustomerSegments.def_price_list IN ('Catalog', 'Tier 1', 'Tier 2', 'Tier 3', 'Tier 4', 'VolumeDisc') AND
                      Customers.currency = 'USD'$singleAccountString
                    GROUP BY Customers.id
                  ) qualifiedCustomers ON (qualifiedCustomers.customer_id = Customers.id)
      GROUP BY CustomerRewardsStatus.customer_id) newTier
WHERE
  newTier.isQualified = 1 AND
 (
    (select tier from CustomerRewardsTiers where customer_id = newTier.customer_id and CustomerRewardsTiers.deleted_at = 0 order by first_qualified_date desc, tier desc limit 1) IS NULL OR 
    (select tier from CustomerRewardsTiers where customer_id = newTier.customer_id and CustomerRewardsTiers.deleted_at = 0 order by first_qualified_date desc, tier desc limit 1) <= newTier.tier
 )
ON DUPLICATE KEY UPDATE
  latest_qualified_date = DATE(NOW()),
  tier = VALUES(tier),
  total_spending = VALUES(total_spending),
  category_mix = VALUES(category_mix),
  commit_sequence = VALUES(commit_sequence)
;
SQLEND;
        return $sql;
    }

    public function tierDowngradeSQL($singleAccount = false, $upgradeReason = 'dailyDowngrade') {
        $singleAccountString = '';
        if ($singleAccount) {
            $singleAccountString = ' AND Customers.id = :customer_id';
        }

        $sql = <<<SQLEND
INSERT INTO CustomerRewardsTiers (id, customer_id, first_qualified_date, latest_qualified_date, tier, total_spending, category_mix, commit_sequence)
  SELECT
    UuidToBin(UUID()),
    newTier.customer_id,
    DATE(NOW()) as first_qualified_date,
    DATE(NOW()) as latest_qualified_date,
    newTier.tier as tier,
    newTier.total_spending as total_spending,
    newTier.category_mix as category_mix,
    :commit_sequence as commit_sequence
  FROM (SELECT
        CustomerRewardsStatus.customer_id,
        CASE
        WHEN COALESCE(CustomerOverrides.customercategory_id, Customers.customercategory_id) IN (14, 7)
          THEN (
            CASE
            WHEN (
              (SUM(qualifying) >= 5 AND SUM(spending) >= 801) OR
              (SUM(qualifying) >= 3 AND SUM(spending) >= 2001) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 5000)
            )
              THEN 4
            WHEN (
              (SUM(qualifying) >= 5) OR
              (SUM(qualifying) >= 3 AND SUM(spending) >= 801) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 2001)
            )
              THEN 3
            WHEN (
              (SUM(qualifying) >= 3) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 801)
            )
              THEN 2
            ELSE 1
            END
          )
        WHEN COALESCE(CustomerOverrides.customercategory_id, Customers.customercategory_id) IN (18)
          THEN (
            CASE
            WHEN (
              (SUM(qualifying) >= 5 AND SUM(spending) >= 2501) OR
              (SUM(qualifying) >= 3 AND SUM(spending) >= 6001) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 14000)
            )
              THEN 4
            WHEN (
              (SUM(qualifying) >= 5) OR
              (SUM(qualifying) >= 3 AND SUM(spending) >= 2501) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 6001)
            )
              THEN 3
            WHEN (
              (SUM(qualifying) >= 3) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 2501)
            )
              THEN 2
            ELSE 1
            END
          )
        ELSE 0
        END                                       AS tier,
        CASE
        WHEN COALESCE(CustomerOverrides.customercategory_id, Customers.customercategory_id) IN (14, 7, 18)
          THEN SUM(spending)
        ELSE 0
        END                                       AS total_spending,
        CASE
        WHEN COALESCE(CustomerOverrides.customercategory_id, Customers.customercategory_id) IN (14, 7, 18)
          THEN SUM(qualifying)
        ELSE 0
        END                                       AS category_mix,
        CASE WHEN CustomerRewardsRegistration.id IS NOT NULL
          THEN 1
        ELSE 0 END                                AS isRegistered,
        COALESCE(qualifiedCustomers.qualified, 0) AS isQualified
      FROM CustomerRewardsStatus
        JOIN Customers ON (Customers.id = CustomerRewardsStatus.customer_id)
        JOIN CustomerRewardsRegistration ON (CustomerRewardsRegistration.customer_id = CustomerRewardsStatus.customer_id)
        LEFT JOIN CustomerOverrides ON (CustomerOverrides.customer_id = Customers.id)
        LEFT JOIN (
                    SELECT
                      Customers.id AS customer_id,
                      CASE WHEN SUM(affiliateGroups.notDirectBilled) = 0
                        THEN 1
                      ELSE 0 END   AS qualified
                    FROM Customers
                      JOIN CustomerSegments ON (CustomerSegments.customer_id = Customers.id)
                      LEFT JOIN CustomerOverrides ON (CustomerOverrides.customer_id = Customers.id)
                      LEFT JOIN (
                                  SELECT
                                    aff_customer_id,
                                    SUM(CASE WHEN CustomerSegments.billto_customer_id != CustomerSegments.customer_id
                                      THEN 1
                                        ELSE 0 END) notDirectBilled
                                  FROM Customers
                                    JOIN CustomerSegments ON (CustomerSegments.customer_id = Customers.id)
                                  WHERE
                                    CustomerSegments.active = 1 AND
                                    CustomerSegments.billto_customer_id = CustomerSegments.customer_id
                                  GROUP BY aff_customer_id
                                ) affiliateGroups
                        ON (affiliateGroups.aff_customer_id = COALESCE(Customers.aff_customer_id, Customers.id))
                    WHERE
                      CustomerSegments.active = 1 AND
                      CustomerSegments.billto_customer_id = CustomerSegments.customer_id AND
                      COALESCE(CustomerOverrides.customercategory_id, Customers.customercategory_id) IN
                      (7, 14, 18) AND
                      CustomerSegments.def_price_list IN ('Catalog', 'Tier 1', 'Tier 2', 'Tier 3', 'Tier 4', 'VolumeDisc') AND
                      Customers.currency = 'USD'$singleAccountString
                    GROUP BY Customers.id
                  ) qualifiedCustomers ON (qualifiedCustomers.customer_id = Customers.id)
      GROUP BY CustomerRewardsStatus.customer_id) newTier
JOIN CustomerRewardsTiers currentTier ON (
      currentTier.customer_id = newTier.customer_id AND currentTier.deleted_at = '0000-00-00 00:00:00' AND 
      currentTier.first_qualified_date = (SELECT MAX(maxCR.first_qualified_date) FROM CustomerRewardsTiers maxCR WHERE maxCR.customer_id = newTier.customer_id and maxCR.deleted_at = 0)
    )
WHERE
  currentTier.tier > newTier.tier
    AND currentTier.latest_qualified_date < DATE(DATE_SUB(DATE_SUB(NOW(), INTERVAL 6 MONTH), INTERVAL 2 WEEK))
;
SQLEND;
        return $sql;
    }
}
