<?php

namespace Hilco\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\MorphToMany;

trait HasTranslations {
    protected $queuedTranslations = [];
    /**
     * @return MorphToMany
     */
    public function translations() {
        return $this->morphMany(WebTranslation::class, 'translatable');
    }

    public static function bootHasTranslations() {
        static::created(function (Model $translatableModel) {
            foreach ($translatableModel->queuedTranslations as $translationInfo) {
                $translatableModel->setTranslation($translationInfo['language'], $translationInfo['key'], $translationInfo['value']);
            }
            $translatableModel->queuedTranslations = [];
        });
    }

    public function getTranslation($key, $language = false, $default = false) {
        if ($language === false) $language = b2b()->activeLanguage();
        if ($default === false) $default = array_get($this->attributes, $key, '');
        $translation = $this->translations->where('language', $language)->where('key', $key)->first();
        return array_get($translation, "value", $default);
    }

    public function getNameAttribute() {
        return $this->getTranslatedName();
    }

    public function getUntranslatedNameAttribute() {
        return $this->attributes['name'];
    }

    public function getTranslatedName($language = false, $default = false) {
        return $this->getTranslation('name', $language, $default);
    }

    public function setTranslation($language, $key, $value) {
        if (!$this->exists) {
            $this->queuedTranslations[] = [
                'language' => $language,
                'key' => $key,
                'value' => $value,
            ];
            return;
        }

        $this->translations()->updateOrCreate(['language' => $language, 'key' => $key],['value' => $value]);
    }

    public function setTranslatedName($language, $name) {
        $this->setTranslation($language, 'name', $name);
    }

    public function setEnNameAttribute($name) {
        $this->setTranslatedName('en', $name);
    }

    public function setDeNameAttribute($name) {
        $this->setTranslatedName('de', $name);
    }

    public function getEnNameAttribute() {
        return $this->getTranslatedName('en', '');
    }

    public function getDeNameAttribute() {
        return $this->getTranslatedName('de', '');
    }

    public function getCombinedNameAttribute() {
        return $this->de_name . ' (' . $this->en_name . ')';
    }

    public function getTranslationArray($key) {
        return $this->translations->where('key', $key)->pluck('value', 'language')->toArray();
    }

    public function getNameTranslationsAttribute() {
        return $this->getTranslationArray('name');
    }
}