<?php

namespace Marcolin\Models;

use Marcolin\SoftDeletes;


/**
 * Marcolin\Models\FOB
 *
 * @property integer $id
 * @property string $fob_code
 * @property string $fob_desc
 * @property string $date_created
 * @property string $date_modified
 * @property string $date_uploaded
 * @property string $deleted_at
 * @mixin \Eloquent
 */
class SalesOrder extends UuidModel
{
    protected $table = "SalesOrders";

    const ORDER_ORIGIN_B2B = 'B';
    const ORDER_ORIGIN_ESB = 'ipad';

    use SoftDeletes;

    public function customer()
    {
        if ($this->isGuestAccountOrder()) {
            return $this->belongsTo(GuestAccount::class)->withTrashed();
        } else {
            return $this->belongsTo(Customer::class)->withTrashed();
        }
    }

    public function salesRep()
    {
        return $this->belongsTo(SalesRep::class, 'salesrep_id', 'id')->withTrashed();
    }

    public function promotions()
    {
        return $this->belongsToMany(Promotion::class, 'Promotion_SalesOrder', 'salesOrder_id', 'promotion_id');
    }

    public function webPromotions()
    {
        return $this->belongsToMany(Promotion::class, 'WebPromotion_SalesOrder', 'salesOrder_id', 'webPromotion_id');
    }

    public function promotionRedemption() {
        return $this->hasOne(PromotionRedemption::class, 'salesorder_id', 'id');
    }

    public function salesOrder_EmailSent() {
        return $this->hasOne(SalesOrder_EmailSent::class, "salesOrder_id", "id");
    }

    public function salesOrder_LocationData() {
        return $this->hasOne(SalesOrder_LocationData::class, "salesOrder_id", "id");
    }

    public function getTotalSunglassPrice($customer)
    {
        $totalQuantity = 0;
        foreach ($this->salesOrderItems()->with('unit.styleRelation')->get() as $salesOrderItem) {
            if ($salesOrderItem->unit != null && $salesOrderItem->unit->styleRelation->productType == 'S') {
                $totalQuantity += $salesOrderItem->quantity * $salesOrderItem->unit->getCustomerPrice($customer, false);
            }
        }
        return $totalQuantity;
    }

    public function getTotalSunglassQuantityAttribute()
    {
        $totalQuantity = 0;
        foreach ($this->salesOrderItems()->with('unit.styleRelation')->get() as $salesOrderItem) {
            if ($salesOrderItem->unit != null && $salesOrderItem->unit->styleRelation->productType == 'S') {
                $totalQuantity += $salesOrderItem->quantity;
            }
        }
        return $totalQuantity;
    }

    public function getTotalOpticalPrice($customer)
    {
        $totalQuantity = 0;
        foreach ($this->salesOrderItems()->with('unit.styleRelation')->get() as $salesOrderItem) {
            if ($salesOrderItem->unit != null && $salesOrderItem->unit->styleRelation->productType == 'V') {
                $totalQuantity += $salesOrderItem->quantity * $salesOrderItem->unit->getCustomerPrice($customer, false);
            }
        }
        return $totalQuantity;
    }

    public function getTotalOpticalQuantityAttribute()
    {
        $totalQuantity = 0;
        foreach ($this->salesOrderItems()->with('unit.styleRelation')->get() as $salesOrderItem) {
            if ($salesOrderItem->unit != null && $salesOrderItem->unit->styleRelation->productType == 'V') {
                $totalQuantity += $salesOrderItem->quantity;
            }
        }
        return $totalQuantity;
    }

    public function getTotalPrice($customer)
    {
        $totalQuantity = 0;
        foreach ($this->salesOrderItems as $salesOrderItem) {
            if ($salesOrderItem->unit != null) {
                $totalQuantity += $salesOrderItem->quantity * $salesOrderItem->unit->getCustomerPrice($customer, false);
            }
        }
        return $totalQuantity;
    }

    public function getTotalQuantityAttribute()
    {
        $totalQuantity = 0;
        foreach ($this->salesOrderItems as $salesOrderItem) {
            $totalQuantity += $salesOrderItem->quantity;
        }
        return $totalQuantity;
    }

    public function getCompleteNotesAttribute()
    {
        $notes = [];
        foreach ($this->salesOrderNotes as $salesOrderNote) {
            $notes[] = $salesOrderNote->orderNote;
        }
        return implode("\n", $notes);

    }

    public function salesOrderItems()
    {
        return $this->hasMany(SalesOrderItem::class, 'salesOrder_id', 'id');
    }

    public function salesOrderNotes()
    {
        return $this->hasMany(SalesOrderNote::class, 'salesOrder_id', 'id');
    }

    public function scopeDateBetween($query, $fromDate, $toDate = false)
    {
        if ($toDate === false) {
            $query->where('orderDate', '>=', $fromDate);
        } else {
            $query->whereBetween('orderDate', [$fromDate, $toDate]);
        }
        return $query;
    }

    public function scopeWebOrders($query)
    {
        return $query->where('orderType', 'B');
    }

    public function isGuestAccountOrder() {
        return preg_match('/T.*/', $this->customerCode) == 1;
    }
}
