<?php

namespace Marcolin\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Support\Facades\DB;

/**
 * Class Promotion
 * @package Hilco\Models
 *
 * @method static Builder active()
 */
class Promotion extends WebModel {
    protected $table = 'Promotions';
    protected $fillable = [
        'name', 'description', 'enabled', 'visible', 'one_time',
        'start_date', 'end_date',
        'unqual_unapply_text', 'unqual_apply_text', 'qual_unapply_text', 'qual_apply_text',
        'unqual_unapply_title', 'unqual_apply_title', 'qual_unapply_title', 'qual_apply_title',
        'homepage_banner_text', 'homepage_banner_link',
        'mailchimp_campaign_id',
        'commit_sequence'
    ];
    protected $casts = [
        'enabled' => 'boolean',
        'visible' => 'boolean',
        'one_time' => 'boolean',
    ];

    use HasCommitSequence;

    // Overrides Model::boot(), used to define custom 'deleted' state
    public static function boot() {
        parent::boot();

        // make sure the required code and 'eligibility' triggers are created with the promotion
        static::created(function (Promotion $promotion) {
            $promotion->triggers()->create(['trigger_type' => 'codeTrigger']);
            $promotion->triggers()->create(['trigger_type' => 'customerTrigger']);
            $promotion->triggers()->create(['trigger_type' => 'sourceTrigger']);
        });

        // make sure all associated triggers and actions are also deleted
        static::deleted(function (Promotion $promotion) {
            foreach ($promotion->triggers as $trigger) {
                $trigger->delete();
            }
            foreach ($promotion->actions as $action) {
                $action->delete();
            }
        });
    }


    // Get "Datepicker" attribute values for Start Date & End Date
    public function getStartDateDatepickerAttribute() {
        $date = ($this->attributes['start_date'] == '0000-00-00') ?
            Carbon::create(2017, 1, 1) :
            Carbon::createFromFormat('Y-m-d', $this->attributes['start_date']);

        return $date->format('Y-m-d');
    }

    public function getEndDateDatepickerAttribute() {
        $date = ($this->attributes['end_date'] == '0000-00-00') ?
            Carbon::now()->addYear(1) :
            Carbon::createFromFormat('Y-m-d', $this->attributes['end_date']);

        return $date->format('Y-m-d');
    }

    // Getters for the 'unqualified' text displays,
    // handles replacing the 'QTY_LEFT', 'AMT_LEFT' strings with the actual remainder values
    public function getUnqualUnapplyTextForWebAttribute() {
        $unqual_unapply_text = $this->unqual_unapply_text;
        $remainders = $this->getRemaindersToQualify();
        if (count($remainders)) {
            foreach ($remainders as $needle => $replacement) {
                $unqual_unapply_text = preg_replace($needle, $replacement, $this->unqual_unapply_text);
            }
        }
        return $unqual_unapply_text;
    }

    public function getUnqualApplyTextForWebAttribute() {
        $unqual_apply_text = $this->unqual_apply_text;
        $remainders = $this->getRemaindersToQualify();
        if (count($remainders)) {
            foreach ($remainders as $needle => $replacement) {
                $unqual_apply_text = preg_replace($needle, $replacement, $this->unqual_apply_text);
            }
        }
        return $unqual_apply_text;
    }

    public static function getMailChimpCampaigns() {
        $mailchimpCampaigns = MailchimpCampaign::all()->pluck('name', 'id');
        $mailchimpCampaigns->prepend('', '');
        return $mailchimpCampaigns;
    }

    public function mailchimpCampaign() {
        if ($this->mailchimp_campaign_id) {
            $mailchimpCampaign = MailchimpCampaign::find($this->mailchimp_campaign_id);
            return $mailchimpCampaign;
        } else {
            return null;
        }
    }


    // Status functions -- get the current status of the promo (eligible to be applied, almost qualified, or qualified)
    public function isEligible(Customer $customer) {
        $allEligible = true;
        foreach ($this->eligibilityTriggers as $eligibilityTrigger) {
            $allEligible = $allEligible && $eligibilityTrigger->details->isTriggered($customer);
        }
        return $allEligible;
    }

    public function isQualifying(Customer $customer, $webCartItems) {
        $allQualified = true;
        foreach ($this->qualificationTriggers as $qualificationTrigger) {
            $allQualified = $allQualified && $qualificationTrigger->details->isTriggered($customer, $webCartItems);
        }
        return $allQualified;
    }

    /**
     * Get array of remainder values a promotion needs to finish qualifying.
     * @param Customer|null $customer
     * @param null $webCartItems
     * @return array of placeholders to remainder values, e.g.
     * [
     *     '/QTY_LEFT/' => 5,
     *     '/AMT_LEFT/' => 25
     * ]
     */
    public function getRemaindersToQualify(Customer $customer = null, $webCartItems = null) {
        if (!isset($customer)) {
            $customer = b2b()->activeCustomer();
            if (!$customer) return [];
        }
        if (!isset($webCartItems)) {
            $webCartItems = app('cartSingleton')->getCartItems();
        }

        $remainderValues = [];
        foreach ($this->qualificationTriggers as $qualificationTrigger) {
            if (!$qualificationTrigger->details->isTriggered($customer, $webCartItems)) {
                if ($qualificationTrigger->trigger_type === 'orderQuantityTrigger') {
                    $qtyRemaining =
                        $qualificationTrigger->details->minimum_quantity -
                        $qualificationTrigger->details->getCurrentQualifyingCartTotal($webCartItems);
                    $remainderValues['/QTY_LEFT/'] = $qtyRemaining;
                } else if ($qualificationTrigger->trigger_type === 'orderValueTrigger') {
                    $amtRemaining =
                        $qualificationTrigger->details->minimum_value -
                        $qualificationTrigger->details->getCurrentQualifyingCartTotal($customer, $webCartItems);
                    $remainderValues['/AMT_LEFT/'] = b2b()->currencySymbol() . number_format($amtRemaining, 2);
                }
            }
        }
        return $remainderValues;
    }

    /**
     * Return true if the associated codeTrigger for this promotion uses a code list (instead of a single-code)
     * @return bool
     */
    public function hasCodeList() {
        $codeTrigger = $this->codeTriggers()->first();
        return $codeTrigger && $codeTrigger->is_list;
    }

    /**
     * Redeem a coupon (if applicable) on a promo.
     * @param $couponCode
     * @return bool
     */
    public function redeemCouponCode($couponCode) {
        $codeTrigger = $this->codeTriggers()->first();
        if ($codeTrigger && $codeTrigger->is_list) {
            $codeTriggerCode = $codeTrigger->codes()->where('coupon_code', $couponCode)->first();
            if ($codeTriggerCode) {
                $codeTriggerCode->is_redeemed = 1;
                $codeTriggerCode->save();
                return true;
            }
        }
        return false;
    }


    // Morph Map defines the polymorphic relationship between promotions and triggers/actions
    protected static $morphMap = [
        'brandTrigger'                  => PromotionBrandTrigger::class,
        'codeTrigger'                   => PromotionCodeTrigger::class,
        'customerTrigger'               => PromotionCustomerTrigger::class,
        'orderQuantityTrigger'          => PromotionCurrentOrderQuantityTrigger::class,
        'orderValueTrigger'             => PromotionCurrentOrderValueTrigger::class,
        'redemptionLimitTrigger'        => PromotionRedemptionLimitTrigger::class,
        'sourceTrigger'                 => PromotionSourceTrigger::class,

        'itemDiscountAction'            => PromotionItemDiscountAction::class,
        'shippingDiscountAction'        => PromotionShippingDiscountAction::class,
    ];


    // Relationships
    public function triggers() {
        return $this->hasMany(PromotionTrigger::class);
    }

    public function actions() {
        return $this->hasMany(PromotionAction::class);
    }

    public function redemptions() {
        return $this->hasMany(PromotionRedemptions::class);
    }

    public function eligibilityTriggers() {
        return $this->triggers()->onlyEligibility();
    }

    public function qualificationTriggers() {
        return $this->triggers()->onlyQualification();
    }

    public function codeTriggers() {
        return $this->morphedByMany(PromotionCodeTrigger::class,
            'trigger', 'PromotionTriggers','promotion_id');
    }

    public function customerTriggers() {
        return $this->morphedByMany(PromotionCustomerTrigger::class,
            'trigger', 'PromotionTriggers', 'promotion_id');
    }

    public function orderQuantityTriggers() {
        return $this->morphedByMany(PromotionCurrentOrderQuantityTrigger::class,
            'trigger', 'PromotionTriggers', 'promotion_id');
    }

    public function orderValueTriggers() {
        return $this->morphedByMany(PromotionCurrentOrderValueTrigger::class,
            'trigger', 'PromotionTriggers', 'promotion_id');
    }

    public function redemptionLimitTriggers() {
        return $this->morphedByMany(PromotionRedemptionLimitTrigger::class,
            'trigger', 'PromotionTriggers', 'promotion_id');
    }

    public function sourceTriggers() {
        return $this->morphedByMany(PromotionSourceTrigger::class,
            'trigger', 'PromotionTriggers', 'promotion_id');
    }

    public function itemDiscountActions() {
        return $this->morphedByMany(PromotionItemDiscountAction::class,
            'action', 'PromotionActions', 'promotion_id');
    }

    public function shippingDiscountActions() {
        return $this->morphedByMany(PromotionShippingDiscountAction::class,
            'action', 'PromotionActions', 'promotion_id');
    }


    /**
     * Get the morph mapping of trigger/action types to their actual class models
     * @return array
     */
    public static function getMorphMap() {
        return self::$morphMap;
    }

    /**
     * Get all promotions scoped by scopeActive()
     * @return \Illuminate\Database\Eloquent\Builder[]|\Illuminate\Database\Eloquent\Collection
     */
    public static function getActivePromotions() {
        return Promotion::active()->get();
    }


    /**
     * Get an array of 'eligible' promotions, where 'eligible' is defined as
     * - 'active', see scopeActive() doc
     * AND
     * - scoped by scopeEligible(), see its doc
     * AND
     * - 'targeting' the 'active customer'
     * where the 'active customer is defined as the aliased account OR the account
     * tied to the currently authenticated web user, and 'targeting' is defined as either specifically targeting the
     * customer with a customer trigger list or by targeting ALL customers
     * @param Customer $customer (defaults to use B2BHelper->activeCustomer
     * @return mixed
     */
    public static function getEligiblePromotions(Customer $customer = null) {
        if (!isset($customer)) {
            $customer = b2b()->activeCustomer();
            if (!$customer) return [];
        }

        $eligiblePromotions = [];
        foreach (Promotion::active()->unredeemed($customer)->get() as $activePromotion) {
            if ($activePromotion->isEligible($customer)) {
                $eligiblePromotions[] = $activePromotion;
            }
        }
        return $eligiblePromotions;
    }

    /**
     * @param Customer $customer
     * @param WebUser $webUser
     * @return array
     */
    public static function getUnQualifiedPromotions(Customer $customer = null, WebUser $webUser = null) {
        if (!isset($customer)) {
            $customer = b2b()->activeCustomer();
            if (!$customer) return [];
        }
        if (!isset($webUser)) {
            $webUser = auth()->user();
            if (!isset($webUser)) return [];
        }

        $unQualifiedPromotions = [];
        $webCartItems = WebCart::cartItems($webUser->id);
        foreach (Promotion::getEligiblePromotions($customer) as $eligiblePromotion) {
            if (!$eligiblePromotion->isQualifying($customer, $webCartItems)) {
                $unQualifiedPromotions[] = $eligiblePromotion;
            }
        }
        return $unQualifiedPromotions;
    }


    // Scopes
    /**
     * Get all 'active' promotions, where active is defined as
     * - enabled == TRUE
     * AND
     * - current date is between the start and end dates OR after the start date OR before the end date, if either date is set
     * @param $query
     * @return mixed
     */
    public function scopeActive ($query) {
        return
            $query
                ->where('enabled', 1)
                ->where(function ($query) {
                    return
                        $query
                            ->where('start_date', '=', '0000-00-00 00:00:00')
                            ->orWhere('start_date', '<=', DB::raw('DATE(NOW())'))
                        ;
                })
                ->where(function ($query) {
                    return $query
                        ->where('end_date', '=', '0000-00-00 00:00:00')
                        ->orWhere('end_date', '>=', DB::raw('DATE(NOW())'))
                        ;
                })
            ;
    }

    /**
     * Get all promotions for 'eligibility' evaluation, based on the following:
     * - not one-time-use
     * OR
     * - one-time-use AND currently unredeemed by the 'active customer', where 'active customer' is defined as the
     * customer currently being aliased or otherwise associated with the authenticated web user (see B2BHelper.php)
     * @param $query
     * @param Customer $customer
     * @return mixed
     */
    public function scopeUnredeemed ($query, Customer $customer) {
        return
            $query
                ->where('one_time', 0)
                ->orWhere(function ($query) use ($customer) {
                    return
                        $query
                            ->where('one_time', 1)
                            ->whereNotExists(function ($query) use ($customer) {
                                return
                                    $query
                                        ->select(DB::raw('1'))
                                        ->from('PromotionRedemptions')
                                        ->whereRaw('promotion_id = Promotions.id')
                                        ->where('customer_id', $customer->id)
                                    ;
                            })
                        ;
                })
            ;
    }
}
