<?php
namespace Marcolin\Models\Enums;

use Marcolin\Models\SalesRep;
use Marcolin\Models\SalesTarget;
use Symfony\Component\HttpFoundation\AcceptHeader;
use DB;

final class SalesTargetTypes extends BasicEnum {
    public static function SALES_DOLLARS() { return self::_create("SalesDollars"); }
    public static function ACCOUNTS_SOLD() { return self::_create("AccountsSold"); }
    public static function NEW_DOORS_OPENED() { return self::_create("NewDoorsOpened"); }
    public static function BRANDS_PER_DOOR() { return self::_create("BrandsPerDoor"); }
    public static function RETURN_RATE() { return self::_create("ReturnRate"); }
    public static function CUSTOMERS_ON_WEB() { return self::_create('WebCustomers'); }
    public static function WEB_DOLLARS() { return self::_create('WebDollars'); }

    public static function ALL_VALUES() {
        return [self::SALES_DOLLARS(), self::WEB_DOLLARS(), self::ACCOUNTS_SOLD(), self::CUSTOMERS_ON_WEB(), self::NEW_DOORS_OPENED(), self::BRANDS_PER_DOOR(), self::RETURN_RATE()];
    }

    public function getDefaultValue() {
        switch ($this) {
            case SalesTargetTypes::RETURN_RATE(): {
                return .15;
            }
            case SalesTargetTypes::BRANDS_PER_DOOR(): {
                return .15;
            }
            case SalesTargetTypes::NEW_DOORS_OPENED(): {
                return 50;
            }
            case SalesTargetTypes::ACCOUNTS_SOLD(): {
                return 70;
            }
            case SalesTargetTypes::SALES_DOLLARS(): {
                return .09;
            }
            case SalesTargetTypes::CUSTOMERS_ON_WEB(): {
                return .7;
            }
            case SalesTargetTypes::WEB_DOLLARS(): {
                return 1000;
            }
        }

        return 0;
    }

    public function getFormatType() {
        switch ($this) {
            case SalesTargetTypes::RETURN_RATE(): {
                return 'Percentage';
            }
            case SalesTargetTypes::BRANDS_PER_DOOR(): {
                return null;
            }
            case SalesTargetTypes::NEW_DOORS_OPENED(): {
                return null;
            }
            case SalesTargetTypes::ACCOUNTS_SOLD(): {
                return 'Percentage';
            }
            case SalesTargetTypes::SALES_DOLLARS(): {
                return 'Currency';
            }
            case SalesTargetTypes::WEB_DOLLARS(): {
                return 'Currency';
            }
            case SalesTargetTypes::CUSTOMERS_ON_WEB(): {
                return 'Percentage';
            }
        }

        return null;
    }

    public function getTitle() {
        switch ($this) {
            case SalesTargetTypes::RETURN_RATE(): {
                return 'Return Rate (Last 12 Mon)';
            }
            case SalesTargetTypes::BRANDS_PER_DOOR(): {
                return 'Brands/Door (Last 12 Mon)';
            }
            case SalesTargetTypes::NEW_DOORS_OPENED(): {
                return 'New Doors (Last 12 Mon)';
            }
            case SalesTargetTypes::ACCOUNTS_SOLD(): {
                return 'Account Penetration % (Last 12 Mon)';
            }
            case SalesTargetTypes::SALES_DOLLARS(): {
                return 'Direct Sales Dollars (YTD)';
            }
            case SalesTargetTypes::CUSTOMERS_ON_WEB(): {
                return 'Customers w/ Active Web Accounts';
            }
            case SalesTargetTypes::WEB_DOLLARS(): {
                return 'Web Sales Dollars (YTD)';
            }
        }

        return '';
    }

    public function getSalesTarget(SalesRep $salesRep) {
        return SalesTarget::salesTargetType($this)->forRep($salesRep)->first();
    }

    public function getTargetValue($salesRep) {
        $salesTarget = $this->getSalesTarget($salesRep);
        if ($salesTarget) {
            return $salesTarget->targetValue;
        } else {
            return null;
        }
    }

    public function isPassedTarget($salesRep) {
        $targetValue = $this->getTargetValue($salesRep);
        $currentValue = $this->getCurrentValue($salesRep);
        $defaultValue = $this->getDefaultValue();

        return ($targetValue != null && $currentValue >= $targetValue) || ($targetValue == null && $currentValue > $defaultValue);
    }

    public function getCurrentValue($salesRep) {
        $value = null;
        switch ($this) {
            case SalesTargetTypes::RETURN_RATE(): {
                $result = DB::table('SalesOrders')
                    ->select(DB::raw("sum(case when orderType = 'R' then orderQuantity end) / sum(case when orderType = 'I' or orderType = 'R' then orderQuantity end) as returnRate"))
                    ->where('salesrep_id', $salesRep->id)
                    ->where('order_status', '=', 'SENT')
                    ->where('orderDate', '>',DB::raw('date_sub(now(), interval 12 month)'))
                    ->first();
                $value = $result->returnRate;
                break;
            }
            case SalesTargetTypes::BRANDS_PER_DOOR(): {
                $result = DB::table('BrandsPerDoorSummaries')->select('brandsPerDoor')->where('salesrep_id', $salesRep->id)->first();
                if ($result) {
                    $value = $result->brandsPerDoor;
                }
                break;
            }
            case SalesTargetTypes::NEW_DOORS_OPENED(): {
                $result = DB::table('NewDoorsSummaries')->select('newDoors')->where('salesrep_id', $salesRep->id)->first();
                $value = $result->newDoors;
                break;
            }
            case SalesTargetTypes::ACCOUNTS_SOLD(): {
                $result = DB::table('SalesOrders')
                    ->select(DB::raw("count(distinct customerCode) / (select count(distinct customerCode) from SalesRep_Customer_Brand join Customers using (salesDistrict_id) where salesrep_id = SalesOrders.salesrep_id and Customers.deleted_at = 0 and SalesRep_Customer_Brand.deleted_at = 0) as acctPen"))
                    ->where('salesrep_id', $salesRep->id)
                    ->where('order_status', '=', 'SENT')
                    ->where('orderDate', '>',DB::raw('date_sub(now(), interval 12 month)'))
                    ->first();
                $value = $result->acctPen;
                break;
            }
            case SalesTargetTypes::SALES_DOLLARS(): {
                $sql = <<<SQL
SELECT 
    round(sum((SELECT 
            sum(materialOpticalPrice * quantity)
        FROM
            SalesOrderItems
                JOIN
            Units ON (Units.id = SalesOrderItems.unit_id)
                JOIN
            PriceListValues ON (PriceListValues.unit_id = Units.id
                AND PriceListValues.deleted_at = 0)
        WHERE
        PriceListValues.priceList_id = Customers.priceList_id AND
            SalesOrders.id = SalesOrderItems.salesorder_id
                AND SalesOrderItems.deleted_at = 0))/100,2) as yearToDate
FROM
    SalesOrders
        JOIN
    Customers ON (Customers.id = SalesOrders.customer_id)
WHERE
    orderType = 'I'
        AND order_status = 'SENT'
        AND orderDate between DATE_FORMAT(NOW(), '%Y') AND NOW()
        AND salesrep_id = ?;
SQL;

                $result = DB::selectOne($sql, [$salesRep->id]);
                $value = $result->yearToDate;
                break;
            }
            case SalesTargetTypes::CUSTOMERS_ON_WEB(): {
                $result = DB::table('Customers')
                    ->select(DB::raw('count(*) as count'))
                    ->whereExists(function ($query)  {
                        $query->select(DB::raw(1))
                            ->from('WebUsers')
                            ->whereRaw('WebUsers.customer_id = Customers.id and WebUsers.deleted_at = 0 and last_login > date_sub(now(), interval 1 year)');
                    })
                    ->whereExists(function ($query) use ($salesRep) {
                        $query->select(DB::raw(1))
                            ->from('SalesRep_Customer_Brand')
                            ->whereRaw("Customers.salesDistrict_id = SalesRep_Customer_Brand.salesDistrict_id and SalesRep_Customer_Brand.deleted_at = 0 and salesrep_id =". $salesRep->id);
                    })
                    ->where('deleted_at', 0)
                ->first();
                $value = $result->count;
                break;
            }
            case SalesTargetTypes::WEB_DOLLARS(): {
                
                break;
            }
        }

        if ($value) {
            return $value;
        } else {
            return "N/A";
        }
    }

    public function getLastYearValue($salesRep) {
        $value = null;
        switch ($this) {
            case SalesTargetTypes::RETURN_RATE(): {
                $result = DB::table('SalesOrders')
                    ->select(DB::raw("sum(case when orderType = 'R' then orderQuantity end) / sum(case when orderType = 'I' or orderType = 'R' then orderQuantity end) as returnRate"))
                    ->where('salesrep_id', $salesRep->id)
                    ->where('order_status', '=', 'SENT')
                    ->where('orderDate', '<',DB::raw('date_sub(now(), interval 1 year)'))
                    ->where('orderDate', '>',DB::raw('date_sub(now(), interval 2 year)'))
                    ->first();
                $value = $result->returnRate;
                break;
            }
            case SalesTargetTypes::BRANDS_PER_DOOR(): {
                $result = DB::table('BrandsPerDoorSummaries')->select('brandsPerDoorLastYear as brandsPerDoor')->where('salesrep_id', $salesRep->id)->first();
                if ($result) {
                    $value = $result->brandsPerDoor;
                }
                break;
            }
            case SalesTargetTypes::NEW_DOORS_OPENED(): {
                $result = DB::table('NewDoorsSummaries')->select('newDoorsLastYear as newDoors')->where('salesrep_id', $salesRep->id)->first();
                $value = $result->newDoors;
                break;
            }
            case SalesTargetTypes::ACCOUNTS_SOLD(): {
                $result = DB::table('SalesOrders')
                    ->select(DB::raw("count(distinct customerCode) / (select count(distinct customerCode) from SalesRep_Customer_Brand join Customers using (salesDistrict_id) where salesrep_id = SalesOrders.salesrep_id and Customers.deleted_at = 0 and SalesRep_Customer_Brand.deleted_at = 0) as acctPen"))
                    ->where('salesrep_id', $salesRep->id)
                    ->where('order_status', '=', 'SENT')
                    ->where('orderDate', '<',DB::raw('date_sub(now(), interval 1 year)'))
                    ->where('orderDate', '>',DB::raw('date_sub(now(), interval 2 year)'))
                    ->first();
                $value = $result->acctPen;
                break;
            }
            case SalesTargetTypes::SALES_DOLLARS(): {
                $sql = <<<SQL
SELECT 
    round(sum((SELECT 
            sum(materialOpticalPrice * quantity)
        FROM
            SalesOrderItems
                JOIN
            Units ON (Units.id = SalesOrderItems.unit_id)
                JOIN
            PriceListValues ON (PriceListValues.unit_id = Units.id
                AND PriceListValues.deleted_at = 0)
        WHERE
        PriceListValues.priceList_id = Customers.priceList_id AND
            SalesOrders.id = SalesOrderItems.salesorder_id
                AND SalesOrderItems.deleted_at = 0))/100,2) as yearToDate
FROM
    SalesOrders
        JOIN
    Customers ON (Customers.id = SalesOrders.customer_id)
WHERE
    orderType = 'I'
        AND order_status = 'SENT'
        AND orderDate between DATE_SUB(DATE_FORMAT(NOW(), '%Y'),interval 1 year) AND DATE_SUB(NOW(), interval 1 year)
        AND salesrep_id = ?;
SQL;

                $result = DB::selectOne($sql, [$salesRep->id]);
                $value = $result->yearToDate;
                break;
            }
            case SalesTargetTypes::CUSTOMERS_ON_WEB(): {
                break;
            }
            case SalesTargetTypes::WEB_DOLLARS(): {
                break;
            }
        }

        if ($value) {
            return $value;
        } else {
            return "N/A";
        }
    }

    public function getRepAvgValue($salesRep) {
        $value = null;
        switch ($this) {
            case SalesTargetTypes::RETURN_RATE(): {
                $result = DB::table('SalesOrders')
                    ->select(DB::raw("sum(case when orderType = 'R' then orderQuantity end) / sum(case when orderType = 'I' or orderType = 'R' then orderQuantity end) as returnRate"))
                    ->whereNotNull('salesrep_id')
                    ->where('order_status', '=', 'SENT')
                    ->where('orderDate', '<',DB::raw('date_sub(now(), interval 1 year)'))
                    ->where('orderDate', '>',DB::raw('date_sub(now(), interval 2 year)'))
                    ->first();
                $value = $result->returnRate;
                break;
            }
            case SalesTargetTypes::BRANDS_PER_DOOR(): {
                $result = DB::table('BrandsPerDoorSummaries')->select(DB::raw('avg(brandsPerDoor) as brandsPerDoor'))->first();
                $value = $result->brandsPerDoor;
                break;
            }
            case SalesTargetTypes::NEW_DOORS_OPENED(): {
                $result = DB::table('NewDoorsSummaries')->select(DB::raw('avg(newDoors) as newDoors'))->first();
                $value = $result->newDoors;
                break;
            }
            case SalesTargetTypes::ACCOUNTS_SOLD(): {
                $result = DB::table('SalesOrders')
                    ->select(DB::raw("count(distinct customerCode) / (select count(distinct customerCode) from SalesRep_Customer_Brand join Customers using (salesDistrict_id) where salesrep_id = SalesOrders.salesrep_id and Customers.deleted_at = 0 and SalesRep_Customer_Brand.deleted_at = 0) as acctPen"))
                    ->whereNotNull('salesrep_id')
                    ->where('order_status', '=', 'SENT')
                    ->where('orderDate', '>',DB::raw('date_sub(now(), interval 12 month)'))
                    ->first();
                $value = $result->acctPen;
                break;
            }
            case SalesTargetTypes::SALES_DOLLARS(): {
                $sql = <<<SQL
                select round(avg(yearToDate),2) as yearToDateAvg from (
SELECT 
    round(sum((SELECT 
            sum(materialOpticalPrice * quantity)
        FROM
            SalesOrderItems
                JOIN
            Units ON (Units.id = SalesOrderItems.unit_id)
                JOIN
            PriceListValues ON (PriceListValues.unit_id = Units.id
                AND PriceListValues.deleted_at = 0)
        WHERE
        PriceListValues.priceList_id = Customers.priceList_id AND
            SalesOrders.id = SalesOrderItems.salesorder_id
                AND SalesOrderItems.deleted_at = 0))/100,2) as yearToDate
FROM
    SalesOrders
        JOIN
    Customers ON (Customers.id = SalesOrders.customer_id)
WHERE
    orderType = 'I'
        AND order_status = 'SENT'
        AND orderDate between DATE_FORMAT(NOW(), '%Y') AND NOW()
GROUP BY salesrep_id) as t;
SQL;

                $result = DB::selectOne($sql);
                $value = $result->yearToDateAvg;
                break;
            }
            case SalesTargetTypes::CUSTOMERS_ON_WEB(): {
                break;
            }
            case SalesTargetTypes::WEB_DOLLARS(): {
                break;
            }
        }

        if ($value) {
            return $value;
        } else {
            return "N/A";
        }
    }
}