<?php

namespace Hilco\Models;

use Debugbar;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Venturecraft\Revisionable\RevisionableTrait;

class WebSilo extends WebModel implements HasLandingPageInterface {
	protected $table = "WebSilos";
	protected $fillable = ['contactus_id', 'name', 'slug', 'webRoleIds', 'is_visible', 'is_default', 'detail', 'default_price_list', 'contact_phone', 'contact_email', 'weblandingpage_id'];
	protected $casts = [
	    'is_visible' => 'boolean'
	];

	public function webBrands() {
		return $this->belongsToMany(WebBrand::class, 'WebBrand_WebSilo', 'websilo_id', 'webbrand_id')->orderBy('display_order')->orderBy('sort_priority');
	}

    public function visibleWebBrands() {
        return $this->belongsToMany(WebBrand::class, 'WebBrand_WebSilo', 'websilo_id', 'webbrand_id')
            ->withPivot('display_order')
            ->visible()
            ->displayOrder()
        ;
    }

	public function contactUs() {
		return $this->belongsTo(WebContactUs::class, 'contactus_id', 'id');
	}

	public function webLinkCategories() {
		return $this->belongsToMany(WebCategory::class, 'WebLinkCategory_WebSilo', 'websilo_id', 'weblinkcategory_id');
	}

	public function webLinks() {
		return $this->belongsToMany(WebCategory::class, 'WebLink_WebSilo', 'websilo_id', 'weblink_id');
	}

	public function webUrls() {
	    return $this->hasMany(WebUrl::class, 'websilo_id', 'id');
    }

	public function getFeaturedChildrenAttribute() {
        return $this->visibleWebBrands()->wherePivot('is_featured', 1)->get();
    }

    public function getChildrenAttribute() {
        return $this->visibleWebBrands()->get();
    }

    public function divisionRules() {
        return $this->belongsToMany(Division::class, 'WebSilo_Division', 'websilo_id', 'division_id');
    }

    public function customerCategorySummaryRules() {
        return $this->belongsToMany(CustomerCategorySummary::class, 'WebSilo_CustomerCategorySummary', 'websilo_id', 'customercategorysummary_id');
    }

    public function webCustomerTypeRules() {
        return $this->belongsToMany(WebCustomerType::class, 'WebSilo_WebCustomerType', 'websilo_id', 'webcustomertype_id');
    }

    public function getDivisionsAttribute() {
        return $this->divisionRules->pluck('id')->toArray();
    }

    public function setDivisionsAttribute($ids) {
        $this->divisionRules()->sync($ids);
    }

    public function getCustomerCategorySummariesAttribute() {
        return $this->customerCategorySummaryRules->pluck('id')->toArray();
    }

    public function setCustomerCategorySummariesAttribute($ids) {
        $this->customerCategorySummaryRules()->sync($ids);
    }

    public function getWebCustomerTypesAttribute() {
        return $this->webCustomerTypeRules->pluck('id')->toArray();
    }

    public function setWebCustomerTypesAttribute($ids) {
        $this->webCustomerTypeRules()->sync($ids);
    }

    public function getUrlsAttribute() {
        return $this->webUrls->pluck('id')->toArray();
    }

    public function setUrlsAttribute($ids) {
        $this->webUrls()->sync($ids);
    }

    public function webLandingPage() {
        return $this->hasOne(WebLandingPage::class, 'id', 'weblandingpage_id');
    }

	public function scopeVisible(Builder $query) {
		return $query->where('WebSilos.is_visible', '1');
	}

	public function scopeCurrent(Builder $query) {
        $query->where('id', self::currentId());
    }

    public static function currentSilo() {
        $webSilo = session()->get('currentWebSilo', false);
        if (!$webSilo) {
            $query = self::with('webLandingPage');
            $customer = false;
            $user = auth()->user();
            if (!is_null($user)) {
                $defaultWebSilo = $user->defaultWebSilo;
                if (!is_null($defaultWebSilo)) {
                    $webSilo = $defaultWebSilo;
                    session()->set('currentWebSilo', $webSilo);
                    return $webSilo;
                } else {
                    $customer = $user->active_customer;
                }
            }

            if ($customer) {
                $cust_cat_summary = $customer->customerCategory->cust_cat_summary;
                $divisions = $customer->divisions->pluck('id');
                $query->whereHas('divisionRules', function ($query) use ($divisions) {
                    $query->whereIn('Divisions.id', $divisions);
                })->whereHas('customerCategorySummaryRules', function ($query) use ($cust_cat_summary) {
                    $query->where('cust_cat_summary', $cust_cat_summary);
                });
            } else {
                $webUrl = WebUrl::current()->first();
                if (!is_null($webUrl)) {
                    $query->whereHas('webUrl', function ($query) use ($webUrl) {
                        $query->where('id', '=', $webUrl->id);
                    });
                }
            }

            $webSilo = $query->first();
            if (is_null($webSilo)) $webSilo = WebSilo::first();
            session()->set('currentWebSilo', $webSilo);
        }
        return $webSilo;
    }

    public static function currentId() {
        return self::currentSilo() ? self::currentSilo()->id : false;
    }

    public static function currentPriceList() {
        return 'Catalog';
    }

    public function scopeNavbarMenu(Builder $query) {
        return $query->with(['webBrands' => function ($query) {
            return $query->orderBy('pivot_display_order', 'asc')->orderBy('name', 'asc');
        }]);
    }

    public function slugUrl() {
        return route('silo.slug', [$this->slug]);
    }

    use HasAssets, HasSlug, HasLandingPage, RevisionableTrait;

    public function getParentRelationName() {
        return false;
    }
}
