<?php

namespace Hilco\Models;
use DB;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Venturecraft\Revisionable\RevisionableTrait;
use Illuminate\Support\Facades\Input;

/**
 * Class WebCategory
 * @package Hilco\Models
 * @method static WebCategory firstOrNew(array $attributes)
 */
class WebCategory extends WebModel implements HasLandingPageInterface {
    protected $table = "WebCategories";
    protected $fillable = ['name', 'slug', 'webBrandIds', 'is_visible', 'detail', 'sort_priority'];
    protected $casts = [
        ['is_visible' => 'boolean']
    ];
    protected $perPage = 24;
    protected $hierarchyParent = WebBrand::class;

//    use KeyWordsFunctional;

    public function getPerPage()
    {
        $tmpPerPage =Input::get('perPage');
        return (isset($tmpPerPage) && $tmpPerPage > 0)?$tmpPerPage:$this->perPage;
    }

    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function webBrands() {
        return $this->belongsToMany(WebBrand::class, 'WebCategory_WebBrand', 'webcategory_id', 'webbrand_id')->withPivot('display_order');
    }

    public function webCollections() {
        return $this->belongsToMany(WebBrand::class, 'WebBrand_WebCategory', 'webcategory_id', 'webcollection_id')->withPivot('display_order');
    }

    public function visibleWebBrands() {
        return $this->belongsToMany(WebBrand::class, 'WebBrand_WebCategory', 'webcategory_id', 'webcollection_id')
            ->withPivot('display_order')
            ->visible()
            ->displayOrder()
        ;
    }

    public function scopeDisplayOrder(Builder $query) {
        return $query->orderBy('pivot_display_order', 'asc')->orderBy('name', 'asc');
    }

    public function scopeAlphabetical(Builder $query) {
        return $query->orderBy('name', 'asc');
    }

    public function getFeaturedChildrenAttribute() {
        return $this->visibleWebBrands()->wherePivot('is_featured', 1)->get();
    }

    public function getChildrenAttribute() {
        return $this->visibleWebBrands()->get();
    }

    public function scopeInGroup($query, $groupId) {
        return $query->whereHas('webBrands', function ($query) use ($groupId) {
            $query->where('webbrand_id', $groupId);
        });
    }

    public function scopeInSilo($query, $webSiloId) {
        return $query->whereHas('webBrands', function ($query) use ($webSiloId) {
            return $query->inSilo($webSiloId);
        });
    }

    public function setWebBrandIdsAttribute($ids) {
        $this->save();
        $this->createRevisionRecord('webBrandIds', $this->webBrands->pluck('id')->toJson(), json_encode($ids));
        $this->webBrands()->sync($ids);
    }

    public function getHierarchyPathsAttribute() {
        $paths = [];
        foreach ($this->webBrands as $webBrand) {
            foreach ($webBrand->webSilos as $webSilo) {
                $paths[] = [
                    'webSilo' => $webSilo,
                    'webBrand' => $webBrand,
                ];
            }
        }
        return $paths;
    }

    public function scopeVisible(Builder $query) {
        return $query
            ->where('WebCategories.is_visible', '1')
            ->whereHas('webBrands', function ($query) {
                return $query->visible();
            })
        ;
    }

    public function slugUrl() {
        return route('category.slug', [$this->slug]);
    }
    public function getAssetPrefix() {
        return "webCategories/{$this->id}/webContexts";
    }

    public function getPrimaryImage() {
        $asset = $this->assetsByType('primary')->first();
        if (!is_null($asset)) return $asset;

        $webCollection = $this->visibleWebBrands->first();
        if ($webCollection) return $webCollection->getPrimaryImage();
    }


    use HasAssets, HasSlug, HasLandingPage, RevisionableTrait, ManyToManyRevisionableTrait;

    public function getParentRelationName() {
        return 'webBrands';
    }
}
