<?php

namespace Marcolin\Models;

use AlgoliaSearch\Laravel\AlgoliaEloquentTrait;
use DB;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Input;
use Storage;
use Venturecraft\Revisionable\RevisionableTrait;

/**
 * Class WebStyle
 * @package Marcolin\Models
 */
class WebStyle extends WebModel implements HasLandingPageInterface
{
    protected $table = "WebStyles";
    protected $perPage = 24;

    use AlgoliaEloquentTrait;

    public static $perEnvironment = true;

    public $algoliaSettings = [
        'searchableAttributes' => [
            'name',
            'style_suffixes'
        ],
        'attributesForFaceting' => [
            'is_best_seller',
            'visibility',

            'style.salesOrg',

            'style.brand.brandName',
            'style.materialGroupDesc',
            'style.product_shape.shape',
            'style.product_fitting.fitting',
            'style.product_rim_type.rimType',
            'style.product_gender.genderDescription',
            'style.product_type.productTypeDescription',

            'webUnitsByColor.is_new',

            'webUnitsByColor.unit.lensClass',
            'webUnitsByColor.unit.product_lens_type.lensType',
            'webUnitsByColor.unit.product_color_family.frameColorFamily',
            'webUnitsByColor.unit.product_color_family.lensColorFamily',

            'sizes',
        ],
        'ranking' => [
            "asc(style.product_type.id)",
            "asc(style.style)",
            "typo",
            "geo",
            "words",
            "filters",
            "proximity",
            "attribute",
            "exact",
            "custom"
        ]
    ];

    public function _reindex($safe = true, $setSettings = true, $mergeOldSettings = false, \Closure $onInsert = null)
    {
        /** @var \AlgoliaSearch\Laravel\ModelHelper $modelHelper */
        $modelHelper = App::make('\AlgoliaSearch\Laravel\ModelHelper');

        $indices = $modelHelper->getIndices($this);
        $indicesTmp = $safe ? $modelHelper->getIndicesTmp($this) : $indices;

        if ($setSettings === true) {
            $setToTmpIndices = ($safe === true);
            $this->_setSettings($setToTmpIndices, $mergeOldSettings);
        }

        static::has('webBrand.webSilos')
            ->has('style.brand')
            ->has('style.units')
            ->has('webUnits.unit')
            ->with('webBrand.webSilos')
            ->with(['webUnits' => function ($query) {
                return $query
                    ->has('unit')
                    ->with('unit', 'assets');
            }])
            ->with('style.brand')
            ->with('style.productDescription')
            ->with('style.productShape')
            ->with('style.productRimType')
            ->with('style.productFitting')
            ->with('style.productGender')
            ->with('style.productType')
            ->webVisibleAll()
            ->chunk(100, function ($models) use ($indicesTmp, $modelHelper, $onInsert) {
                /** @var \AlgoliaSearch\Index $index */
                foreach ($indicesTmp as $index) {
                    $records = [];
                    $recordsAsEntity = [];

                    foreach ($models as $model) {
                        if ($modelHelper->indexOnly($model, $index->indexName)) {
                            $records[] = $model->getAlgoliaRecordDefault($index->indexName);

                            if ($onInsert && is_callable($onInsert)) {
                                $recordsAsEntity[] = $model;
                            }
                        }
                    }

                    $index->addObjects($records);

                    if ($onInsert && is_callable($onInsert)) {
                        call_user_func_array($onInsert, [$recordsAsEntity]);
                    }
                }
            });

        if ($safe) {
            for ($i = 0; $i < count($indices); $i++) {
                $modelHelper->algolia->moveIndex($indicesTmp[$i]->indexName, $indices[$i]->indexName);
            }

            $this->_setSettings(false); // To a setSettings to set the slave on the master
        }
    }

    public function indexOnly()
    {
        return true;
    }

    public function getAlgoliaRecord()
    {
        $record = array_only($this->toArray(), [
            'id',
            'is_best_seller',
            'slug',
        ]);

        $record['displayName'] = $this->displayName;
        $record['name'] = $this->name;
        $record['visibility'] = [];
        if ($this->isWebVisibleStandard) {
            $record['visibility'][] = 'standard';
        }
        if ($this->isWebVisibleHarleyDavidson) {
            $record['visibility'][] = 'harley';
        }
        if ($this->isWebVisibleAutocat) {
            $record['visibility'][] = 'autocat';
        }
        if ($this->isWebVisibleTomFordLimited) {
            $record['visibility'][] = 'tom_ford_limited';
        }

        $record['style'] = $this->style->toArray();
        foreach ($this->webUnitsByColor as $key => $webUnit) {
            $webUnitArray = $webUnit->toArray();

            $unwanted_keys = [
                'part_number',
                'unit_id',
                'webstyle_id',
                'salesOrg',
                'style',
                'style_id',
                'styleDesc',
                'is_visible',
                'is_family_image',
                'min_quantity',
                'quantity_step',
                'quantity_presets',
                'dimensione3',
                'productFrontMaterial_id',
                'frontMaterialCode',
                'productTempleMaterial_id',
                'templeMaterialCode',
                'productLensType_id',
                'lensTypeCode',
                'salesStatusCode',
                'productColorFamily_id',
                'productSalesStatus_id',
                'flex',
                'rxAble',
                'flagReturnable',
                'seasonCode',
                'assets',
            ];
            $this->recursive_unset($webUnitArray, $unwanted_keys);

            $record['webUnitsByColor'][] = $webUnitArray;
            array_set($record, "webUnitsByColor.$key.imageUrls.thumbnail", $webUnit->primaryImage(100, 80));
            array_set($record, "webUnitsByColor.$key.imageUrls.normal", $webUnit->primaryImage(300, 150));
        };
        $record['colors'] = $this->webVisibleColors;
        $record['sizes'] = $this->webVisibleSizes;

        $unwanted_keys = [
            'date_created',
            'date_modified',
            'date_uploaded',
            'deleted_at',
            'commit_sequence',
            'closeout',
            'phaseOut',
            'newColor',
            'oldMaterialCode',
            'oldColorCode',
            'colorFix',
            'productType',
            'brand_id',
            'brandCode',
            'productGender_id',
            'gender',
            'productShape_id',
            'shapeCode',
            'productRimType_id',
            'rimTypeCode',
            'productFitting_id',
            'fittingCode',
            'productType_id',
        ];
        $this->recursive_unset($record, $unwanted_keys);

        for($index = 0; $index < strlen($record['style']['style']); $index++) {
            $record['style_suffixes'][] = substr($record['style']['style'], $index);
        }

        return $record;
    }

    function recursive_unset(&$array, $unwanted_keys)
    {
        foreach ($unwanted_keys as $unwanted_key) {
            unset($array[$unwanted_key]);
        }
        foreach ($array as &$value) {
            if (is_array($value)) {
                $this->recursive_unset($value, $unwanted_keys);
            }
        }
    }

    protected $indexSettings = [
        'analysis' => [
            'char_filter' => [
                'replace' => [
                    'type' => 'mapping',
                    'mappings' => [
                        '&=> and '
                    ],
                ],
            ],
            'filter' => [
                'word_delimiter' => [
                    'type' => 'word_delimiter',
                    'split_on_numerics' => false,
                    'split_on_case_change' => false,
                    'generate_word_parts' => true,
                    'generate_number_parts' => true,
                    'catenate_all' => true,
                    'preserve_original' => true,
                    'catenate_numbers' => true,
                ]
            ],
            'analyzer' => [
                'default' => [
                    'type' => 'custom',
                    'char_filter' => [
                        'html_strip',
                        'replace',
                    ],
                    'tokenizer' => 'whitespace',
                    'filter' => [
                        'lowercase',
                        'word_delimiter',
                    ],
                ],
            ],
        ],
    ];

    protected $mappingProperties = [
        'title' => [
            'type' => 'string',
            'analyzer' => 'standard'
        ]
    ];

    public static $elasticFuzziness = 1;

    /**
     * @description return elastic query with fuzziness
     */
    public static function buildElasticQuery($keyWords)
    {
        $params = [
            'multi_match' => [
                'query' => $keyWords,
                'fuzziness' => self::$elasticFuzziness,
                'fields' => ['_all']
            ]
        ];
        return $params;
    }

    /**
     * @description index all the entries in an Eloquent model (if needed)
     */
    public static function elasticIndexingIfNeeded()
    {
        $mappingExists = self::mappingExists();
        if (!$mappingExists) {
            self::createIndex();
            self::putMapping();
            self::addAllToIndex();
        }
    }

    public function getPerPage()
    {
        $tmpPerPage = Input::get('perPage');
        return (isset($tmpPerPage) && $tmpPerPage > 0) ? $tmpPerPage : $this->perPage;
    }

    public function style()
    {
        return $this->belongsTo(Style::class);
    }

    public function webStyle_Algolia()
    {
        return $this->hasMany(WebStyle_Algolia::class, 'webStyle_id', 'id');
    }

    public function algoliaTable()
    {
        return $this->hasMany(WebStyle_Algolia::class, 'webStyle_id', 'id');
    }

    public function getAlgoliaClassAttribute() {
        return WebStyle_Algolia::class;
    }

    public function getAlgoliaIDFieldAttribute() {
        return 'webStyle_id';
    }

    public function toString() {
        return $this->style->style;
    }

    public function webUnits()
    {
        return $this->hasMany(WebUnit::class, 'webstyle_id', 'id')->has('unit')
            ->with('unit.productLensType')
            ->with('unit.productColorFamily');
    }

    public function webVisibleWebUnits()
    {
        return $this->webUnits()->where(function ($query) {
            return $query->webVisible();
        });
    }

    public function webVisibleWebUnitsAll()
    {
        return $this->webUnits()->where(function ($query) {
            return $query->webVisibleAll();
        });
    }

    public function getWebUnitsByColorAttribute()
    {
        return $this
            ->webUnits
            ->groupBy('unit.color')
            ->map(function ($item, $key) {
                return $item->first();
            })->values();
    }

    public function webBrand()
    {
        return $this->belongsTo(WebBrand::class, 'webbrand_id', 'id');
    }

    public function webDetailTabs()
    {
        return $this->belongsToMany(WebDetailTab::class, 'WebDetailTab_WebStyle', 'webstyle_id', 'webdetailtab_id')->withPivot('is_visible');
    }

    public function favoritedWebUsers()
    {
        return $this->belongsToMany(WebUser::class, 'WebStyle_WebUser', 'webstyle_id', 'webuser_id');
    }

    public function autocatCatalogItems()
    {
        return $this->morphMany(AutocatCatalogItem::class, 'item');
    }

    public function isFavorite()
    {
        return $this->favoritedWebUsers->where('id', auth()->id())->count() ? true : false;
    }

    public function getIsWebVisibleAttribute()
    {
        return static::webVisible()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleAllAttribute()
    {
        return static::webVisibleAll()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleAutocatAttribute()
    {
        return static::webVisibleAutocat()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleStandardAttribute()
    {
        return static::webVisibleStandard()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleHarleyDavidsonAttribute()
    {
        return static::webVisibleHarleyDavidson()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleTomFordLimitedAttribute()
    {
        return static::webVisibleTomFordLimited()->where('id', '=', $this->id)->count() > 0;
    }

    public function scopeWebVisible($query)
    {
        $query
            ->whereHas('style', function ($query) {
                return $query->webVisible();
            });
        return $query;
    }

    public function scopeWebVisibleAll($query)
    {
        $query
            ->whereHas('style', function ($query) {
                return $query->webVisibleAll();
            });
        return $query;
    }

    public function scopeWebVisibleAutocat($query)
    {
        $query
            ->whereHas('style', function ($query) {
                return $query->webVisibleAutocat();
            });
        return $query;
    }

    public function scopeWebVisibleStandard($query)
    {
        $query
            ->whereHas('style', function ($query) {
                return $query->webVisibleStandard();
            });
        return $query;
    }

    public function scopeWebVisibleHarleyDavidson($query)
    {
        $query
            ->whereHas('style', function ($query) {
                return $query->webVisibleHarleyDavidson();
            });
        return $query;
    }

    public function scopeWebVisiblePPE ($query)
    {
        $query->whereHas('webBrand', function ($query) {
            return $query->ppe();
        })->whereHas('style', function ($query) {
            return $query->webVisiblePPE();
        })->with(
            [
                'style' =>
                    function ($query) {
                        return $query->webVisiblePPE();
                    },
                'webUnits' =>
                    function ($query) {
                        return
                            $query
                                ->webVisiblePPE()
                                ->with(
                                    [
                                        'unit' =>
                                            function ($query) {
                                                return $query->webVisiblePPE();
                                            },
                                    ]
                                );
                    },
            ]
        );
    }
    
    public function scopeWebVisibleTomFordLimited($query)
    {
        $query
            ->whereHas('style', function ($query) {
                return $query->webVisibleTomFordLimited();
            });
        return $query;
    }

    public function scopeAlphabetical(Builder $query)
    {
        return $query->orderBy('name', 'asc');
    }

    public function scopeManagedBy($query, $product_manager)
    {
        return $query->whereHas('webUnits', function ($query) use ($product_manager) {
            return $query->managedBy($product_manager);
        });
    }

    public function scopeInBrand($query, $webBrand)
    {
        $webBrandId = ($webBrand instanceof WebBrand) ? $webBrand->id : $webBrand;
        return $query->where('webBrand_id', '=', $webBrandId);
    }

    public function scopeInSilo($query, $webSilo)
    {
        $webSiloId = ($webSilo instanceof WebSilo) ? $webSilo->id : $webSilo;
        return $query->whereHas('webBrand', function ($query) use ($webSiloId) {
            return $query->inSilo($webSiloId);
        });
    }

    public static function in($model)
    {
        if ($model instanceof WebBrand) {
            return self::inGroup($model);
        } else if ($model instanceof WebCategory) return self::inCategory($model);
        else if ($model instanceof WebBrand) return self::inCollection($model);
        else if ($model instanceof WebSilo) return self::inSilo($model);
        else return self::newQuery();
    }

    public function scopeForDisplay(Builder $query, $slug)
    {
        return $query
            ->has('webVisibleWebUnits')
            ->with(['webVisibleWebUnits' => function ($query) {
                return $query
                    ->with([
                        'assets',
                        'unit' => function ($query) {
                            return $query
                                ->with([
                                    'styleRelation',
                                    'stockLevels',
                                    'priceListValues',
                                ]);
                        }
                    ]);
            },
                'style' => function ($query) {
                    return $query
                        ->with([
                            'productShape',
                            'productFitting',
                            'productRimType',
                            'productGender',
                            'productDescription',
                            'brand'
                        ]);
                }
            ])
            ->with(['favoritedWebUsers' => function ($query) {
                $query->where('WebUsers.id', auth()->id());
            }])
            ->slug($slug);
    }

    public function scopeForBreadcrumbs($query)
    {
        $query->with([
            'webBrand' => function ($lQuery) {
                if (session('breadcrumb.collectionId')) {
                    $lQuery->where('webBrand.id', '=', session('breadcrumb.collectionId'));
                }
                $lQuery->with(['webCategories' => function ($cQuery) {
                    if (session('breadcrumb.categoryId')) {
                        $cQuery->where('WebCategories.id', '=', session('breadcrumb.categoryId'));
                    }
                    $cQuery->with(['webBrand' => function ($gQuery) {
                        if (session('breadcrumb.groupId')) {
                            $gQuery->where('webBrand.id', '=', session('breadcrumb.groupId'));
                        }
                    }]);
                }]);
            }
        ]);
        return $query;
    }

    public function scopePpe($query)
    {
        return $query->whereHas('webBrand', function ($query) {
            return $query->ppe();
        });
    }

    public static function getImageUrl($partNumber, $width = null, $height = null)
    {
        $url = config('marcolin.images.url');
        if (!isset($url)) $url = config('marcolin-b2b.images.url');
        $url .= '/' . $partNumber;
        if ($width) $url .= "/$width";
        if ($height) $url .= "/$height";
        return $url;
    }

    public function scopeJoinWebUnit(Builder $query)
    {
        return $query->leftJoin('WebUnits', 'WebUnits.webfamily_id', '=', 'WebStyles.id');
    }

    public function slugUrl()
    {
        return route('style.slug', [$this->slug]);
    }

    public function scopeForSearchOnly(Builder $query)
    {
        return $query->where('search_only', 1);
    }

    public function scopeNotForSearchOnly(Builder $query)
    {
        return $query->where('search_only', 0);
    }

    public function scopeBestSellers(Builder $query)
    {
        return $query->where('best_seller', 1);
    }

    public function getOrderableAttribute()
    {
        return $this->style->brand->orderable;
    }

    public function getDetailAttribute()
    {
        return $this->style->detail;
    }

    public function getNameAttribute($name)
    {
        if (isset($name) && strlen($name) > 0) {
            return $name;
        }

        $name = $this->style->name;
        if (isset($name) && strlen($name) > 0) {
            return $name;
        }

        return $this->slug;
    }

    public function getDisplayNameAttribute()
    {
        if (isset($name) && strlen($name) > 0) {
            return $name;
        }

        $name = $this->style->display_name;
        if (isset($name) && strlen($name) > 0) {
            return $name;
        }

        return $this->slug;
    }

    public function getDisplayNameForUACAttribute() {
        return $this->name;
    }

    public function getPrimaryImage()
    {
        $asset = $this->assetsByType('primary')->first();
        if (!is_null($asset)) return $asset;

        foreach ($this->webVisibleWebUnits as $webUnit) {
            if (!is_null($webUnit)) {
                $asset = $webUnit->getPrimaryImage();
                if (!is_null($asset)) {
                    return $asset;
                }
            }
        }

        return null;
    }

    public function hasPrimaryImage()
    {
        return $this->getPrimaryImage() !== null;
    }

    public function getHasImageAttribute()
    {
        foreach ($this->webUnits as $webUnit) {
            if (!is_null($webUnit)) {
                $asset = $webUnit->getPrimaryImage();
                if (!is_null($asset)) {
                    return true;
                }
            }
        }

        return false;
    }

    use HasAssets, HasSlug, RevisionableTrait, HasLandingPage, ManyToManyRevisionableTrait;

    public function getParentRelationName()
    {
        return 'webBrand';
    }

    public function getLabelStyleAttribute()
    {
        return array_get(self::$labelStyles, $this->label);
    }

    public function getCustomerPriceAttribute()
    {
        $soldToCustomer = b2b()->activeCustomer();
        return $this->style->getCustomerPrice($soldToCustomer);
    }

    public function getPriceRangeAttribute()
    {
        $webUnits = $this->webVisibleWebUnits;

        if (is_null($webUnits)) return null;

        $activeCustomer = b2b()->activeCustomer();

        $min = PHP_INT_MAX;
        $max = 0;
        foreach ($webUnits as $webUnit) {
            $price = $webUnit->getCustomerPrice($activeCustomer);
            $min = min($min, $price);
            $max = max($max, $price);
        }
        return [$min, $max];
    }

    public function getColorsAttribute()
    {
        $colors = [];
        foreach ($this->webUnits as $webUnit) {
            $color = array_get($webUnit, 'unit.color', false);
            if ($color) $colors[$color] = array_get($webUnit, 'unit.formattedColor', $color);
        }
        asort($colors);
        return $colors;
    }

    public function getWebVisibleColorsAttribute()
    {
        $colors = [];
        foreach ($this->webVisibleWebUnitsAll as $webUnit) {
            $color = array_get($webUnit, 'unit.color', false);
            if ($color) $colors[$color] = array_get($webUnit, 'unit.formattedColor', $color);
        }
        asort($colors);
        return $colors;
    }

    public function getSizesAttribute()
    {
        $sizes = [];
        foreach ($this->webUnits as $webUnit) {
            $size = array_get($webUnit, 'unit.size', false);
            if ($size) $sizes[$size] = $size;
        }
        sort($sizes);
        return $sizes;
    }

    public function getWebVisibleSizesAttribute()
    {
        $sizes = [];
        foreach ($this->webVisibleWebUnitsAll as $webUnit) {
            $size = array_get($webUnit, 'unit.size', false);
            if ($size) $sizes[$size] = $size;
        }
        sort($sizes);
        return $sizes;
    }

    public function primaryImage($width = false, $height = false)
    {
        return WebAsset::compileUrlForKey('pn/' . $this->slug, $width, $height);
    }


    public function getStyleDescriptionAttribute()
    {
        $productDescriptions = array_get($this, 'style.productDescription', false);
        if (!$productDescriptions) return '';
        $productDescription = $productDescriptions->first();

        return array_get($productDescription, 'styleDesc');
    }

    public function getProductDescriptionAttribute()
    {
        $productDescriptions = array_get($this, 'style.productDescription', false);
        if (!$productDescriptions) return '';
        $productDescription = $productDescriptions->first();

        return array_get($productDescription, 'productDesc');
    }

    public function getShapeAttribute()
    {
        return array_get($this, 'style.productShape.shape', '');
    }

    public function getFittingAttribute()
    {
        return array_get($this, 'style.productFitting.fitting', '');
    }

    public function getRimTypeAttribute()
    {
        return array_get($this, 'style.productRimType.rimType', '');
    }

    public function getHasImageAttribute()
    {
        foreach ($this->webUnits as $webUnit) {
            if (!is_null($webUnit)) {
                $asset = $webUnit->getPrimaryImage();
                if (!is_null($asset)) {
                    return true;
                }
            }
        }

        return false;
    }
}
