<?php

namespace Marcolin\Models;

use Illuminate\Auth\Authenticatable;
use Illuminate\Auth\Passwords\CanResetPassword;
use Illuminate\Contracts\Auth\Access\Authorizable as AuthorizableContract;
use Illuminate\Contracts\Auth\Authenticatable as AuthenticatableContract;
use Illuminate\Contracts\Auth\CanResetPassword as CanResetPasswordContract;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Foundation\Auth\Access\Authorizable;
use Illuminate\Notifications\Notifiable;
use Venturecraft\Revisionable\RevisionableTrait;

/**
 * Marcolin\Models\WebUser
 *
 * @property integer $id
 * @property string $email
 * @property string $password
 * @property string $name
 * @property integer $customer_id
 * @property string $remember_token
 * @property string $authy_status
 * @property string $authy_id
 * @property string $date_created
 * @property string $date_modified
 * @property string $date_uploaded
 * @property string $deleted_at
 * @property string $last_login
 * @method static \Illuminate\Database\Query\Builder|\Marcolin\Models\WebUser whereId($value)
 * @method static \Illuminate\Database\Query\Builder|\Marcolin\Models\WebUser whereEmail($value)
 * @method static \Illuminate\Database\Query\Builder|\Marcolin\Models\WebUser wherePassword($value)
 * @method static \Illuminate\Database\Query\Builder|\Marcolin\Models\WebUser whereName($value)
 * @method static \Illuminate\Database\Query\Builder|\Marcolin\Models\WebUser whereCustomerId($value)
 * @method static \Illuminate\Database\Query\Builder|\Marcolin\Models\WebUser whereRememberToken($value)
 * @method static \Illuminate\Database\Query\Builder|\Marcolin\Models\WebUser whereDateCreated($value)
 * @method static \Illuminate\Database\Query\Builder|\Marcolin\Models\WebUser whereDateModified($value)
 * @method static \Illuminate\Database\Query\Builder|\Marcolin\Models\WebUser whereDateUploaded($value)
 * @method static \Illuminate\Database\Query\Builder|\Marcolin\Models\WebUser whereDeletedAt($value)
 * @method static \Illuminate\Database\Query\Builder|\Marcolin\Models\WebUser whereLastLogin($value)
 * @method static \Illuminate\Database\Query\Builder|\Marcolin\Models\WebUser whereAuthyStatus($value)
 * @method static \Illuminate\Database\Query\Builder|\Marcolin\Models\WebUser whereAuthyId($value)
 * @method static WebUser firstOrNew($attributes)
 * @mixin \Eloquent
 */
class WebUser extends WebModel implements AuthenticatableContract, AuthorizableContract, CanResetPasswordContract
{
    use Authenticatable, Authorizable, CanResetPassword, Notifiable;

    protected $table = "WebUsers";
    protected $fillable = ['email', 'username', 'password', 'username', 'name', 'customer_id', 'is_enabled', 'webRoles', 'webRoleIds', 'webPermissionIds', 'needs_password_reset', 'default_websilo_id'];
    protected $hidden = ['password', 'remember_token'];

    public function customer()
    {
        return $this->belongsTo(Customer::class, 'customer_id', 'id');
    }

    public function webAuthEvents()
    {
        return $this->hasMany(WebAuthEvent::class, 'email', 'email');
    }

    public function b2bReturnItems()
    {
        return $this->hasMany(B2BReturnItem::class, 'webuser_id');
    }

    public function b2bReturns()
    {
        return $this->hasMany(B2BReturn::class, 'webuser_id');
    }

    public function workingB2BReturn()
    {
        return $this->hasMany(B2BReturn::class, 'webuser_id')->workingReturn();
    }

    public function webRoles()
    {
        return $this->belongsToMany(WebRole::class, 'WebRole_WebUser', 'webuser_id', 'webrole_id');
    }

    public function webPermissions()
    {
        return $this->belongsToMany(WebPermission::class, 'WebPermission_WebUser', 'webuser_id', 'webpermission_id');
    }

    public function cmsPermissions()
    {
        $cmsPermissions = [];
        foreach ($this->effective_permissions as $permission){
            if (preg_match('/^cms-/',$permission->slug) === 1){
                $cmsPermissions[] = $permission;
            }
        }
        return Collection::make($cmsPermissions);
    }

    public function webPreferences()
    {
        return $this->belongsToMany(WebPreference::class, 'WebPreference_WebUser', 'webuser_id', 'webpreference_id')->withPivot('value');
    }

    public function defaultWebSilo()
    {
        return $this->belongsTo(WebSilo::class, 'default_websilo_id', 'id');
    }

    public function published()
    {
        return $this->hasMany('Marcolin\Models\Content','publisher_webUser_id');
    }

    public function uploads() {
        return $this->hasMany('Marcolin\Models\Upload', 'uploader_webUser_id');
    }

    public function uacReports()
    {
        return $this->hasMany(UACReport::class, 'webuser_id', 'id')->with('uacReportModules');
    }

    public function sharedReports()
    {
        return $this->belongsToMany(UACReport::class, 'UACReport_ViewerWebUser', 'viewer_webuser_id', 'uacreport_id')->with('uacReportModules');
    }

    public function getAllReportsAttribute() {
        return $this->uacReports->merge($this->sharedReports);
    }

    public function webUserAlerts()
    {
        return $this->hasMany(WebUserAlert::class, 'webuser_id', 'id');
    }

    public function getUnAcknowledgedAlertsAttribute() {
        return $this->webUserAlerts()->whereNull('date_acknowledged')->get();
    }

    public function getActiveAlertsAttribute() {
        return $this->webUserAlerts()->where('active', '=', 1)->get();
    }

    public function subscriptions() {
        return $this->hasMany(Subscription::class, 'webUser_id');
    }

    public function isSubscribed(Category $category) {
        return !$this->subscriptions()->where('cmsCategory_id','=',$category->id)->get()->isEmpty();
    }

    public function subscribe(Category $category) {
        $newSubscription = new Subscription([
            'cmsCategory_id' => $category->id,
        ]);
        $this->subscriptions()->save($newSubscription);
    }

    public function unsubscribe(Category $category) {
        Subscription::where('webUser_id', '=', $this->id)->where('cmsCategory_id', '=', $category->id)->delete();
    }

    public function favorites() {
        return $this->hasMany(Favorite::class, 'webUser_id');
    }

    public function hasFavorite(Content $content) {
        return !$this->favorites()->where('cmsContent_id','=',$content->id)->get()->isEmpty();
    }

    public function addFavorite(Content $content) {
        $newFavorite = new Favorite([
            'cmsContent_id' => $content->id,
        ]);
        $this->favorites()->save($newFavorite);
    }

    public function removeFavorite(Content $content) {
        Favorite::where('webUser_id','=',$this->id)->where('cmsContent_id', '=', $content->id)->delete();
    }

    public function favoriteContent($limit, $orderBy='date_created', $desc=false) {
        $content = new Collection();
        foreach($this->favorites as $favorite) {
            $content->add($favorite->content);
        }
        return $content->sortBy($orderBy, $desc)->take($limit);
    }

    public function subscribedContent($limit, $orderBy='date_created', $desc=false) {
        $subscribedContent = new Collection();
        foreach($this->subscriptions as $subscription) {
            foreach ($subscription->category->allContentFlat() as $content) {
                $subscribedContent->put($content->id, $content);
            }
        }
        return $subscribedContent->sortBy($orderBy, $desc)->take($limit);
    }

    public function favoriteStyles()
    {
        return $this->belongsToMany(WebStyle::class, 'WebStyle_WebUser', 'webuser_id', 'webstyle_id')
            ->has('webVisibleWebUnits')
            ->with('webVisibleWebUnits.unit.priceListValues')
            ->with('webVisibleWebUnits.unit.styleRelation');
    }

    public function autocatCatalogs()
    {
        return $this->hasMany(AutocatCatalog::class, 'webuser_id', 'id');
    }

    public function can($permissionSlug, $arguments = [])
    {
        return $this->checkPermission($permissionSlug);
    }

    public function managedWebRoles()
    {
        return $this->webRoles->pluck('webPermissions')->flatten()->pluck('managedWebRoles')->flatten()->keyBy('id');
    }

    public function associatedSalesRep(){
        return $this->managedWebRoles()->isEmpty()? SalesRep::where('customer_id','=', $this->customer->id)->first() : false;
    }

    public function hasRole($roleSlug)
    {
        return $this->webRoles->where('slug', $roleSlug)->count() > 0;
    }

    public function hasDirectPermission($permissions)
    {
        if ($permissions instanceof WebPermission) $permissions = $permissions->slug;
        if (!is_array($permissions)) $permissions = [$permissions];
        return $this->webPermissions->whereIn('slug', $permissions)->count() > 0;
    }

    public function hasRolePermission($permissions)
    {
        if ($permissions instanceof WebPermission) $permissions = $permissions->slug;
        if (!is_array($permissions)) $permissions = [$permissions];
        return $this->webRoles->pluck('webPermissions')->flatten()->whereIn('slug', $permissions)->count() > 0;
    }

    protected function checkPermission($permissionSlug)
    {
        return $this->effective_permissions->where('slug', $permissionSlug)->count() > 0;
    }

    public function getEffectivePermissionsAttribute()
    {
        $rolePemissions = $this->webRoles->pluck('webPermissions')->flatten()->keyBy('id');
        $userPermissions = $this->webPermissions->keyBy('id');
        return $rolePemissions->merge($userPermissions);
    }

    public function hasActiveCustomer()
    {
        return !is_null($this->customer);
    }

    public function getActiveCustomerAttribute()
    {
        return $this->customer;
    }

    public function setWebRolesAttribute($webRoleIds)
    {
        //todo confirm these are saved
        $this->webRoles()->sync($webRoleIds);
    }

    public function setWebRoleIdsAttribute($ids)
    {
        return $this->setWebRolesAttribute($ids);
    }

    public function getWebRoleIdsAttribute()
    {
        return $this->webRoles->pluck('id')->toArray();
    }

    public function setWebPermissionIdsAttribute($ids)
    {
        //todo confirm these are saved
        $this->webPermissions()->sync($ids);
    }

    public static function create(array $attributes = [])
    {
        $idDependentKeys = ['webRoles','webRoleIds','webPermissionIds'];
        $idDependentAttributes = array_only($attributes, $idDependentKeys);
        $attributes = array_except($attributes, $idDependentKeys);

        $model = static::query()->create($attributes);
        foreach ($idDependentAttributes as $key => $value){
            $model->$key = $value;
        }

        return $model;
    }

    use RevisionableTrait;

}
