<?php

namespace Marcolin\Models;

use Auth;
use DateInterval;
use DateTime;
use Illuminate\Database\Eloquent\Builder;
use Image as Image2;
use Storage;

/**
 * Marcolin\Models\Product
 *
 * @property mixed $id
 * @property string $date_created
 * @property string $date_modified
 * @property string $deleted_at
 * @method static \Illuminate\Database\Query\Builder|\Marcolin\Models\Unit whereId($value)
 */
class Unit extends WebModel
{
    protected $table = 'Units';

    const IN_STOCK = 'inStock';
    const OUT_OF_STOCK = 'outOfStock';
    const NOT_AVAILABLE = 'notAvailable';
    const UNKNOWN = 'unknown';

    public function styleRelation()
    {
        return $this->belongsTo(Style::class, 'style_id', 'id');
    }

    function webStyle()
    {
        return $this->hasOne(WebStyle::class, 'style_id', 'style_id');
    }

    public function webUnit()
    {
        return $this->hasOne(WebUnit::class);
    }

    public function smithLensFrames()
    {
        return $this->hasMany(SmithLensFrame::class, 'safilo_unit_id', 'id');
    }

    public function productLensType()
    {
        return $this->belongsTo(ProductLensType::class, 'productLensType_id', 'id');
    }

    public function productColorFamily()
    {
        return $this->belongsTo(ProductColorFamily::class, 'productColorFamily_id', 'id');
    }

    public function productSalesStatus()
    {
        return $this->belongsTo(ProductSalesStatus::class, 'productSalesStatus_id', 'id');
    }

    public function productFrontMaterial()
    {
        return $this->belongsTo(ProductFrontMaterial::class, 'productFrontMaterial_id', 'id');
    }

    public function productTempleMaterial()
    {
        return $this->belongsTo(ProductTempleMaterial::class, 'productTempleMaterial_id', 'id');
    }

    public function getColorInfoAttribute()
    {
        return $this->color . ' - ' . $this->colorDescription;
    }

    public function getColorDescriptionAttribute()
    {
        return title_case($this->attributes['colorDescription']);
    }

    public function priceListValues()
    {
        return $this->hasMany(PriceListValue::class);
    }

    public function stockLevels()
    {
        return $this->belongsToMany(StockLevel::class, 'StockLevel_Unit', 'unit_id', 'stocklevel_id');
    }

    public function unitMarkets($customer = false)
    {
        if ($customer === false) {
            $customer = b2b()->activeCustomer();
        }
        if ($customer) {
            return $this->hasMany(UnitMarket::class, 'unit_id', 'id')->where('marketCode', '=', $customer->marketCode);
        } else {
            return $this->hasMany(UnitMarket::class, 'unit_id', 'id');
        }
    }

    public function getCustomerPrice()
    {
        $market = $this->unitMarkets->first();
        if (isset($market)) {
            return $market->price;
        } else {
            return null;
        }
    }

    public function getCustomerDiscount($customer)
    {
//        $customerDiscount = $customer->customerDiscounts()->where('brand_id', '=', $this->styleRelation->brand_id)->first();
//        return $customerDiscount->discountPercent / 100;
        return 0;
    }

    public function getFormattedColorAttribute()
    {
        return "$this->color  $this->colorDescription ($this->lensCode $this->lensColor)";
    }

    public function getFormattedSizeAttribute()
    {
        return $this->frontSize . '-' . $this->templeSize;
    }

    public function primaryImage($width = false, $height = false)
    {
        return $this->webUnit->primaryImage($width, $height);
    }

    public function getIsWebVisibleAttribute()
    {
        return static::webVisible()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleAllAttribute()
    {
        return static::webVisibleAll()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleAutocatAttribute()
    {
        return static::webVisible()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleStandardAttribute()
    {
        return static::webVisibleStandard()->where('id', '=', $this->id)->count() > 0;
    }

    public function getIsWebVisibleHarleyDavidsonAttribute()
    {
        return static::webVisibleHarleyDavidson()->where('id', '=', $this->id)->count() > 0;
    }

    public function scopeWebVisible(Builder $query)
    {
        $productVisibilityOverride = config('Safilo.product_visibility_override');
        if (!isset($productVisibilityOverride)) {
            $customer = b2b()->activeCustomer();
            if ($customer !== null && $customer !== false) {
                $productVisibilityOverride = $customer->productVisibility;
            } else {
                $productVisibilityOverride = "STANDARD";
            }
        }

        switch (strtoupper($productVisibilityOverride)) {
            case "AUTOCAT":
                return $query->webVisibleAutocat();
            case "STANDARD":
            default:
                return $query->webVisibleStandard();
        }
    }

    public function scopeWebVisibleAll(Builder $query) {
        $query
            ->where(function ($query) {
                return $query->webVisibleStandard();
            })
            ->orWhere(function ($query) {
                return $query->webVisibleAutocat();
            })
        ;
        return $query;
    }

    public function scopeWebVisibleAutocat(Builder $query) {
        $query
            ->where(function ($query) {
                return $query->webVisibleStandard();
            });
        return $query;
    }

    public function scopeWebVisibleStandard(Builder $query)
    {
        $query
            ->whereHas("webUnit", function (Builder $where) {
                $where
                    ->whereRaw("`WebUnits`.`is_visible` = 1");
            })
            ->whereHas("styleRelation", function (Builder $where) {
//                $where
//                    ->whereHas("productType", function (Builder $where) {
//                        $where
//                            ->whereRaw("`ProductTypes`.`productType` IN ('S', 'V', 'C')");
//                    });
            })
            ->whereHas("styleRelation.webStyle", function (Builder $where) {
                $where
                    ->whereRaw("`WebStyles`.`is_visible` = 1");
            })
            ->has("styleRelation.brand")
            ->whereHas("styleRelation.brand.webBrand", function (Builder $where) {
                $where
                    ->whereRaw("`WebBrands`.`is_visible` = 1");
            })
            ->whereHas("unitMarkets", function (Builder $where) {
                $activeCustomer = b2b()->activeCustomer();
                $marketCode = $activeCustomer !== false? $activeCustomer->marketCode : '3O';

                $where
                    ->where("UnitMarkets.marketCode", '=', $marketCode)
                    ->whereRaw("`UnitMarkets`.`discontinued` = 0")
                ;
            })
//            ->whereRaw("`Units`.`commercialDate` != '0000-00-00'")
//            ->where(function (Builder $where) {
//                $where
//                    ->where(function (Builder $where) {
//                        $where
//                            ->whereHas("productSalesStatus", function (Builder $where) {
//                                $where
//                                    ->whereRaw("`ProductSalesStatuses`.`salesStatusCode` = ''");
//                            });
//                    })
//                    ->orWhere(function (Builder $where) {
//                        $where
//                            ->whereHas("productSalesStatus", function (Builder $where) {
//                                $where
//                                    ->whereRaw("`ProductSalesStatuses`.`salesStatusCode` = '03'");
//                            })
//                            ->whereRaw("`Units`.`endDate` >= DATE_SUB(NOW(), INTERVAL 6 MONTH)")
//                            ->whereHas('stockLevels', function (Builder $where) {
//                                $where
//                                    ->whereRaw("`StockLevels`.`availableQuantity` >= 10");
//                            })
//                        ;
//                    });
//            })
        ;
        return $query;
    }


    public function scopeSmithRxVisible(Builder $query)
    {
        $query->has("smithLensFrames");
        return $query;
    }

    public function scopeWebVisibleKeyAccounts(Builder $query)
    {
        return $query;
    }

    public function scopeMissingImages(Builder $query)
    {
        $query
            ->doesntHave('webUnit.assets');
    }

    public function scopeNew($query)
    {
        return $query->whereHas('unitMarkets', function ($query) {
            return $query->new();
        });
    }

    public function getInStockLabelAttribute()
    {
        return config('safilo.inStockLabels.0', 'In Stock');
    }

    public function getOutOfStockLabelAttribute()
    {
        return config('safilo.outOfStockLabels.0', 'Out of stock');
    }


    public function getStockStatusAttribute()
    {
        $market = $this->unitMarkets->first();

        if (isset($market)) {
            if ($market->backorder === 0 || $market->backorderDate === '0000-00-00') {
                return Unit::IN_STOCK;
            } else {
                return Unit::OUT_OF_STOCK;
            }
        } else {
            return Unit::NOT_AVAILABLE;
        }
    }

    public function getStockLabelAttribute()
    {
        $status = $this->stock_status;
        switch ($status) {
            case Unit::IN_STOCK:
                return $this->in_stock_label;
            case Unit::OUT_OF_STOCK:
                $market = $this->unitMarkets->first();
                if (isset($market)) {
                    $label = Carbon::parse($market->backorderDate)->toFormattedDateString();
                }else{
                    $label = $this->out_of_stock_label;
                }
                return $label;
            case Unit::NOT_AVAILABLE:
                return 'Not Available';
        }
        return $this->stock_status;
    }

    public function isBackOrdered() {
        return $this->stock_status === Unit::OUT_OF_STOCK;
    }

    public function getIsNewAttribute() {
        foreach ($this->unitMarkets as $unitMarket) {
            if ($unitMarket->new) return 1;
        }
        return 0;
    }

    public function getDisplayNameForUACAttribute() {
        return
            $this->webStyle->style->styleDesc .
            ' - ' .
            $this->color .
            ' ' .
            $this->colorDescription .
            ' (' .
            $this->lensCode .
            ' ' .
            $this->lensColor .
            ') - ' .
            $this->size
            ;
    }

    use HasModelTranslations;
}

