<?php

namespace Marcolin\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Support\Facades\DB;

/**
 * Class Promotion
 * @package Hilco\Models
 *
 * @method static Builder active()
 */
class Promotion extends WebModel
{
    protected static $morphMap = [
        'codeTrigger' => PromotionCodeTrigger::class,
        'customerTrigger' => PromotionCustomerTrigger::class,
        'currentOrderQuantityTrigger' => PromotionCurrentOrderQuantityTrigger::class,
        'currentOrderValueTrigger' => PromotionCurrentOrderValueTrigger::class,
        'sourceTrigger' => PromotionSourceTrigger::class,
        'freeItemAction' => PromotionFreeItemAction::class,
        'itemDiscountAction' => PromotionItemDiscountAction::class,
        'orderDiscountAction' => PromotionOrderDiscountAction::class,
        'shippingDiscountAction' => PromotionShippingDiscountAction::class,
    ];
    protected $table = 'Promotions';
    protected $fillable = ['name', 'code', 'description', 'start_date', 'end_date', 'enabled', 'b2b', 'mailchimp_campaign_id'];
    protected $casts = [
        'enabled' => 'boolean',
        'start_date' => 'datetime',
        'end_date' => 'datetime',
    ];

    public function actions()
    {
        return $this->hasMany(PromotionAction::class);
    }

    public static function createActionFromType($actionType)
    {
        $class = self::getActionClassFromType($actionType);
        return new $class;
    }

    public static function getActionClassFromType($actionType)
    {
        return array_get(Relation::morphMap(), $actionType, $actionType);
    }

    public function getActionRelationFromType($actionType)
    {
        $relationName = $actionType . 's';
        $relation = $this->$relationName();
        return $relation;
    }

    public static function getActivePromotions()
    {
        return Promotion::where('enabled', '=', '1')->get();
    }

    public function getAllActionsAttribute()
    {
        return collect([
            $this->itemDiscountActions->flatten(),
        ])->flatten();
    }

    public static function getMorphMap()
    {
        return self::$morphMap;
    }

    public function itemDiscountActions()
    {
        return $this->morphedByMany(PromotionItemDiscountAction::class, 'action', 'PromotionActions', 'promotion_id')
            ->withPivot(['sequence']);
    }

    public function scopeActive($query)
    {
        $query
            ->where('enabled', 1)
            ->where(function ($query) {
                return $query
                    ->where('start_date', '=', '0000-00-00 00:00:00')
                    ->orWhere('start_date', '<=', DB::raw('NOW()'));
            })
            ->where(function ($query) {
                return $query
                    ->where('end_date', '=', '0000-00-00 00:00:00')
                    ->orWhere('end_date', '>=', DB::raw('NOW()'));
            });
    }

    public function triggers()
    {
        return $this->hasMany(PromotionTrigger::class);
    }

    public function getStartDateDatepickerAttribute() {
        $date = ($this->attributes['start_date'] == '0000-00-00') ? Carbon::create(2017, 1, 1) : Carbon::createFromFormat('Y-m-d', $this->attributes['start_date']);

        return $date->format('Y-m-d');
    }

    public function getEndDateDatepickerAttribute() {
        $date = ($this->attributes['end_date'] == '0000-00-00') ? Carbon::now()->addYear(1) : Carbon::createFromFormat('Y-m-d', $this->attributes['end_date']);

        return $date->format('Y-m-d');
    }

    public static function getMailChimpCampaigns() {
        $mailchimpCampaigns = MailchimpCampaign::all()->pluck('name', 'id');
        $mailchimpCampaigns->prepend('', '');
        return $mailchimpCampaigns;
    }

    public function mailchimpCampaign() {
        if ($this->mailchimp_campaign_id) {
            $mailchimpCampaign = MailchimpCampaign::find($this->mailchimp_campaign_id);
            return $mailchimpCampaign;
        } else {
            return null;
        }
    }
}
