<?php

namespace Marcolin;

use Marcolin\Models\Customer;
use Marcolin\Models\WebSilo;
use Marcolin\Models\WebUrl;

class B2BHelper {
    protected $activeCustomer;
    protected $activeWebSilo;
    protected $activeShippingAddress;

    public function __construct() {
        $activeShippingAddress = null;
        $this->loadActiveCustomer();
    }

    protected function loadActiveCustomer() {
        $this->activeCustomer = false;

        $aliasedCustomerId = session()->get('aliasedCustomerId', false);

        if ($aliasedCustomerId) {
            $this->activeCustomer = Customer::find($aliasedCustomerId)->with('customerDiscounts')->first();
        } else {
            $user = auth()->user();
            if ($user) $this->activeCustomer = $user->customer->with('customerDiscounts')->first();
        }

        $this->setActiveShippingAddress();

        $this->loadActiveWebSilo();
    }

    protected function loadActiveWebSilo() {
        $this->activeWebSilo = false;

        $activeWebSiloId = session()->get('activeWebSiloId', false);

        if (!$activeWebSiloId) {
            if (auth()->user()) {
                $defaultWebSilo = auth()->user()->defaultWebSilo;
                if ($defaultWebSilo) {
                    $activeWebSiloId = $defaultWebSilo->id;
                }
            }
            if (!$activeWebSiloId) {
                $query = WebSilo::with('webLandingPage');

                if ($this->activeCustomer) {
                    $cust_cat_summary = $this->activeCustomer->customerCategory->cust_cat_summary;
                    $divisions = $this->activeCustomer->divisions->pluck('id');
                    $query->whereHas('divisionRules', function ($query) use ($divisions) {
                        $query->whereIn('Divisions.id', $divisions);
                    })->whereHas('customerCategorySummaryRules', function ($query) use ($cust_cat_summary) {
                        $query->where('cust_cat_summary', $cust_cat_summary);
                    });
                } else {
                    $webUrl = WebUrl::current()->first();
                    if (!is_null($webUrl)) {
                        $query->whereHas('webUrl', function ($query) use ($webUrl) {
                            $query->where('id', '=', $webUrl->id);
                        });
                    }
                }
                $result = $query->first();
                if (!is_null($result)) $activeWebSiloId = $result->id;
            }

            if (!$activeWebSiloId) $activeWebSiloId = WebSilo::first()->id;
        }
        $this->activeWebSilo = WebSilo::with('webLandingPage')->find($activeWebSiloId);
        session()->put('activeWebSiloId', $activeWebSiloId);

        return $this->activeWebSilo;
    }

    public function aliasAs($customerId = false) {
        if ($customerId) {
            session()->put('aliasedCustomerId', $customerId);
        } else {
            session()->forget('aliasedCustomerId');
            session()->forget('activeWebSiloId');
            $this->loadActiveWebSilo();
        }
        $this->loadActiveCustomer();
    }

    public function isAliased() {
        return (session()->get('aliasedCustomerId', false));

    }

    public function stopAliasing() {
        $this->aliasAs(false);
    }

    public function setActiveShippingAddress($customerShippingAddress = false) {
        if ($customerShippingAddress === false) {
            if ($this->activeCustomer) {
                $customerShippingAddress = $this->activeCustomer->default_shipping_address;
            }
        }
        $this->activeShippingAddress = $customerShippingAddress;
        return $this->activeShippingAddress;
    }

    public function activeCustomer() {
        return $this->activeCustomer;
    }

    public function activeShippingAddress() {
        return $this->activeShippingAddress;
    }

    public function activeCountry() {
        if ($this->activeShippingAddress) return $this->activeShippingAddress()->country;

        return null;
    }

    public function activeCurrency() {
        $activeCurrency = 'USD';

        if ($this->activeCustomer) {
            $activeCurrency = $this->activeCustomer->currency;
        }

        return $activeCurrency;
    }

    public function activeWebSilo() {
        return $this->activeWebSilo;
    }

    public function currencySymbol($currencyCode = false) {
        if ($currencyCode === false) $currencyCode = $this->activeCurrency();
        if ($currencyCode == 'GBP') return '&pound;';

        return '$';
    }

    public function formatPrice($price, $decimals = -1, $symbol = '$') {
        if ($decimals = -1) {
            if ($price == round($price)) $decimals = 0;
            else $decimals = 2;
        }
        return $symbol . number_format($price, $decimals);
    }

    public function activeSalesOrg() {
        return array_get($this->activeCustomer, 'salesOrg', 'UN01');
    }
}